package com.bxm.mccms.common.core.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.bxm.acl.facade.model.UserVo;
import com.bxm.mccms.common.core.entity.DeveloperBill;
import com.bxm.mccms.common.core.mapper.DeveloperBillMapper;
import com.bxm.mccms.common.core.service.IDeveloperBillService;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.bxm.mccms.common.helper.exception.McCmsException;
import com.bxm.mccms.common.helper.util.UserRoleUtil;
import com.bxm.mccms.common.integration.ssp.developer.DeveloperIntegration;
import com.bxm.mccms.common.model.income.DeveloperBillDTO;
import com.bxm.mccms.common.model.income.DeveloperBillVO;
import com.bxm.mccms.common.model.income.DeveloperWithdrawlVO;
import com.bxm.mcssp.facade.model.developer.DeveloperFacadeQueryDTO;
import com.bxm.mcssp.facade.model.developer.DeveloperFacadeVO;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * <h3>类的基本描述</h3>
 *
 * @author hcmony
 * @since V1.0.0, 2020/1/9 16:50
 */
@Slf4j
@Service
public class DeveloperBillServiceImpl extends ServiceImpl<DeveloperBillMapper, DeveloperBill> implements IDeveloperBillService {

    @Autowired
    private DeveloperIntegration developerIntegration;
    @Autowired
    private DeveloperBillMapper developerBillMapper;

    @Override
    public boolean init() {
        String month = LocalDate.now().plusMonths(-1).format(DateTimeFormatter.ofPattern("yyyy-MM"));
        QueryWrapper<DeveloperBill> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq(DeveloperBill.MONTH, month);
        if (CollectionUtils.isNotEmpty(list(queryWrapper))) {
            log.warn("[{}] Developer Bill ", month);
            return false;
        }
        final List<DeveloperBill> developerBills = developerBillMapper.selectByMonth(month);
        return saveBatch(developerBills);
    }

    @Transactional(rollbackFor = Exception.class, timeout = 30)
    @Override
    public Boolean publish(Long id, String month) {
        UserVo user = UserRoleUtil.getUser();
        if (id != null) {
            DeveloperBill bill = getById(id);
            if (bill == null) {
                throw new McCmsException("没有对应的账单" + id);
            }
            if (DeveloperBill.UNCONFIRMED != bill.getStatus()) {
                throw new McCmsException("账单不是未确认状态" + id);
            }
            bill.setStatus(DeveloperBill.BXM_CONFIRMED);
            bill.setModifyUser(user.getUsername());
            return updateById(bill);
        }
        DeveloperBill bill = new DeveloperBill();
        bill.setStatus(DeveloperBill.BXM_CONFIRMED);
        bill.setModifyUser(user.getUsername());
        UpdateWrapper<DeveloperBill> updateWrapper = new UpdateWrapper<>();
        updateWrapper.eq(DeveloperBill.MONTH, month);
        updateWrapper.eq(DeveloperBill.STATUS, DeveloperBill.UNCONFIRMED);
        return update(bill, updateWrapper);

    }

    @Transactional(rollbackFor = Exception.class, timeout = 30)
    @Override
    public Boolean deduction(String remark, BigDecimal money, Long id) {
        UserVo user = UserRoleUtil.getUser();
        DeveloperBill bill = getById(id);
        if (bill == null) {
            throw new McCmsException("没有对应的账单" + id);
        }
        if (DeveloperBill.UNCONFIRMED != bill.getStatus()) {
            throw new McCmsException("账单不是未确认状态" + id);
        }

        //zmm:-10,原因
        StringBuilder sb = new StringBuilder();
        sb.append(bill.getDeductionLog()).append("操作人:").append(user.getUsername()).append(",加款").append(money).append(",原因:").append(remark).append(";");
        bill.setDeductionLog(sb.toString());
        bill.setCashMoney(money.add(bill.getCashMoney()));
        bill.setModifyUser(user.getUsername());
        return updateById(bill);
    }

    @Override
    public IPage<DeveloperBillVO> findAll(DeveloperBillDTO query) {
        DeveloperFacadeQueryDTO dto = new DeveloperFacadeQueryDTO();
        dto.setMjCode(query.getMj());
        dto.setDeveloperKeyword(query.getKeywords());
        List<DeveloperFacadeVO> providerAppList = developerIntegration.list(dto);
        if (CollectionUtils.isEmpty(providerAppList)) {
            log.warn("Developer List  is Empty . ");
            return new Page<>();
        }

        Map<Long, DeveloperFacadeVO> collect = providerAppList.stream().collect(Collectors.toMap(DeveloperFacadeVO::getId, DeveloperFacadeVO -> DeveloperFacadeVO));

        QueryWrapper<DeveloperBill> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq(DeveloperBill.MONTH, query.getMonth());
        queryWrapper.in(DeveloperBill.DEVELOPER_ID, providerAppList.stream().map(DeveloperFacadeVO::getId).collect(Collectors.toList()));

        Page<DeveloperBill> p = new Page<>();
        p.setCurrent(query.getCurrent());
        p.setSize(query.getSize());
        IPage<DeveloperBill> page = page(p, queryWrapper);
        Page<DeveloperBillVO> result = new Page<>();
        if (CollectionUtils.isEmpty(page.getRecords())) {
            return result;
        }
        BeanUtils.copyProperties(page, result);
        List<DeveloperBillVO> developerBillVOList = new ArrayList<>(page.getRecords().size());
        page.getRecords().forEach(developerBill -> {
            DeveloperBillVO vo = new DeveloperBillVO();
            BeanUtils.copyProperties(developerBill, vo);
            vo.setDeveloperName(collect.get(vo.getDeveloperId()).getDeveloperName());
            vo.setMj(collect.get(vo.getDeveloperId()).getMjName());
            developerBillVOList.add(vo);
        });
        result.setRecords(developerBillVOList);
        return result;
    }

    @Override
    public IPage<DeveloperWithdrawlVO> findWithdrawalAll(DeveloperBillDTO query) {
        DeveloperFacadeQueryDTO dto = new DeveloperFacadeQueryDTO();
        dto.setDeveloperKeyword(query.getKeywords());
        List<DeveloperFacadeVO> providerAppList = developerIntegration.list(dto);
        if (CollectionUtils.isEmpty(providerAppList)) {
            return new Page<>();
        }

        Map<Long, DeveloperFacadeVO> collect = providerAppList.stream().collect(Collectors.toMap(DeveloperFacadeVO::getId, DeveloperFacadeVO -> DeveloperFacadeVO, (v1, v2) -> v2));
        QueryWrapper<DeveloperBill> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq(DeveloperBill.MONTH, query.getMonth());
        queryWrapper.in(DeveloperBill.DEVELOPER_ID, providerAppList.stream().map(DeveloperFacadeVO::getId).collect(Collectors.toList()));
        queryWrapper.ge(DeveloperBill.MONTH, query.getStartDate());
        queryWrapper.le(DeveloperBill.MONTH, query.getEndDate());
        Page p = new Page();
        p.setCurrent(query.getCurrent());
        p.setSize(query.getSize());
        IPage<DeveloperWithdrawlVO> page = page(p, queryWrapper);

        page.getRecords().forEach(vo -> {
            DeveloperFacadeVO develeper = collect.get(vo.getDeveloperId());
            if (develeper == null) {
                return;
            }
            vo.setDeveloperName(develeper.getDeveloperName());
            vo.setCompanyName(develeper.getCompanyName());

        });
        return page;
    }
}
