package com.bxm.mccms.common.core.service.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.metadata.OrderItem;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.bxm.mccms.common.core.entity.Dsp;
import com.bxm.mccms.common.core.mapper.DspMapper;
import com.bxm.mccms.common.core.service.IDspService;
import com.bxm.mccms.common.helper.exception.McCmsException;
import com.bxm.mccms.common.pushable.DspPushable;
import com.bxm.mcssp.common.entity.IDAndNameVO;
import com.google.common.collect.Lists;

/**
 * <p>
 * 广告平台 服务实现类
 * </p>
 *
 * @author Allen Hu
 * @since 2019-12-18
 */
@Service
public class DspServiceImpl extends ServiceImpl<DspMapper, Dsp> implements IDspService {
    
    @Autowired
    private DspPushable dspPushable;

    @Override
    public IPage<Dsp> pageByLike(Page page, Dsp query) {
        if(CollectionUtils.isEmpty(page.getOrders())){
            page.setOrders(Lists.newArrayList(OrderItem.desc("id")));
        }
        LambdaQueryWrapper<Dsp> wrapper = new QueryWrapper<Dsp>().lambda();
        if(StringUtils.isNotBlank(query.getDspName())){
            wrapper.and(c -> c.like(Dsp::getDspName, query.getDspName())
                    .or().like(Dsp::getId, query.getDspName()));
        }
        return getBaseMapper().selectPage(page, wrapper);
    }

    @Override
    public List<IDAndNameVO> list(String keyword) {
        LambdaQueryWrapper<Dsp> queryWrapper = new QueryWrapper<Dsp>().lambda()
                .eq(Dsp::getDeleted, 0);
        if (StringUtils.isNotBlank(keyword)){
            queryWrapper.and(i -> i.like(Dsp::getDspName, keyword).or().like(Dsp::getId, keyword));
        }
        List<Dsp> list = super.list(queryWrapper);
        List<IDAndNameVO> resps = new ArrayList<>();
        if(CollectionUtils.isEmpty(list)){
            return resps;
        }
        for(Dsp dsp : list){
            IDAndNameVO resp = new IDAndNameVO();
            resp.setId(dsp.getId());
            resp.setName(dsp.getDspName());
            resps.add(resp);
        }
        return resps;
    }

    @Override
    public Dsp getByOnly(String dspCode) {
        LambdaQueryWrapper<Dsp> wrapper = new QueryWrapper<Dsp>()
                .lambda();
        wrapper.eq(Dsp::getDspCode, dspCode);
        List<Dsp> list = getBaseMapper().selectList(wrapper);
        if(CollectionUtils.isEmpty(list)){
            return null;
        }
        return list.get(0);
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Boolean saveInfo(Dsp t) {
        Dsp old = getByOnly(t.getDspCode());
        if(null != old){
            throw new McCmsException("平台代码“"+t.getDspCode()+"”已存在");
        }
        int rows = getBaseMapper().insert(t);
        //消息推送
        dspPushable.push(getById(t.getId()));
        return rows > 0;
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Boolean updateInfo(Dsp t) {
        Dsp old = getByOnly(t.getDspCode());
        if(null != old && !Objects.equals(old.getId(), t.getId())){
            throw new McCmsException("平台代码“"+t.getDspCode()+"”已存在");
        }
        int rows = getBaseMapper().updateById(t);
        //消息推送
        dspPushable.push(getById(t.getId()));
        return rows > 0;
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Boolean updateDeleted(Long id, Integer deleted) {
        Dsp t = getById(id);
        if(null == t){
            throw new McCmsException("数据不存在");
        }
        t.setDeleted(deleted);
        getBaseMapper().updateById(t);

        //消息推送
        dspPushable.push(t);
        return Boolean.TRUE;
    }
}
