package com.bxm.mccms.common.core.service.impl;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.bxm.acl.facade.model.UserVo;
import com.bxm.mccms.common.core.entity.PositionChannelIncome;
import com.bxm.mccms.common.core.mapper.PositionChannelIncomeMapper;
import com.bxm.mccms.common.core.service.IPositionChannelIncomeService;
import com.bxm.mccms.common.core.service.IPositionIncomeService;
import com.bxm.mccms.common.helper.exception.McCmsException;
import com.bxm.mccms.common.helper.util.StartAndEndTimeUtil;
import com.bxm.mccms.common.model.income.PositionChannelIncomeExcelDTO;
import com.bxm.mccms.common.model.income.PositionChannelIncomeListVO;
import com.bxm.mccms.common.model.income.PositionChannelIncomeQueryDTO;
import com.bxm.mccms.common.model.income.PositionChannelIncomeSyncDTO;
import com.bxm.mccms.common.model.income.PositionChannelIncomeUploadVO;
import com.bxm.mcssp.common.util.DateUtil;

import cn.afterturn.easypoi.excel.ExcelImportUtil;
import cn.afterturn.easypoi.excel.entity.ImportParams;

/**
 * <p>
 * 广告位渠道收益记录 服务实现类
 * </p>
 *
 * @author zhengwangeng
 * @since 2021-01-15
 */
@Service
public class PositionChannelIncomeServiceImpl extends BaseServiceImpl<PositionChannelIncomeMapper, PositionChannelIncome> implements IPositionChannelIncomeService {

    @Autowired
    private IPositionIncomeService iPositionIncomeService;

    @Override
    public Page<PositionChannelIncomeListVO> pageBySearch(Page page, PositionChannelIncomeQueryDTO dto) {
        return getBaseMapper().pageBySearch(page, dto);
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public PositionChannelIncomeUploadVO upload(UserVo userVo, MultipartFile file) {
        ImportParams importParams = new ImportParams();
        PositionChannelIncomeUploadVO positionChannelIncomeUploadVO = new PositionChannelIncomeUploadVO();
        List<PositionChannelIncomeExcelDTO> list = null;
        try {
            list = ExcelImportUtil.importExcel(file.getInputStream(), PositionChannelIncomeExcelDTO.class, importParams);
        } catch (Exception e) {
            log.error(e.getMessage());
            throw new McCmsException("文件读取失败！");
        }
        if (CollectionUtils.isEmpty(list)) {
            throw new McCmsException("导入数据为空！");
        }

        checkRepeated(list);

        ArrayList<PositionChannelIncome> positionChannelIncomes = new ArrayList<>(list.size());
        HashMap<String, PositionChannelIncome> positionChannelIncomeHashMap = new HashMap<>(list.size());
        try{
            for (PositionChannelIncomeExcelDTO positionChannelIncomeExcelDTO : list) {
                PositionChannelIncome positionChannelIncome = new PositionChannelIncome();
                BeanUtils.copyProperties(positionChannelIncomeExcelDTO, positionChannelIncome);

                positionChannelIncome.setDataId(DateUtil.dateTo8String(positionChannelIncomeExcelDTO.getDatetime()) +
                        positionChannelIncomeExcelDTO.getSdkChannelType() + positionChannelIncomeExcelDTO.getPositionId());
                positionChannelIncome.setCreateUser(userVo.getUsername());
                positionChannelIncome.setCreateTime(new Date());

                //分成比率转换和计算收益
                //这里通过easypoi得到的分成比率为小数，所以后面保存的时候需要乘以100
                BigDecimal divideInto = positionChannelIncomeExcelDTO.getDivideInto();
                //BigDecimal divideInto = new BigDecimal(positionChannelIncomeExcelDTO.getDivideInto()
                //        .replace(CommonConstant.BaseCharacter.PERCENT, StringUtils.EMPTY));
                BigDecimal zero = new BigDecimal("0");
                BigDecimal hundred = new BigDecimal("1");
                if (divideInto == null || zero.compareTo(divideInto) > 0 || divideInto.compareTo(hundred) > 0) {
                    throw new McCmsException("分成比率不能为空，且必须在0-100之间！");
                }
                positionChannelIncome.setDivideInto(divideInto.multiply(new BigDecimal("100")));
                BigDecimal developerIncome = positionChannelIncome.getTotalIncome().multiply(divideInto).divide(hundred, BigDecimal.ROUND_CEILING, BigDecimal.ROUND_HALF_UP);
                positionChannelIncome.setDeveloperIncome(developerIncome);
                positionChannelIncome.setBxmIncome(positionChannelIncome.getTotalIncome().subtract(developerIncome));

                positionChannelIncomes.add(positionChannelIncome);
                positionChannelIncomeHashMap.put(positionChannelIncome.getDataId(), positionChannelIncome);
            }
        } catch (McCmsException e){
            throw e;
        } catch (Exception e){
            throw new McCmsException("导入数据格式不规范，请检查！");
        }

        //得到已经存在的数据
        QueryWrapper<PositionChannelIncome> existQueryWrapper = new QueryWrapper<>();
        existQueryWrapper.in(PositionChannelIncome.DATA_ID, positionChannelIncomeHashMap.keySet());
        List<PositionChannelIncome> existList = list(existQueryWrapper);
        List<String> existPositionIdList = existList.stream().map(PositionChannelIncome::getPositionId).collect(Collectors.toList());

        if (CollectionUtils.isNotEmpty(existPositionIdList)){
            //删除老数据
            remove(existQueryWrapper);
        }

        saveBatch(positionChannelIncomes, 100);

        positionChannelIncomeUploadVO.setAddQuantity(positionChannelIncomes.size() - existList.size());
        positionChannelIncomeUploadVO.setUpdateQuantity(existList.size());
        positionChannelIncomeUploadVO.setUpdatePositionIds(existPositionIdList);

        return positionChannelIncomeUploadVO;
    }

    private void checkRepeated(List<PositionChannelIncomeExcelDTO> list) {
        // 筛选重复的
        List<PositionChannelIncomeExcelDTO> repeated = list.stream()
                .filter(
                        t1 -> list.stream().filter(t2 -> t1.getPositionId().equals(t2.getPositionId())
                                && t1.getDatetime().equals(t2.getDatetime())
                                && t1.getSdkChannelType().equals(t2.getSdkChannelType())).count() > 1
                )
                .collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(repeated)) {
            //if (repeated.size() > 5) {
            //    repeated = repeated.subList(4, repeated.size() - 1);
            //}
            Set<String> messageSet = new HashSet<>();
            for (PositionChannelIncomeExcelDTO dto : repeated) {
                StringBuffer message = new StringBuffer();
                message.append(DateUtil.dateTo8String(dto.getDatetime())).append(dto.getSdkChannelType())
                        .append(dto.getPositionId())
                        .append("\n");
                messageSet.add(message.toString());
            }

            throw new McCmsException("有【"+repeated.size()+"】条数据重复：\n" + messageSet.toString());
        }
    }

    @Override
    public Boolean syncChannelData(UserVo userVo, PositionChannelIncomeSyncDTO positionChannelIncomeSyncDTO) {
        // 1、时间参数准备：返回key = startTime 和 endTime
        Map<String, Date> dateTime = StartAndEndTimeUtil.getStartAndEndTime(positionChannelIncomeSyncDTO.getStartTime(),
                positionChannelIncomeSyncDTO.getEndTime());
        // 将开始时间与结束时间之间的所有为空的时间补齐
        String[] dateArray = StartAndEndTimeUtil.setResultDateTime(dateTime);
        if (dateArray == null || dateArray.length == 0) {
            //查询的日期有误
            throw new McCmsException("查询的日期有误。");
        }
        if (PositionChannelIncomeSyncDTO.INTERACT_CHANNEL.equals(positionChannelIncomeSyncDTO.getChannel())) {
            for (String dateStr : dateArray) {
                iPositionIncomeService.syncAdsDataToPositionChannelIncome(dateStr, positionChannelIncomeSyncDTO);
            }
        } else if (PositionChannelIncomeSyncDTO.GDT_CHANNEL.equals(positionChannelIncomeSyncDTO.getChannel())) {
            for (String dateStr : dateArray) {
                iPositionIncomeService.syncAdnetIncomeDataToPositionChannelIncome(dateStr, positionChannelIncomeSyncDTO);
            }
        }
        return true;
    }
}
