package com.bxm.mccms.common.core.service.impl;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Objects;

import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.env.Environment;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.OrderItem;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.bxm.mccms.common.core.entity.PositionDspPos;
import com.bxm.mccms.common.core.mapper.PositionDspPosMapper;
import com.bxm.mccms.common.core.service.IPositionDspPosService;
import com.bxm.mccms.common.helper.constant.CommonConstant;
import com.bxm.mccms.common.helper.enums.DspStaticEnum;
import com.bxm.mccms.common.helper.exception.McCmsException;
import com.bxm.mccms.common.integration.ssp.position.PositionIntegration;
import com.bxm.mccms.common.model.position.PositionDspPosAnalyzeVO;
import com.bxm.mccms.common.model.position.PositionDspPosDTO;
import com.bxm.mccms.common.model.position.PositionDspPosVO;
import com.bxm.mccms.common.pushable.DispatcherPushable;
import com.bxm.mccms.facade.constant.Constants;
import com.bxm.mcssp.common.enums.position.PositionSceneTypeEnum;
import com.bxm.mcssp.facade.model.position.PositionFacadeVO;
import com.bxm.warcar.utils.DateHelper;

import lombok.extern.slf4j.Slf4j;

/**
 * <p>
 * 广告位流量分配 服务实现类
 * </p>
 *
 * @author Allen Hu
 * @since 2019-12-18
 */
@Slf4j
@Service
public class PositionDspPosServiceImpl extends ServiceImpl<PositionDspPosMapper, PositionDspPos> implements IPositionDspPosService {

    @Autowired
    private PositionIntegration positionIntegration;
    @Autowired
    private DispatcherPushable dispatcherPushable;

    @Autowired
    private Environment environment;

    @Override
    public Page<PositionDspPosAnalyzeVO> pageByAnalyze(Page page, PositionDspPosDTO query) {
        List<OrderItem> orders = page.getOrders();
        boolean isNotExistId = true;
        for (OrderItem order : orders) {
            if (PositionDspPos.ID.equals(order.getColumn())){
                isNotExistId = false;
            }
        }
        if (org.apache.commons.collections.CollectionUtils.isEmpty(orders) || isNotExistId) {
            orders.add(OrderItem.desc(PositionDspPos.ID));
        }

        Page<PositionDspPosAnalyzeVO> resp = getBaseMapper().pageByAnalyze(page, 
                DateHelper.format(DateHelper.PATTERN_STR10), query);
        return resp;
    }

    @Override
    public PositionDspPosVO getDetail(Long id) {
        PositionDspPos dspPos = getBaseMapper().selectById(id);
        if(null == dspPos){
            throw new McCmsException("数据不存在");
        }
        PositionDspPosVO dspPosDTO = new PositionDspPosVO();
        BeanUtils.copyProperties(dspPos, dspPosDTO);

        return dspPosDTO;
    }

    @Override
    public PositionDspPos getByOnly(String positionId, Long dspId) {
        LambdaQueryWrapper<PositionDspPos> wrapper = new QueryWrapper<PositionDspPos>().lambda();
        wrapper.eq(PositionDspPos::getPositionId, positionId);
        wrapper.eq(PositionDspPos::getDspId, dspId);
        List<PositionDspPos> list = getBaseMapper().selectList(wrapper);
        if(CollectionUtils.isEmpty(list)){
            return null;
        }
        return list.get(0);
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Boolean saveInfo(PositionDspPos t){
        PositionDspPos old = getByOnly(t.getPositionId(), t.getDspId());
        if(null != old){
            throw new McCmsException("广告平台已绑定");
        }
        //场景类型为Tab也的广告位只能配置场景DSP，不能添加其他广告dsp
        PositionFacadeVO position = positionIntegration.findByPositionId(t.getPositionId());
        PositionSceneTypeEnum positionSceneTypeEnum =
                PositionSceneTypeEnum.get(position.getPositionScene());
        if (positionSceneTypeEnum == null){
            throw new McCmsException("广告位场景类型异常");
        }
        switch (positionSceneTypeEnum) {
            case TAB_PAGE:
                validationDspId(t.getDspId(), positionSceneTypeEnum, DspStaticEnum.SCENE_DSP);
                break;
            case LOCK_SCREEN:
                validationDspId(t.getDspId(), positionSceneTypeEnum, DspStaticEnum.LOCK_SCREET_DSP);
                break;
            default:
                break;
        }

        t.setOpened(Constants.Opened.OPENED.getType());
        int rows = getBaseMapper().insert(t);
        //消息推送
        dispatcherPushable.push(getById(t.getId()));
        return rows > 0;
    }

    private void validationDspId(Long configDspId, PositionSceneTypeEnum positionSceneTypeEnum, DspStaticEnum dspStaticEnum) {
        Long dspId = null;
        List<String> activeProfiles = Arrays.asList(environment.getActiveProfiles());
        if (activeProfiles.contains(CommonConstant.Environment.PROFILES_DEV) || activeProfiles.contains(CommonConstant.Environment.PROFILES_TEST)) {
            dspId = dspStaticEnum.getTestId();
        } else {
            dspId = dspStaticEnum.getProId();
        }
        if(configDspId.equals(dspId) == false){
            throw new McCmsException(positionSceneTypeEnum.getDesc() + "广告位，仅能增加" + dspStaticEnum.getName());
        }
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Boolean updateInfo(PositionDspPos t){
        PositionDspPos old = getByOnly(t.getPositionId(), t.getDspId());
        if(null != old && !Objects.equals(old.getId(), t.getId())){
            throw new McCmsException("广告平台已绑定");
        }
        //不允许修改广告平台
        t.setDspId(old.getDspId());
        int rows = getBaseMapper().updateById(t);
        //消息推送
        dispatcherPushable.push(getById(t.getId()));
        return rows > 0;
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Boolean updateOpened(Long id, Integer opened){
        PositionDspPos t = getById(id);
        if(null == t){
            throw new McCmsException("数据不存在");
        }
        t.setOpened(opened);
        getBaseMapper().updateById(t);

        //消息推送
        dispatcherPushable.push(t);
        return Boolean.TRUE;
    }

    @Override
    public List<PositionDspPos> getByDspId(List<Long> dspIds) {
        QueryWrapper<PositionDspPos> wrapper = new QueryWrapper<PositionDspPos>();
        wrapper.in(PositionDspPos.DSP_ID,dspIds);
        return getBaseMapper().selectList(wrapper);
    }

    @Override
    public Boolean addDsp(String positionId, DspStaticEnum dspStaticEnum, String dspPosid) {
        Long dspId;
        List<String> activeProfiles = Arrays.asList(environment.getActiveProfiles());
        if (activeProfiles.contains(CommonConstant.Environment.PROFILES_DEV) || activeProfiles.contains(CommonConstant.Environment.PROFILES_TEST)) {
            //测试环境
            dspId = dspStaticEnum.getTestId();
        } else {
            //生产环境
            dspId = dspStaticEnum.getProId();
        }

        PositionDspPos t = new PositionDspPos();
        t.setPositionId(positionId);
        t.setDspId(dspId);
        t.setDspPosid(dspPosid);
        t.setDspBasePrice(new BigDecimal("0"));
        t.setPriority(1);
        t.setCreateUser("admin");
        t.setCreateTime(new Date());

        t.setPositionId(positionId);
        return saveInfo(t);
    }
}
