package com.bxm.mccms.common.core.service.impl;

import java.math.BigDecimal;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.client.RestTemplate;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.bxm.acl.facade.model.UserVo;
import com.bxm.datapark.web.model.SspPositionFinanceDaily;
import com.bxm.mccms.common.core.entity.DeveloperBill;
import com.bxm.mccms.common.core.entity.Dsp;
import com.bxm.mccms.common.core.entity.PositionCalcConfig;
import com.bxm.mccms.common.core.entity.PositionChannelIncome;
import com.bxm.mccms.common.core.entity.PositionDspPos;
import com.bxm.mccms.common.core.entity.PositionIncome;
import com.bxm.mccms.common.core.entity.PositionIncomeLog;
import com.bxm.mccms.common.core.entity.SceneSetting;
import com.bxm.mccms.common.core.mapper.PositionIncomeMapper;
import com.bxm.mccms.common.core.service.IDeveloperBillService;
import com.bxm.mccms.common.core.service.IDeveloperIncomeService;
import com.bxm.mccms.common.core.service.IDspService;
import com.bxm.mccms.common.core.service.IPositionCalcConfigService;
import com.bxm.mccms.common.core.service.IPositionChannelIncomeService;
import com.bxm.mccms.common.core.service.IPositionDspPosService;
import com.bxm.mccms.common.core.service.IPositionIncomeLogService;
import com.bxm.mccms.common.core.service.IPositionIncomeService;
import com.bxm.mccms.common.core.service.ISceneSettingService;
import com.bxm.mccms.common.helper.constant.DatagrabKeyGenerator;
import com.bxm.mccms.common.helper.exception.McCmsException;
import com.bxm.mccms.common.helper.util.AdnetUtil;
import com.bxm.mccms.common.helper.util.NumberUtil;
import com.bxm.mccms.common.helper.util.OceanengineUtil;
import com.bxm.mccms.common.helper.util.UserRoleUtil;
import com.bxm.mccms.common.integration.adsmedia.PositionIncomeIntegration;
import com.bxm.mccms.common.integration.datapark.DataparkWebIntegration;
import com.bxm.mccms.common.integration.ssp.developer.DeveloperIntegration;
import com.bxm.mccms.common.integration.ssp.position.PositionIntegration;
import com.bxm.mccms.common.integration.ssp.position.PositionSdkConfigIntegration;
import com.bxm.mccms.common.model.income.PositionChannelIncomeSyncDTO;
import com.bxm.mccms.common.model.income.PositionIncomeDTO;
import com.bxm.mccms.common.model.income.PositionIncomeQueryDTO;
import com.bxm.mccms.common.model.income.PositionIncomeVO;
import com.bxm.mccms.common.model.income.datagrab.AdnetIncome;
import com.bxm.mccms.common.model.income.datagrab.AuthenticationInfoCacheVO;
import com.bxm.mccms.common.model.income.datagrab.OceanengineIncome;
import com.bxm.mccms.facade.enums.SceneDspEnum;
import com.bxm.mcssp.common.enums.position.CustomPositionTypeEnum;
import com.bxm.mcssp.common.enums.position.PositionSceneTypeEnum;
import com.bxm.mcssp.common.enums.position.PositionSdkConfigChannelEnum;
import com.bxm.mcssp.common.util.DateUtil;
import com.bxm.mcssp.facade.model.developer.DeveloperFacadeQueryDTO;
import com.bxm.mcssp.facade.model.developer.DeveloperFacadeVO;
import com.bxm.mcssp.facade.model.position.PositionFacadeQueryDTO;
import com.bxm.mcssp.facade.model.position.PositionFacadeVO;
import com.bxm.mcssp.facade.model.position.sdkconfig.PositionSdkConfigFacadeQueryDTO;
import com.bxm.mcssp.facade.model.position.sdkconfig.PositionSdkConfigFacadeVO;
import com.bxm.warcar.cache.impls.redis.JedisFetcher;
import com.bxm.warcar.cache.impls.redis.JedisUpdater;
import com.bxm.warcar.utils.localdate.LocalDateTimeHelper;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import lombok.extern.slf4j.Slf4j;

/**
 * <h3>类的基本描述</h3>
 *
 * @author hcmony
 * @since V1.0.0, 2020/1/9 16:50
 */
@Slf4j
@Service
public class PositionIncomeServiceImpl extends ServiceImpl<PositionIncomeMapper, PositionIncome> implements IPositionIncomeService {

    @Autowired
    private DataparkWebIntegration dataparkWebIntegration;
    @Autowired
    private IPositionCalcConfigService positionCalcConfigService;
    @Autowired
    private IPositionIncomeLogService positionIncomeLogService;
    @Autowired
    private DeveloperIntegration developerIntegration;
    @Autowired
    private PositionIntegration positionIntegration;
    @Autowired
    private PositionIncomeMapper positionIncomeMapper;
    @Autowired
    private IDeveloperIncomeService developerIncomeService;
    @Autowired
    private IDspService dspService;
    @Autowired
    private IPositionDspPosService positionDspPosService;
    @Autowired
    private PositionSdkConfigIntegration positionSdkConfigIntegration;
    @Autowired
    private JedisFetcher jedisFetcher;
    @Autowired
    private JedisUpdater jedisUpdater;
    @Autowired
    private IDeveloperBillService billService;

    @Resource
    private RestTemplate restTemplate;
    @Autowired
    private ISceneSettingService sceneSettingService;
    @Autowired
    private PositionIncomeIntegration positionIncomeIntegration;
    @Autowired
    private IPositionChannelIncomeService positionChannelIncomeService;

    /**
     * 穿山甲和广点通在DSP中的对应ID
     */
    private static final int DSP_GDT_SDK = 4;
    private static final int DSP_JLYQ_SDK = 5;

    @Override
    public int init() {
        String datetime = LocalDateTimeHelper.formatToString(LocalDateTime.now().plusDays(-1), LocalDateTimeHelper.PATTERN_STR10);

        // 1、去数据表查所有广告位的数据 rpt_ssp_position_finance_daily
        SspPositionFinanceDaily search = new SspPositionFinanceDaily();
        search.setDatetime(datetime);
        List<SspPositionFinanceDaily> positionDspData = dataparkWebIntegration.getPositionDspData(search);

        // 2,查询所有SDK广告位与dsp配置与上一步组合
        final List<PositionFacadeVO> positionFacadeVOList = positionIntegration.getAllList(new PositionFacadeQueryDTO());
        Map<String, PositionFacadeVO> positionFacadeVOMap = positionFacadeVOList.stream().collect(Collectors.toMap(PositionFacadeVO::getPositionId, PositionFacadeVO -> PositionFacadeVO, (v1, v2) -> v1));
        composite(positionDspData, datetime, positionFacadeVOMap);
        if (CollectionUtils.isEmpty(positionDspData)) {
            log.warn("[{}] Position Data is Empty.", datetime);
            return -1;
        }

        // 3、广告位合作类型配置
        List<PositionCalcConfig> list = positionCalcConfigService.list();
        if (CollectionUtils.isEmpty(list)) {
            log.warn("Position settleType config is Empty.");
            return -1;
        }
        Map<String, PositionCalcConfig> positionCalcConfigMap = list.stream().collect(Collectors.toMap(PositionCalcConfig::getPositionId, PositionCalcConfig -> PositionCalcConfig));

        //4,查询dsp配置相关，不同dsp计算模式不一致

        //5、计算存入
        List<PositionIncome> positionIncomeList = Lists.newArrayList();
        positionDspData.forEach(data -> {
            PositionCalcConfig calcConfig = positionCalcConfigMap.get(data.getPositionId());
            if (calcConfig == null) {
                log.warn("Position [{}] settleType config is Empty.", data.getPositionId());
                return;
            }
            PositionIncome positionIncome = saveIncome(data, datetime, calcConfig);
            if (positionFacadeVOMap.get(positionIncome.getPositionId()) == null) {
                log.warn("Position [{}]  is Empty.", positionIncome.getPositionId());
                return;
            }
            positionIncome.setDockingMethodType(positionFacadeVOMap.get(positionIncome.getPositionId()).getDockingMethodType());
            PositionFacadeVO positionFacadeVO = positionFacadeVOMap.get(positionIncome.getPositionId());
            positionIncome.setDockingMethodType(positionFacadeVO.getDockingMethodType());
            positionIncome.setPositionScene(positionFacadeVO.getPositionScene());
            positionIncomeList.add(positionIncome);
        });

        // 6
        addSdkPositionIncome(positionFacadeVOList, positionIncomeList, datetime, positionCalcConfigMap);

        saveBatch(positionIncomeList);

        return positionIncomeList.size();
    }

    /**
     * //todo 如果后面广告位量大，会多出很多冗余数据，应该要查询最近有数据的广告位
     *
     * @param positionFacadeVOList
     * @param positionIncomeList
     * @param datetime
     */
    private void addSdkPositionIncome(List<PositionFacadeVO> positionFacadeVOList, List<PositionIncome> positionIncomeList, String datetime, Map<String, PositionCalcConfig> positionCalcConfigMap) {

        for (int i = 0; i < positionFacadeVOList.size(); i++) {
            PositionFacadeVO position = positionFacadeVOList.get(i);
            PositionIncome csjPositionIncome = new PositionIncome();
            csjPositionIncome.setPositionId(position.getPositionId());
            csjPositionIncome.setAppId(position.getAppId());
            csjPositionIncome.setDatetime(datetime);
            csjPositionIncome.setDeveloperId(position.getDeveloperId());
            csjPositionIncome.setBiddingConsume(PositionIncome.INIT_INCOME);
            csjPositionIncome.setTotalIncome(PositionIncome.INIT_INCOME);
            csjPositionIncome.setDspId(-1L);
            csjPositionIncome.setTotalSend(0L);
            csjPositionIncome.setTotalOpen(0L);
            csjPositionIncome.setTotalClick(0L);
            csjPositionIncome.setUv(0L);
            csjPositionIncome.setDockingMethodType(position.getDockingMethodType());
            csjPositionIncome.setPositionScene(position.getPositionScene());
            csjPositionIncome.setCreateUser("admin");
            csjPositionIncome.setSdkChannelType(PositionSdkConfigChannelEnum.CSJ.getCode());
            final PositionCalcConfig calcConfig = positionCalcConfigMap.get(position.getPositionId());
            if (calcConfig != null) {
                calculatePositionIncome(datetime, csjPositionIncome, calcConfig);
            }
            //广点通
            PositionIncome gdtPositionIncome = new PositionIncome();
            BeanUtils.copyProperties(csjPositionIncome, gdtPositionIncome);
            gdtPositionIncome.setSdkChannelType(PositionSdkConfigChannelEnum.GDT.getCode());
            //百度
            PositionIncome bdPositionIncome = new PositionIncome();
            BeanUtils.copyProperties(csjPositionIncome, bdPositionIncome);
            bdPositionIncome.setSdkChannelType(PositionSdkConfigChannelEnum.BD.getCode());
            //快手
            PositionIncome ksPositionIncome = new PositionIncome();
            BeanUtils.copyProperties(csjPositionIncome, ksPositionIncome);
            ksPositionIncome.setSdkChannelType(PositionSdkConfigChannelEnum.KS.getCode());
            //爱豆
            PositionIncome adPositionIncome = new PositionIncome();
            BeanUtils.copyProperties(csjPositionIncome, adPositionIncome);
            adPositionIncome.setSdkChannelType(PositionSdkConfigChannelEnum.AD.getCode());

            //todo 有兜底的才初始化
            if (PositionSceneTypeEnum.INSPIRE_VIDEO.getType().equals(position.getPositionScene())) {
                PositionIncome ddPositionIncome = new PositionIncome();
                BeanUtils.copyProperties(csjPositionIncome, ddPositionIncome);
                ddPositionIncome.setSdkChannelType(PositionSdkConfigChannelEnum.BACKUP.getCode());
                positionIncomeList.add(ddPositionIncome);
            }
            positionIncomeList.add(csjPositionIncome);
            positionIncomeList.add(gdtPositionIncome);
            positionIncomeList.add(bdPositionIncome);
            positionIncomeList.add(ksPositionIncome);
            positionIncomeList.add(adPositionIncome);

        }
    }


    private List<SspPositionFinanceDaily> composite(List<SspPositionFinanceDaily> positionDspData, String datetime, Map<String, PositionFacadeVO> positionFacadeVOMap) {
        final List<PositionDspPos> positionDspPosList = positionDspPosService.list();
        if (CollectionUtils.isEmpty(positionDspPosList)) {
            log.warn("All PositionDspPos is Empty.");
            return positionDspData;
        }
        Map<String, SspPositionFinanceDaily> positionDspDataMap = new HashMap<>(positionDspData.size());
        positionDspData.forEach(data -> {
            positionDspDataMap.put(data.getPositionId() + "-" + data.getDspId(), data);
        });

        positionDspPosList.forEach(position -> {
            PositionFacadeVO positionFacadeVO = positionFacadeVOMap.get(position.getPositionId());
            SspPositionFinanceDaily positionFinanceDaily = positionDspDataMap.get(position.getPositionId() + "-" + position.getDspId());
            //sdk为2 并且没有数据
            if (positionFacadeVO == null) {
                log.warn("position={} vo is null .", position.getPositionId());
                return;
            }
            if (positionFacadeVO.getDockingMethodType() == PositionIncome.DOCKING_METHOD_TYPE_SDK && positionFinanceDaily == null) {
                SspPositionFinanceDaily daily = new SspPositionFinanceDaily();
                daily.setDatetime(datetime);
                daily.setAppId(positionFacadeVO.getAppId());
                daily.setDeveloperId(positionFacadeVO.getDeveloperId());
                daily.setDspId(position.getDspId());
                daily.setPositionId(position.getPositionId());
                //todo 没有添加默认值
                positionDspData.add(daily);
            }
        });
        return positionDspData;
    }

    private PositionIncome saveIncome(SspPositionFinanceDaily data, String datetime, PositionCalcConfig calcConfig) {

        PositionIncome income = new PositionIncome();
        income.setPositionId(data.getPositionId());
        income.setAppId(data.getAppId());
        income.setDeveloperId(data.getDeveloperId());
        income.setBiddingConsume(null == data.getBidIncome() ? PositionIncome.INIT_INCOME : data.getBidIncome());
        income.setTotalIncome(income.getBiddingConsume());
        income.setDspId(data.getDspId());
        income.setTotalSend(null == data.getRequestPv() ? 0 : data.getRequestPv().longValue());
        income.setTotalOpen(null == data.getExposurePv() ? 0 : data.getExposurePv().longValue());
        income.setTotalClick(null == data.getClickPv() ? 0 : data.getClickPv().longValue());
        income.setUv(null == data.getRequestUv() ? 0 : data.getRequestUv().longValue());
        income.setCreateUser("admin");

        //相关收益计算
        calculatePositionIncome(datetime, income, calcConfig);
        income.setDatetime(datetime);
        income.setStatus(PositionIncome.NO_SUBMIT);

        return income;
    }

    private void calculatePositionIncome(String datetime, PositionIncome income, PositionCalcConfig calcConfig) {
        BigDecimal positionIncome = PositionIncome.INIT_INCOME;
        String settleConfig = "";

        switch (calcConfig.getOldCooperationType()) {
            case PositionIncome.SETTLE_TYPE_DIVIDEINTO: {
                positionIncome = NumberUtil.divide(calcConfig.getOldDivideInto(), PositionIncome.DIVIDEINTO_BASE).multiply(income.getTotalIncome()).setScale(BigDecimal.ROUND_CEILING, BigDecimal.ROUND_FLOOR);
                settleConfig = calcConfig.getOldDivideInto().toString() + "%";
                break;
            }
            case PositionIncome.SETTLE_TYPE_BIDDING: {
                if (PositionIncome.CPM == calcConfig.getOldBiddingType()) {
                    positionIncome = NumberUtil.divide(new BigDecimal(income.getTotalOpen()), PositionIncome.CPM_BASE).multiply(calcConfig.getOldBasePrice()).setScale(BigDecimal.ROUND_CEILING, BigDecimal.ROUND_FLOOR);
                }
                if (PositionIncome.CPC == calcConfig.getOldBiddingType()) {
                    positionIncome = new BigDecimal(income.getTotalOpen()).multiply(calcConfig.getOldBasePrice()).setScale(BigDecimal.ROUND_CEILING, BigDecimal.ROUND_FLOOR);
                }
                settleConfig = calcConfig.getOldBasePrice().toString();
                //CPA
                break;
            }
            case PositionIncome.SETTLE_TYPE_CONTRACT: {
                // 判断包段时间
                if (compareLocalDate(calcConfig.getOldStartDate(), datetime) > 0
                        || compareLocalDate(calcConfig.getOldEndDate(), datetime) < 0) {
                    log.warn(" [startDate:{}] > [{}] or [endDate:{}] < [{}] .", calcConfig.getOldStartDate(), datetime, calcConfig.getOldEndDate(), datetime);
                    break;
                }
                positionIncome = calcConfig.getOldBasePrice();
                StringBuilder sb = new StringBuilder();
                sb.append(calcConfig.getOldStartDate()).append("-").append(calcConfig.getEndDate()).append(":").append(calcConfig.getBasePrice());
                settleConfig = sb.toString();
                break;
            }
            default: {
                log.warn("Not found this CooperationType [{}].", calcConfig.getOldCooperationType());
            }
        }
        income.setSettleConfig(settleConfig);
        income.setSettleType(calcConfig.getOldCooperationType());
        income.setPositionPreIncome(positionIncome);
        income.setBxmPreIncome(income.getTotalIncome().subtract(income.getPositionPreIncome()));
    }


    @Transactional(rollbackFor = Exception.class, timeout = 30)
    @Override
    public Boolean update(PositionIncomeDTO positionIncomeDTO) {

        UserVo user = UserRoleUtil.getUser();

        DeveloperFacadeQueryDTO dto = new DeveloperFacadeQueryDTO();
        if (UserRoleUtil.isMj(user)) {
            dto.setMjCode(user.getUsername());
        }
        List<DeveloperFacadeVO> providerAppList = developerIntegration.list(dto);
        if (CollectionUtils.isEmpty(providerAppList)) {
            log.warn("Developer List is Empty . mj = {}", dto.getMjCode());
            throw new McCmsException("Developer List is Empty. ");
        }

        PositionIncome positionIncome = getById(positionIncomeDTO.getId());
        if (positionIncome == null) {
            throw new McCmsException("没有对应的广告位收入记录，id = " + positionIncomeDTO.getId());
        }

        final String month = positionIncome.getDatetime().substring(0, 7);
        QueryWrapper<DeveloperBill> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq(DeveloperBill.MONTH, month);
        queryWrapper.eq(DeveloperBill.DEVELOPER_ID, positionIncome.getDeveloperId());
        final DeveloperBill bill = billService.getOne(queryWrapper);

        if (bill != null) {
            throw new McCmsException(month + "月账单已经生成，不允许修改 ");
        }
        //
        //如果不是运营经理不允许修改已经提交的账单
        if (UserRoleUtil.isMj(user) && PositionIncome.NO_SUBMIT != positionIncome.getStatus()) {
            throw new McCmsException("账单不是未提交的状态，id = " + positionIncome.getId());
        }

        // 目前只支持修改总消耗
        positionIncome.setTotalIncome(positionIncomeDTO.getTotalIncome() == null ? positionIncome.getTotalIncome() : positionIncomeDTO.getTotalIncome());
        positionIncome.setPositionPreIncome(positionIncomeDTO.getPositionPreIncome() == null ? positionIncome.getPositionPreIncome() : positionIncomeDTO.getPositionPreIncome());

        // 如果是sdk ，可以修改其他值总曝光量.总点击量.上游预估平台消耗.开发者预估收益
        if (positionIncome.getDockingMethodType() == PositionIncome.DOCKING_METHOD_TYPE_SDK) {
            positionIncome.setTotalSend(positionIncomeDTO.getTotalSend() == null ? positionIncome.getTotalSend() : positionIncomeDTO.getTotalSend());
            positionIncome.setTotalOpen(positionIncomeDTO.getTotalOpen() == null ? positionIncome.getTotalOpen() : positionIncomeDTO.getTotalOpen());
            positionIncome.setTotalClick(positionIncomeDTO.getTotalClick() == null ? positionIncome.getTotalClick() : positionIncomeDTO.getTotalClick());
        }
        positionIncome.setModifyUser(user.getUsername());
        positionIncomeDTO.setDatetime(positionIncome.getDatetime());
        positionIncomeDTO.setPositionId(positionIncome.getPositionId());
        positionIncome.setStatus(PositionIncome.NO_SUBMIT);
        this.savePositionIncomeLog(positionIncomeDTO, positionIncome, user.getUsername());
        this.updateById(positionIncome);
        //this.updateDeveloperIncome(providerAppList, user, positionIncomeDTO);
        return true;
    }


    /**
     * 操作人:admin,操作内容：修改平台消耗为9999999.01,时间:2020-01-13 15:53:06
     * 操作人:admin,操作内容：修改平台消耗为9999999.01,时间:2020-01-13 16:17:18,操作人:admin,操作内容：修改平台消耗为9999999.00,时间:2020-01-13 16:18:21,操作人:admin,操作内容：修改平台消耗为9999999.00,时间:2020-01-13 16:19:42
     */
    private void savePositionIncomeLog(PositionIncomeDTO positionIncomeDTO, PositionIncome positionIncome, String userName) {

        PositionIncomeLog positionIncomeLog = new PositionIncomeLog();
        positionIncomeLog.setPositionIncomeId(positionIncome.getId());
        StringBuilder sb = new StringBuilder();

        sb.append("操作人:").append(userName);
        if (positionIncomeDTO.getTotalIncome() != null) {
            PositionCalcConfig positionCalcConfig =
                    positionCalcConfigService.findByPositionId(positionIncome.getPositionId());
            if (positionCalcConfig == null) {
                log.error("广告位 {},没有收益配置。", positionIncome.getPositionId());
            }
            calculatePositionIncome(positionIncome.getDatetime(), positionIncome, positionCalcConfig);
            sb.append(",操作内容：修改平台消耗为").append(positionIncome.getTotalIncome());
        }
        if (positionIncomeDTO.getPositionPreIncome() != null) {
            positionIncome.setBxmPreIncome(positionIncome.getTotalIncome().subtract(positionIncome.getPositionPreIncome()));
            sb.append(",操作内容：修改开发者预估收益为").append(positionIncome.getPositionPreIncome());
        }
        if (positionIncomeDTO.getTotalSend() != null) {
            sb.append(",操作内容：修改总请求为").append(positionIncome.getTotalSend());
        }
        if (positionIncomeDTO.getTotalOpen() != null) {
            sb.append(",操作内容：修改总曝光为").append(positionIncome.getTotalOpen());
        }
        if (positionIncomeDTO.getTotalClick() != null) {
            sb.append(",操作内容：修改总点击为").append(positionIncome.getTotalClick());
        }

        sb.append(",时间:").append(LocalDateTimeHelper.formatToString(LocalDateTimeHelper.PATTERN_STR19));
        if (positionIncome.getModifyLogId() != null && positionIncome.getModifyLogId() > 0) {
            positionIncomeLog.setId(positionIncome.getModifyLogId());
        }
        positionIncomeLog.setRemark(sb.toString());
        positionIncomeLogService.saveOrAppend(positionIncomeLog);
        positionIncome.setModifyLogId(positionIncomeLog.getId());
    }

    @Override
    public IPage<PositionIncomeVO> findAll(UserVo user, PositionIncomeQueryDTO positionIncomeDTO) {
        String mjCode = positionIncomeDTO.getMj();

        // 当登录的mj/bd和传过来参数mj/bd不一致时不查
        if (UserRoleUtil.isMj(user)) {
            if (StringUtils.isNotBlank(mjCode) && !mjCode.equals(user.getUsername())) {
                return new Page<>();
            }
            mjCode = user.getUsername();
        }

        positionIncomeDTO.setMj(mjCode);

        //1. position income list by condition
        IPage<PositionIncomeVO> resultPage = new Page<>();

        IPage<PositionIncome> page = new Page<>();
        PositionFacadeQueryDTO dto = new PositionFacadeQueryDTO();
        dto.setPositionId(positionIncomeDTO.getPositionId());
        dto.setMjCode(positionIncomeDTO.getMj());
        dto.setDeveloperKeyword(positionIncomeDTO.getDeveloperKeyword());
        dto.setAppKeyword(positionIncomeDTO.getAppKeyword());
        dto.setPositionKeyword(positionIncomeDTO.getPositionKeyword());
        dto.setPositionScene(positionIncomeDTO.getPositionScene());
        if (dto.getPositionScene() == null) {
            dto.setPositionScenes(PositionSceneTypeEnum.getTypeListByCustomPositionTypeEnum(CustomPositionTypeEnum.get(positionIncomeDTO.getCustomPositionType())));
        }

        List<PositionFacadeVO> positionFacadeVOList = positionIntegration.getAllList(dto);
        if (CollectionUtils.isEmpty(positionFacadeVOList)) {
            return resultPage;
        }
        if (StringUtils.isBlank(positionIncomeDTO.getPositionId())) {
            log.info("广告位收入汇总数据");
            List<String> positionIdList = positionFacadeVOList.stream().map(PositionFacadeVO::getPositionId).distinct().collect(Collectors.toList());

            positionIncomeDTO.setPositionIdList(positionIdList);

            page.setCurrent(positionIncomeDTO.getCurrent());
            page.setSize(positionIncomeDTO.getSize());
            page = positionIncomeMapper.findPage(page, positionIncomeDTO);
        } else {
            log.info("广告位收入详情");
            QueryWrapper<PositionIncome> queryWrapper = new QueryWrapper<>();

            queryWrapper.in(PositionIncome.POSITION_ID, positionIncomeDTO.getPositionId());

            if (StringUtils.isNotBlank(positionIncomeDTO.getDatetime())) {
                queryWrapper.eq(PositionIncome.DATETIME, positionIncomeDTO.getDatetime());
            }
            if (positionIncomeDTO.getSettleType() != null) {
                queryWrapper.eq(PositionIncome.SETTLE_TYPE, positionIncomeDTO.getSettleType());
            }

            page.setCurrent(positionIncomeDTO.getCurrent());
            page.setSize(positionIncomeDTO.getSize());

            page = page(page, queryWrapper);
        }
        if (page.getRecords().isEmpty()) {
            return resultPage;
        }

        Map<String, PositionFacadeVO> positionFacadeMap = positionFacadeVOList.stream().collect(Collectors.toMap(PositionFacadeVO::getPositionId, PositionFacadeVO -> PositionFacadeVO, (v1, v2) -> v1));

        //dsp 平台数据填充
        final List<Dsp> dspList = dspService.list();
        if (dspList.isEmpty()) {
            log.warn("Dsp is Empty.");
            return resultPage;
        }
        Map<Long, String> dspMap = dspList.stream().collect(Collectors.toMap(Dsp::getId, Dsp::getDspName));

        //最终结果合并
        BeanUtils.copyProperties(page, resultPage);
        resultPage.setRecords(packingResult(page, positionFacadeMap, dspMap));
        return resultPage;
    }

    private List<PositionIncomeVO> packingResult(IPage<PositionIncome> page, Map<String, PositionFacadeVO> positionFacadeMap, Map<Long, String> dspMap) {
        List<PositionIncomeVO> resultList = Lists.newArrayList();
        page.getRecords().forEach(income -> {
            PositionIncomeVO vo = new PositionIncomeVO();
            BeanUtils.copyProperties(income, vo);
            PositionFacadeVO positionFacadeVO = positionFacadeMap.get(vo.getPositionId());
            if (positionFacadeVO == null) {
                log.info("[Position:{}] is not found .", vo.getPositionId());
                return;
            }
            vo.setPositionName(positionFacadeVO.getPositionName());
            vo.setAppName(positionFacadeVO.getAppName());
            vo.setDeveloperName(positionFacadeVO.getDeveloperName());
            vo.setMj(positionFacadeVO.getMjName());
            vo.setDsp(dspMap.get(income.getDspId()));
            resultList.add(vo);
        });
        return resultList;
    }

    /**
     * 发布后生成账单
     *
     * @return
     */
    @Transactional(rollbackFor = Exception.class, timeout = 30)
    @Override
    public Boolean publish(PositionIncomeDTO positionIncomeDTO) {
        UserVo user = UserRoleUtil.getUser();
        DeveloperFacadeQueryDTO dto = new DeveloperFacadeQueryDTO();
        if (UserRoleUtil.isMj(user)) {
            dto.setMjCode(user.getUsername());
        }
        List<DeveloperFacadeVO> providerAppList = developerIntegration.list(dto);
        if (CollectionUtils.isEmpty(providerAppList)) {
            log.warn("Developer List is Empty . mj = {}", dto.getMjCode());
            throw new McCmsException("Developer List is Empty. ");
        }

        //1.修改未确认的广告位数据。
        final int r1 = positionIncomeMapper.updateStatusByPositionId(positionIncomeDTO.getPositionId(),
                PositionIncome.READY_SUBMIT,
                user.getUsername(),
                positionIncomeDTO.getDatetime());

        this.updateDeveloperIncome(providerAppList, user, positionIncomeDTO);

        return r1 > 0;
    }

    private void updateDeveloperIncome(List<DeveloperFacadeVO> providerAppList, UserVo user, PositionIncomeDTO positionIncomeDTO) {
        List<Long> developerIdList = providerAppList.stream().map(DeveloperFacadeVO::getId).collect(Collectors.toList());
        // positionIncomeDTO.setStatus(PositionIncome.READY_SUBMIT);
        positionIncomeDTO.setDeveloperIdList(developerIdList);
        final List<PositionIncome> positionIncomes = positionIncomeMapper.findByDeveloperIdList(positionIncomeDTO);

        if (CollectionUtils.isEmpty(positionIncomes)) {
            throw new McCmsException("Position Income is Empty. ");
        }
        //2.修改开发者列表的数据。
        developerIncomeService.updateConsume(positionIncomes);
    }


    /**
     * '
     * 判断
     *
     * @param l1
     * @param datetime
     * @return
     */
    private int compareLocalDate(LocalDate l1, String datetime) {
        LocalDate now = LocalDate.parse(datetime, DateTimeFormatter.ofPattern(LocalDateTimeHelper.PATTERN_STR10));
        return l1.compareTo(now);
    }

    @Override
    public Boolean syncDatagrabArgs(AuthenticationInfoCacheVO authenticationInfoCacheVO) {
        jedisUpdater.update(DatagrabKeyGenerator.getKey(), authenticationInfoCacheVO);
        return true;
    }

    @Override
    public Boolean syncOceanengineData(String date) {
        AuthenticationInfoCacheVO authenticationInfoCacheVO = jedisFetcher.fetch(DatagrabKeyGenerator.getKey(), AuthenticationInfoCacheVO.class);
        if (StringUtils.isBlank(authenticationInfoCacheVO.getOceanengineCookie()) ||
                StringUtils.isBlank(authenticationInfoCacheVO.getOceanengineXcsrfToken())) {
            log.error("没有配置穿山甲的Cookie相关值。");
            return false;
        }
        //查询昨天的时间
        String dateTo8String = StringUtils.defaultIfBlank(date, DateUtil.dateTo8String(DateUtil.getDateBefore(new Date(), 1)));

        String cookies = authenticationInfoCacheVO.getOceanengineCookie();
        String XCSRFToken = authenticationInfoCacheVO.getOceanengineXcsrfToken();
        Map<String, Object> requestArgs = new HashMap<>(4);
        requestArgs.put("StartDate", dateTo8String);
        requestArgs.put("EndDate", dateTo8String);
        requestArgs.put("Page", 1);
        requestArgs.put("PageSize", 100);

        List<OceanengineIncome.Entitie> entities = OceanengineUtil.queryAllData(restTemplate, cookies, XCSRFToken, requestArgs);
        if (CollectionUtils.isEmpty(entities)) {
            log.warn("未查询到广告位数据。");
            return true;
        }
        System.out.println("共查询到 " + entities.size() + " 条记录！");
        Map<String, OceanengineIncome.Entitie> codeIdMap = entities.stream().collect(HashMap::new, (k, v) -> k.put(String.valueOf(v.getCodeId()), v), HashMap::putAll);
        List<String> channelPositionIds = new ArrayList<>(codeIdMap.keySet());

        //得到SDK配置中的平台的广告位ID
        PositionSdkConfigFacadeQueryDTO positionSdkConfigFacadeQueryDTO = new PositionSdkConfigFacadeQueryDTO();
        positionSdkConfigFacadeQueryDTO.setChannelType(PositionSdkConfigChannelEnum.CSJ.getCode());
        positionSdkConfigFacadeQueryDTO.setChannelPositionIds(channelPositionIds);
        List<PositionSdkConfigFacadeVO> positionSdkConfigFacadeVOList = positionSdkConfigIntegration.getList(positionSdkConfigFacadeQueryDTO);
        if (CollectionUtils.isEmpty(positionSdkConfigFacadeVOList)) {
            log.warn("Position SDK Config is empty!");
            return true;
        }
        Map<String, String> positionSdkConfigFacadeVOMap = positionSdkConfigFacadeVOList.stream().collect(HashMap::new, (k, v) -> k.put(v.getChannelPositionId(), v.getPositionId()), HashMap::putAll);

        //看看昨天是否有收益
        QueryWrapper<PositionIncome> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq(PositionIncome.STATUS, PositionIncome.NO_SUBMIT);
        //queryWrapper.eq(PositionIncome.DSP_ID, DSP_JLYQ_SDK);
        queryWrapper.eq(PositionIncome.SDK_CHANNEL_TYPE, PositionSdkConfigChannelEnum.CSJ.getCode());
        queryWrapper.eq(PositionIncome.DATETIME, dateTo8String);
        queryWrapper.in(PositionIncome.POSITION_ID, positionSdkConfigFacadeVOMap.values());
        List<PositionIncome> positionIncomeList = list(queryWrapper);
        if (CollectionUtils.isEmpty(positionIncomeList)) {
            log.warn("[{}] Position income is empty!", dateTo8String);
            return true;
        }

        Map<String, PositionIncome> positionIncomeMap = positionIncomeList.stream().collect(HashMap::new, (k, v) -> k.put(v.getPositionId(), v), HashMap::putAll);
        //填充数据
        List<PositionIncome> needUpdatePositionList = new ArrayList<>();
        //Set<String> needUpdatePositionSet = new HashSet<>();
        for (Map.Entry<String, String> stringPositionIncomeEntry : positionSdkConfigFacadeVOMap.entrySet()) {
            OceanengineIncome.Entitie entitie = codeIdMap.get(stringPositionIncomeEntry.getKey());
            if (entitie == null) {
                log.warn(stringPositionIncomeEntry.getKey() + "没有找到平台对应的收益");
                continue;
            }
            String positionId = stringPositionIncomeEntry.getValue();
            //if (needUpdatePositionSet.contains(positionId)){
            //    continue;
            //}
            PositionIncome positionIncome = positionIncomeMap.get(positionId);
            if (positionIncome == null) {
                log.warn(stringPositionIncomeEntry.getKey() + "没有找到对应的收益");
                continue;
            }
            //needUpdatePositionSet.add(positionId);
            //填充 展示数，点击数，预计收益 等数据
            positionIncome.setTotalOpen(Long.valueOf(entitie.getImpression()));
            positionIncome.setTotalClick(Long.valueOf(entitie.getClick()));
            positionIncome.setTotalIncome(new BigDecimal(String.valueOf(entitie.getIncome())));

            needUpdatePositionList.add(positionIncome);
        }

        if (CollectionUtils.isEmpty(needUpdatePositionList)) {
            log.warn("没有匹配的广告位收益数据需要更新！");
            return true;
        }
        //更新
        String userName = "JOB";
        for (PositionIncome positionIncome : needUpdatePositionList) {
            PositionIncomeDTO positionIncomeDTO = new PositionIncomeDTO();
            positionIncomeDTO.setTotalOpen(positionIncome.getTotalOpen());
            positionIncomeDTO.setTotalClick(positionIncome.getTotalClick());
            positionIncomeDTO.setTotalIncome(positionIncome.getTotalIncome());
            positionIncome.setModifyUser(userName);
            positionIncome.setModifyTime(new Date());
            //记录日志
            savePositionIncomeLog(positionIncomeDTO, positionIncome, userName);
            updateById(positionIncome);
        }

        log.warn("共更新{}条数据！", needUpdatePositionList.size());
        return true;
    }

    @Override
    public Boolean syncAdnetData(String date) {
        List<PositionIncome> needUpdatePositionList = getAdnetIncomeData(date, true, null);
        if (CollectionUtils.isEmpty(needUpdatePositionList)) {
            log.warn("没有匹配的广告位收益数据需要更新！");
            return true;
        }
        //更新
        String userName = "JOB";
        for (PositionIncome positionIncome : needUpdatePositionList) {
            PositionIncomeDTO positionIncomeDTO = new PositionIncomeDTO();
            positionIncomeDTO.setTotalOpen(positionIncome.getTotalOpen());
            positionIncomeDTO.setTotalClick(positionIncome.getTotalClick());
            positionIncomeDTO.setTotalIncome(positionIncome.getTotalIncome());
            positionIncome.setModifyUser(userName);
            positionIncome.setModifyTime(new Date());
            //记录日志
            savePositionIncomeLog(positionIncomeDTO, positionIncome, userName);
            updateById(positionIncome);
        }
        log.warn("共更新{}条数据！", needUpdatePositionList.size());
        return true;
    }

    @Override
    public Boolean syncAdnetIncomeDataToPositionChannelIncome(String datetime, PositionChannelIncomeSyncDTO positionChannelIncomeSyncDTO) {
        BigDecimal divideInto = new BigDecimal("100");

        //从广点通平台获取收益数据
        List<PositionIncome> positionIncome = getAdnetIncomeData(datetime, false, positionChannelIncomeSyncDTO.getPositionId());
        if (CollectionUtils.isEmpty(positionIncome)) {
            throw new McCmsException("未找到广点通渠道该日期的数据。");
        }
        //从广告位收益表获取收益数据
        QueryWrapper<PositionIncome> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq(PositionIncome.SDK_CHANNEL_TYPE, PositionSdkConfigChannelEnum.GDT.getCode());
        queryWrapper.eq(PositionIncome.DATETIME, datetime);
        if(StringUtils.isNotBlank(positionChannelIncomeSyncDTO.getPositionId())){
            queryWrapper.in(PositionIncome.POSITION_ID, positionChannelIncomeSyncDTO.getPositionId());
        }
        List<PositionIncome> existPositionIncome = list(queryWrapper);
        Map<String, PositionIncome> existPositionIncomeMap = null;
        if (CollectionUtils.isNotEmpty(existPositionIncome)) {
            existPositionIncomeMap = existPositionIncome.stream().collect(HashMap::new, (k, v) -> k.put(v.getPositionId(), v), HashMap::putAll);
        }

        ArrayList<PositionChannelIncome> positionChannelIncomes = new ArrayList<>(50);
        // 6.更新到数据库中
        String channel = "广点通";
        for (PositionIncome vo : positionIncome) {
            PositionChannelIncome positionChannelIncome = new PositionChannelIncome();
            positionChannelIncome.setDataId(datetime + channel + vo.getPositionId());
            positionChannelIncome.setPositionId(vo.getPositionId());
            positionChannelIncome.setDatetime(DateUtil.convertStr2Date(vo.getDatetime()));
            positionChannelIncome.setSdkChannelType(channel);
            positionChannelIncome.setTotalOpen(vo.getTotalOpen());
            positionChannelIncome.setTotalClick(vo.getTotalClick());
            positionChannelIncome.setTotalIncome(vo.getTotalIncome());

            if (existPositionIncomeMap == null) {
                //从广点通平台获取收益数据-需要重新计算
                calcIncome(positionChannelIncome, divideInto);
            } else {
                PositionIncome temp = existPositionIncomeMap.get(vo.getPositionId());
                if (temp == null) {
                    //从广点通平台获取收益数据-需要重新计算
                    calcIncome(positionChannelIncome, divideInto);
                } else {
                    //直接复制数据
                    positionChannelIncome.setDeveloperIncome(temp.getPositionPreIncome());
                    positionChannelIncome.setBxmIncome(vo.getTotalIncome().subtract(temp.getPositionPreIncome()));
                    if (temp.getPositionPreIncome().compareTo(BigDecimal.ZERO) == 0 || temp.getTotalIncome().compareTo(BigDecimal.ZERO) == 0) {
                        //默认为100
                        positionChannelIncome.setDivideInto(divideInto);
                    } else {
                        //分成比率=媒体收益/总收益
                        BigDecimal divideIntoNew = temp.getPositionPreIncome().divide(vo.getTotalIncome(), BigDecimal.ROUND_CEILING, BigDecimal.ROUND_HALF_UP)
                                .multiply(new BigDecimal("100"));
                        positionChannelIncome.setDivideInto(divideIntoNew);
                    }
                }
            }

            positionChannelIncome.setCreateUser("admin");
            positionChannelIncome.setCreateTime(new Date());
            positionChannelIncomes.add(positionChannelIncome);
        }

        if (CollectionUtils.isNotEmpty(positionChannelIncomes)) {
            Set<String> dataIdSet = positionChannelIncomes.stream().map(PositionChannelIncome::getDataId).collect(Collectors.toSet());
            checkRepeated(dataIdSet);

            try {
                positionChannelIncomeService.saveBatch(positionChannelIncomes);
            } catch (Exception e){
                log.error(e.getMessage());
                throw new McCmsException("数据已经存在。");
            }
        }
        return true;
    }

    public List<PositionIncome> getAdnetIncomeData(String date, boolean isPositionIncome, String syncPositionId){
        AuthenticationInfoCacheVO authenticationInfoCacheVO = jedisFetcher.fetch(DatagrabKeyGenerator.getKey(), AuthenticationInfoCacheVO.class);
        if (StringUtils.isBlank(authenticationInfoCacheVO.getAdnetCookie())) {
            log.error("没有配置广点通的Cookie相关值。");
            throw new McCmsException("没有配置广点通的Cookie相关值");
        }
        //查询昨天的时间
        String dateTo8String = StringUtils.defaultIfBlank(date, DateUtil.dateTo8String(DateUtil.getDateBefore(new Date(), 1)));

        String cookies = authenticationInfoCacheVO.getAdnetCookie();
        Map<String, Object> requestArgs = new HashMap<>(4);
        requestArgs.put("start_date", dateTo8String);
        requestArgs.put("end_date", dateTo8String);
        requestArgs.put("page", 1);
        requestArgs.put("page_size", 100);

        List<AdnetIncome.Entitie> entities = AdnetUtil.queryAllData(restTemplate, cookies, requestArgs);
        if (CollectionUtils.isEmpty(entities)) {
            log.warn("未查询到广告位数据。");
            throw new McCmsException("未查询到广告位数据");
        }
        System.out.println("共查询到 " + entities.size() + " 条记录！");
        Map<String, AdnetIncome.Entitie> codeIdMap = entities.stream().collect(HashMap::new, (k, v) -> k.put(String.valueOf(v.getPlacement_id()), v), HashMap::putAll);
        List<String> channelPositionIds = new ArrayList<>(codeIdMap.keySet());

        //得到SDK配置中的平台的广告位ID
        PositionSdkConfigFacadeQueryDTO positionSdkConfigFacadeQueryDTO = new PositionSdkConfigFacadeQueryDTO();
        positionSdkConfigFacadeQueryDTO.setChannelType(PositionSdkConfigChannelEnum.GDT.getCode());
        positionSdkConfigFacadeQueryDTO.setChannelPositionIds(channelPositionIds);
        positionSdkConfigFacadeQueryDTO.setPositionId(syncPositionId);
        List<PositionSdkConfigFacadeVO> positionSdkConfigFacadeVOList = positionSdkConfigIntegration.getList(positionSdkConfigFacadeQueryDTO);
        if (CollectionUtils.isEmpty(positionSdkConfigFacadeVOList)) {
            log.warn("广告位SDK配置为空");
            throw new McCmsException("广告位SDK配置为空");
        }
        Map<String, String> positionSdkConfigFacadeVOMap = positionSdkConfigFacadeVOList.stream().collect(HashMap::new, (k, v) -> k.put(v.getChannelPositionId(), v.getPositionId()), HashMap::putAll);
        //List<String> positionIds = new ArrayList<>(positionSdkConfigFacadeVOMap.keySet());

        QueryWrapper<PositionIncome> queryWrapper = new QueryWrapper<>();
        if (isPositionIncome){
            //广告位收益只查询未提交的
            queryWrapper.eq(PositionIncome.STATUS, PositionIncome.NO_SUBMIT);
        }
        //queryWrapper.eq(PositionIncome.DSP_ID, DSP_GDT_SDK);
        queryWrapper.eq(PositionIncome.SDK_CHANNEL_TYPE, PositionSdkConfigChannelEnum.GDT.getCode());
        queryWrapper.eq(PositionIncome.DATETIME, dateTo8String);
        queryWrapper.in(PositionIncome.POSITION_ID, positionSdkConfigFacadeVOMap.values());
        List<PositionIncome> positionIncomeList = list(queryWrapper);
        if (CollectionUtils.isEmpty(positionIncomeList)) {
            log.warn("[{}] Position income is empty!", dateTo8String);
            return Collections.emptyList();
        }

        Map<String, PositionIncome> positionIncomeMap = positionIncomeList.stream().collect(HashMap::new, (k, v) -> k.put(v.getPositionId(), v), HashMap::putAll);
        //填充数据
        List<PositionIncome> needUpdatePositionList = new ArrayList<>();
        //Set<String> needUpdatePositionSet = new HashSet<>();
        for (Map.Entry<String, String> stringPositionIncomeEntry : positionSdkConfigFacadeVOMap.entrySet()) {
            AdnetIncome.Entitie entitie = codeIdMap.get(stringPositionIncomeEntry.getKey());
            if (entitie == null) {
                log.warn(stringPositionIncomeEntry.getKey() + "没有找到平台对应的收益");
                continue;
            }
            String positionId = stringPositionIncomeEntry.getValue();
            //if (needUpdatePositionSet.contains(positionId)){
            //    continue;
            //}
            PositionIncome positionIncome = positionIncomeMap.get(positionId);
            if (positionIncome == null) {
                log.warn(stringPositionIncomeEntry.getKey() + "没有找到对应的收益");
                continue;
            }
            //needUpdatePositionSet.add(positionId);
            //填充 展示数，点击数，预计收益 等数据
            positionIncome.setTotalOpen(Long.valueOf(entitie.getPv()));
            positionIncome.setTotalClick(Long.valueOf(entitie.getClick()));
            positionIncome.setTotalIncome(new BigDecimal(String.valueOf(entitie.getRevenue())));

            needUpdatePositionList.add(positionIncome);
        }
        return needUpdatePositionList;
    }

    @Override
    public Boolean syncAdsData(String datetime) {
        if (StringUtils.isBlank(datetime)) {
            datetime = LocalDateTimeHelper.formatToString(LocalDateTime.now().plusDays(-1), LocalDateTimeHelper.PATTERN_STR10);
        }
        Map<String, PositionIncome> finalDspPositionMaps = Maps.newHashMapWithExpectedSize(100);
        List<com.bxm.adsmedia.facade.income.PositionIncomeVO> positionIncomeVos = getInteractIncomeData(finalDspPositionMaps, datetime, true, null);
        // 6.更新到数据库中
        for (com.bxm.adsmedia.facade.income.PositionIncomeVO vo : positionIncomeVos) {
            PositionIncome income = finalDspPositionMaps.get(vo.getPositionId());
            PositionIncomeDTO positionIncomeDTO = new PositionIncomeDTO();
            positionIncomeDTO.setTotalIncome(new BigDecimal(vo.getIncome()));
            income.setTotalIncome(positionIncomeDTO.getTotalIncome());
            income.setModifyUser("admin");
            income.setModifyTime(new Date());
            income.setStatus(PositionIncome.NO_SUBMIT);
            //记录日志
            savePositionIncomeLog(positionIncomeDTO, income, "admin");
            updateById(income);
        }
        return true;
    }

    @Override
    public Boolean syncAdsDataToPositionChannelIncome(String datetime, PositionChannelIncomeSyncDTO positionChannelIncomeSyncDTO) {
        BigDecimal divideInto = new BigDecimal("100");
        Map<String, PositionIncome> finalDspPositionMaps = Maps.newHashMapWithExpectedSize(100);
        List<com.bxm.adsmedia.facade.income.PositionIncomeVO> positionIncomeVos = getInteractIncomeData(finalDspPositionMaps, datetime, false, positionChannelIncomeSyncDTO.getPositionId());

        ArrayList<PositionChannelIncome> positionChannelIncomes = new ArrayList<>(50);
        // 6.更新到数据库中
        String channel = "互动";
        for (com.bxm.adsmedia.facade.income.PositionIncomeVO vo : positionIncomeVos) {
            //这里是互动的广告位ID，需要转换
            PositionIncome income = finalDspPositionMaps.get(vo.getPositionId());
            vo.setPositionId(income.getPositionId());

            PositionChannelIncome positionChannelIncome = new PositionChannelIncome();
            positionChannelIncome.setDataId(datetime + channel + vo.getPositionId());
            positionChannelIncome.setPositionId(vo.getPositionId());
            positionChannelIncome.setDatetime(DateUtil.convertStr2Date(vo.getDate()));
            positionChannelIncome.setSdkChannelType(channel);
            positionChannelIncome.setTotalOpen(0L);
            positionChannelIncome.setTotalClick(0L);
            positionChannelIncome.setTotalIncome(new BigDecimal(vo.getIncome()));

            calcIncome(positionChannelIncome, divideInto);

            positionChannelIncome.setCreateUser("admin");
            positionChannelIncome.setCreateTime(new Date());
            positionChannelIncomes.add(positionChannelIncome);
        }

        if (CollectionUtils.isNotEmpty(positionChannelIncomes)) {
            Set<String> dataIdSet = positionChannelIncomes.stream().map(PositionChannelIncome::getDataId).collect(Collectors.toSet());
            checkRepeated(dataIdSet);
            try {
                positionChannelIncomeService.saveBatch(positionChannelIncomes);
            } catch (Exception e){
                log.error(e.getMessage());
                throw new McCmsException("数据已经存在。");
            }
        }
        return true;
    }

    private void checkRepeated(Set<String> dataIdSet){
        //得到已经存在的数据
        QueryWrapper<PositionChannelIncome> existQueryWrapper = new QueryWrapper<>();
        existQueryWrapper.in(PositionChannelIncome.DATA_ID, dataIdSet);
        List<PositionChannelIncome> existList = positionChannelIncomeService.getBaseMapper().selectList(existQueryWrapper);
        List<String> repeated = existList.stream().map(PositionChannelIncome::getDataId).collect(Collectors.toList());

        if (CollectionUtils.isNotEmpty(repeated)){
            Set<String> messageSet = new HashSet<>();
            for (String dataId : repeated) {
                StringBuffer message = new StringBuffer();
                message.append(dataId).append("\n");
                messageSet.add(message.toString());
            }

            throw new McCmsException("有【"+repeated.size()+"】条数据已存在：\n" + messageSet.toString());
        }
    }

    private void calcIncome(PositionChannelIncome positionChannelIncome, BigDecimal divideInto){
        BigDecimal zero = new BigDecimal("0");
        BigDecimal hundred = new BigDecimal("100");
        if (divideInto == null || zero.compareTo(divideInto) > 0 || divideInto.compareTo(hundred) > 0) {
            throw new McCmsException("分成比率不能为空，且必须在0-100之间！");
        }
        positionChannelIncome.setDivideInto(divideInto);
        BigDecimal developerIncome = positionChannelIncome.getTotalIncome().multiply(divideInto)
                .divide(hundred, BigDecimal.ROUND_CEILING, BigDecimal.ROUND_HALF_UP);
        positionChannelIncome.setDeveloperIncome(developerIncome);
        positionChannelIncome.setBxmIncome(positionChannelIncome.getTotalIncome().subtract(developerIncome));
    }

    private List<com.bxm.adsmedia.facade.income.PositionIncomeVO> getInteractIncomeData(
            Map<String, PositionIncome> finalDspPositionMaps, String datetime, boolean isPositionIncome, String syncPositionId){
        //查询指定广告位
        if (StringUtils.isNotBlank(syncPositionId)) {
            return positionIncomeIntegration.getPositionIncome(Arrays.asList(syncPositionId), datetime);
        }

        // 1.找到所有配置了图文 18、视频 14 、场景13 DSP的广告位
        List<PositionDspPos> dspPos = positionDspPosService.getByDspId(Lists.newArrayList(14L, 18L));

        final List<SceneSetting> sencePositions = sceneSettingService.getBaseMapper().selectList(new QueryWrapper<SceneSetting>());
        if (CollectionUtils.isNotEmpty(sencePositions)) {
            for (SceneSetting scene : sencePositions) {
                // 这里只考虑投互动广告收益
                if (SceneDspEnum.SceneType.SCENE_TYPE_INTERACT.getType() == scene.getSceneType() && StringUtils.isNotBlank(scene.getH5Url())) {
                    String sspPositionId = subPositionId(scene.getH5Url());
                    if (StringUtils.isBlank(sspPositionId)) {
                        continue;
                    }
                    PositionDspPos pos = new PositionDspPos();
                    pos.setDspId(13L);
                    pos.setDspPosid(sspPositionId);
                    pos.setPositionId(scene.getPositionId());
                    dspPos.add(pos);
                }
            }
        }

        if (CollectionUtils.isEmpty(dspPos)) {
            log.warn("sync Ads Data is empty ！");
            return Collections.emptyList();
        }

        // 2.处理ssp 广告位与dsp平台映射
        Map<String, PositionDspPos> positionIdDspIdMap = Maps.newHashMapWithExpectedSize(dspPos.size());
        for (PositionDspPos pos : dspPos) {
            positionIdDspIdMap.put(pos.getPositionId(), pos);
        }


        // 3. 昨天ssp广告位收益表
        QueryWrapper<PositionIncome> queryWrapper = new QueryWrapper<>();

        queryWrapper.eq(PositionIncome.DATETIME, datetime);
        if (!UserRoleUtil.isLeader(UserRoleUtil.getUser()) && isPositionIncome){
            //广告位收益只查询未提交的
            queryWrapper.eq(PositionIncome.STATUS, PositionIncome.NO_SUBMIT);
        }

        List<PositionIncome> positionIncomeList = list(queryWrapper);
        if (CollectionUtils.isEmpty(positionIncomeList)) {
            log.warn("[{}] Position income is empty!", datetime);
            return Collections.emptyList();
        }

        // 4.处理互动广告位与ssp广告位收益映射
        List<String> finalPositions = Lists.newArrayListWithCapacity(positionIncomeList.size());
        for (PositionIncome income : positionIncomeList) {
            PositionDspPos pos = positionIdDspIdMap.get(income.getPositionId());
            if (pos == null) {
                continue;
            }

            if (pos.getDspId().equals(income.getDspId())) {
                finalPositions.add(pos.getDspPosid());
                finalDspPositionMaps.put(pos.getDspPosid(), income);
            }

        }
        if (MapUtils.isEmpty(finalDspPositionMaps)) {
            log.warn("[{}] Position income is empty!", datetime);
            return Collections.emptyList();
        }

        // 5.根据广告位查找昨日收益
        List<com.bxm.adsmedia.facade.income.PositionIncomeVO> positionIncomeVos = positionIncomeIntegration.getPositionIncome(finalPositions, datetime);
        return positionIncomeVos;
    }

    /**
     * 从互动url 中获取广告位
     *
     * @param url
     * @return
     */
    private static String subPositionId(String url) {
        try {
            if (StringUtils.isBlank(url)) {
                return null;
            }
            if (url.contains("appKey") && url.contains("appEntrance")) {
                String appkey = url.substring(url.indexOf("appKey") + 7);
                appkey = appkey.substring(0, appkey.indexOf("&"));

                String appEntrance = url.substring(url.indexOf("appEntrance") + 12);
                appEntrance = appEntrance.substring(0, appEntrance.indexOf("&"));

                return (appkey + "-" + appEntrance);
            }
        } catch (Exception e) {
            log.error(" 从互动url中获取广告位失败{} ", url);
        }
        return null;
    }

    public static void main(String[] args) throws URISyntaxException, MalformedURLException {
        String url = "https://i.iwanbei.cn/activities?appKey=5e40b051798c4b0d99dc521ec809da04&appEntrance=55554&business=money&i=__IMEI__&f=__IDFA__";
        System.out.println(subPositionId(url));

    }
}
