package com.bxm.mccms.common.core.service.impl;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.bxm.acl.facade.model.UserVo;
import com.bxm.mccms.common.core.entity.Receipt;
import com.bxm.mccms.common.core.mapper.ReceiptMapper;
import com.bxm.mccms.common.core.service.IReceiptService;
import com.bxm.mccms.common.helper.exception.McCmsException;
import com.bxm.mccms.common.model.income.ReceiptExcelDTO;
import com.bxm.mccms.common.model.income.ReceiptQueryDTO;
import com.bxm.mccms.facade.enums.ReceiptStatusEnum;
import com.bxm.mcssp.common.util.DateUtil;

import cn.afterturn.easypoi.excel.ExcelImportUtil;
import cn.afterturn.easypoi.excel.entity.ImportParams;

/**
 * <p>
 * 收款 服务实现类
 * </p>
 *
 * @author zhengwangeng
 * @since 2020-07-29
 */
@Service
public class ReceiptServiceImpl extends BaseServiceImpl<ReceiptMapper, Receipt> implements IReceiptService {

    @Override
    public Page<Receipt> pageBySearch(Page page, ReceiptQueryDTO dto) {
        return getBaseMapper().pageBySearch(page, dto);
    }

    @Override
    public boolean upload(UserVo userVo, MultipartFile file) {
        ImportParams importParams = new ImportParams();
        try {
            List<ReceiptExcelDTO> list = ExcelImportUtil.importExcel(file.getInputStream(), ReceiptExcelDTO.class, importParams);
            if (CollectionUtils.isEmpty(list)) {
                throw new McCmsException("导入数据为空！");
            }
            ArrayList<Receipt> receipts = new ArrayList<>(list.size());
            for (ReceiptExcelDTO receiptExcelDTO : list) {
                Receipt receipt = new Receipt();
                BeanUtils.copyProperties(receiptExcelDTO, receipt);

                receipt.setCreateUser(userVo.getUsername());
                receipt.setCreateTime(new Date());

                receipts.add(receipt);
            }
            saveBatch(receipts);
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
        return true;
    }

    @Override
    public boolean delete(UserVo userVo, Long id) {
        Receipt receipt = findByIdWithNotNull(id);
        ReceiptStatusEnum receiptStatusEnum = ReceiptStatusEnum.get(receipt.getStatus());
        if (ReceiptStatusEnum.UNSETTLED.equals(receiptStatusEnum) == false) {
            throw new McCmsException("只有未结算状态的记录可以删除！");
        }
        removeById(receipt.getId());
        return true;
    }

    @Override
    public boolean updateDetailUrl(UserVo userVo, Receipt dto) {
        if (dto.getId() == null) {
            throw new McCmsException("ID不能为空！");
        }
        if (StringUtils.isBlank(dto.getIncomeDetailUrl()) && StringUtils.isBlank(dto.getStatementDetailUrl())) {
            throw new McCmsException("明细单地址不能同时为空！");
        }
        Receipt receipt = findByIdWithNotNull(dto.getId());
        if (StringUtils.isNotBlank(dto.getIncomeDetailUrl()) && StringUtils.isNotBlank(receipt.getIncomeDetailUrl())) {
            throw new McCmsException("收益明细单已上传，无法重新上传！");
        }
        receipt.setIncomeDetailUrl(dto.getIncomeDetailUrl());

        if (StringUtils.isNotBlank(dto.getStatementDetailUrl()) && StringUtils.isNotBlank(receipt.getStatementDetailUrl())) {
            throw new McCmsException("结算明细单已上传，无法重新上传！");
        }
        receipt.setStatementDetailUrl(dto.getStatementDetailUrl());
        receipt.setModifyUser(userVo.getUsername());
        receipt.setModifyTime(new Date());
        updateById(receipt);
        return true;
    }

    @Override
    public boolean updateStatementIncome(UserVo userVo, Receipt dto) {
        if (dto.getId() == null) {
            throw new McCmsException("ID不能为空！");
        }
        if (dto.getStatementIncome() == null || dto.getStatementIncome().doubleValue() < 0) {
            throw new McCmsException("结算收益数据非法！");
        }
        Receipt receipt = findByIdWithNotNull(dto.getId());
        if (ReceiptStatusEnum.UNSETTLED.getValue() != receipt.getStatus()) {
            throw new McCmsException("未结算的记录才能操作，请确认！");
        }

        receipt.setStatementIncome(dto.getStatementIncome());
        receipt.setFlowInfo(DateUtil.dateTo8String(new Date()) + "\t" + ReceiptStatusEnum.SETTLED.getName());
        receipt.setStatus(ReceiptStatusEnum.SETTLED.getValue());
        receipt.setModifyUser(userVo.getUsername());
        receipt.setModifyTime(new Date());
        updateById(receipt);

        return true;
    }

    @Override
    public boolean updateStatus(UserVo userVo, Receipt dto) {
        if (dto.getId() == null) {
            throw new McCmsException("ID不能为空！");
        }
        ReceiptStatusEnum receiptStatusEnum = ReceiptStatusEnum.get(dto.getStatus());
        if (receiptStatusEnum == null) {
            throw new McCmsException("状态非法！");
        }
        Receipt receipt = findByIdWithNotNull(dto.getId());
        if (receipt.getStatus() + 1 != dto.getStatus()){
            throw new McCmsException("非法的状态流转，请确认当前状态！");
        }
        if (ReceiptStatusEnum.UNSETTLED.getValue() == receipt.getStatus()) {
            throw new McCmsException("未结算的记录无法流转，请确认当前状态！");
        }
        if (ReceiptStatusEnum.COLLECTED.getValue() == receipt.getStatus()) {
            throw new McCmsException("已收款记录无法流转，请确认当前状态！");
        }
        StringBuffer flowInfo = new StringBuffer(receipt.getFlowInfo());
        flowInfo.append("\n");
        flowInfo.append(DateUtil.dateTo8String(new Date()) + "\t" + receiptStatusEnum.getName());
        receipt.setFlowInfo(flowInfo.toString());
        receipt.setStatus(dto.getStatus());
        receipt.setModifyUser(userVo.getUsername());
        receipt.setModifyTime(new Date());
        updateById(receipt);

        return true;
    }
}
