package com.bxm.mccms.common.core.service.impl;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.bxm.acl.facade.model.UserVo;
import com.bxm.datapark.web.model.SspCreativeDaily;
import com.bxm.mccms.common.core.entity.SceneDspEntranceCreative;
import com.bxm.mccms.common.core.entity.SceneSetting;
import com.bxm.mccms.common.core.entity.SysDict;
import com.bxm.mccms.common.core.mapper.SceneDspEntranceCreativeMapper;
import com.bxm.mccms.common.core.mapper.SceneSettingMapper;
import com.bxm.mccms.common.core.service.ISceneDspEntranceCreativeService;
import com.bxm.mccms.common.core.service.ISceneSettingService;
import com.bxm.mccms.common.core.service.ISysDictService;
import com.bxm.mccms.common.helper.constant.CommonConstant;
import com.bxm.mccms.common.helper.enums.RoleEnum;
import com.bxm.mccms.common.integration.acl.AclUserIntegration;
import com.bxm.mccms.common.integration.datapark.DataparkWebIntegration;
import com.bxm.mccms.common.integration.ssp.position.PositionIntegration;
import com.bxm.mccms.common.model.position.CreativeDTO;
import com.bxm.mccms.common.model.position.CreativeListVO;
import com.bxm.mccms.common.model.position.SceneDspEntranceCreativeQueryDTO;
import com.bxm.mccms.common.model.position.SceneDspEntranceCreativeVO;
import com.bxm.mccms.facade.cache.CreativesKey;
import com.bxm.mccms.facade.constant.Constants;
import com.bxm.mccms.facade.enums.SceneDspEntranceCreativeEnum;
import com.bxm.mccms.facade.enums.SceneDspEnum;
import com.bxm.mccms.facade.model.pushable.SceneSettingCacheVO;
import com.bxm.mcssp.common.exception.BusinessException;
import com.bxm.mcssp.common.util.DateUtil;
import com.bxm.mcssp.facade.model.position.PositionFacadeVO;
import com.bxm.warcar.cache.impls.redis.JedisFetcher;
import com.bxm.warcar.cache.impls.redis.JedisUpdater;
import com.bxm.warcar.utils.JsonHelper;

import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.Pipeline;

/**
 * <p>
 * 广告位场景-媒体入口素材 服务实现类
 * </p>
 *
 * @author zhengwangeng
 * @since 2020-05-28
 */
@Service
public class SceneDspEntranceCreativeServiceImpl extends BaseServiceImpl<SceneDspEntranceCreativeMapper, SceneDspEntranceCreative> implements ISceneDspEntranceCreativeService {


    @Autowired
    private DataparkWebIntegration dataparkWebIntegration;
    @Autowired
    private AclUserIntegration aclUserIntegration;
    @Autowired
    private PositionIntegration positionIntegration;

    @Autowired
    private ISceneSettingService sceneSettingService;
    @Autowired
    private ISysDictService sysDictService;
    @Autowired
    private SceneSettingMapper sceneSettingMapper;

    @Autowired
    private JedisFetcher jedisFetcher;
    @Autowired
    private JedisUpdater jedisUpdater;

    @Override
    public Page<CreativeListVO> pageBySearch(UserVo user, Page page, SceneDspEntranceCreativeQueryDTO dto) {
        dto.setCreativeSource(SceneDspEntranceCreativeEnum.CreativeSource.CREATIVE.getType());
        Page<CreativeListVO> creativeListVOPage = baseMapper.pageBySearch(page, dto);
        List<CreativeListVO> records = creativeListVOPage.getRecords();
        if (CollectionUtils.isNotEmpty(records)) {
            Map<String, String> uiMap = aclUserIntegration.queryUserByRoleCode(RoleEnum.CMS_UI.getCode(), Boolean.FALSE);
            List<SysDict> sysDictList = sysDictService.listAvailable(CommonConstant.SysDict.CREATIVE_TAG, null);
            Map<String, SysDict > sysDictMap = sysDictList.stream().collect(Collectors.toMap(SysDict::getDictValue, SysDict -> SysDict));
            for (CreativeListVO record : records) {
                record.setCreateUser(uiMap.getOrDefault(record.getCreateUser(), record.getCreateUser()));

                String tags = record.getTags();
                if (StringUtils.isNotBlank(tags)) {
                    String[] tagsArray = tags.split(CommonConstant.BaseCharacter.COMMA);
                    StringBuffer tagStringBuffer = new StringBuffer();
                    for (String tag : tagsArray) {
                        SysDict sysDict = sysDictMap.getOrDefault(tag, new SysDict());
                        SysDict parentDict = sysDictMap.getOrDefault(sysDict.getParentDictValue(), new SysDict());
                        tagStringBuffer.append(parentDict.getDictName());
                        tagStringBuffer.append(CommonConstant.BaseCharacter.BAR);
                        tagStringBuffer.append(sysDict.getDictName());
                        tagStringBuffer.append(CommonConstant.BaseCharacter.COMMA);
                    }

                    record.setTags(tagStringBuffer.toString());
                }
            }
        }

        return creativeListVOPage;
    }

    @Override
    public String checkCreativeDbEqualCache() {
        StringBuffer noticeInfo = new StringBuffer();

        QueryWrapper<SceneDspEntranceCreative> creativeQueryWrapper = new QueryWrapper<>();
        creativeQueryWrapper.eq(SceneDspEntranceCreative.OPENED, Constants.Opened.OPENED.getType());
        List<SceneDspEntranceCreative> creativeList = super.list(creativeQueryWrapper);

        //素材库是否同步
        checkCreativeSync(noticeInfo, creativeList);

        //素材和标签是否同步
        checkTagAndCreativeSync(noticeInfo, creativeList);

        return noticeInfo.toString();
    }

    private void checkTagAndCreativeSync(StringBuffer noticeInfo, List<SceneDspEntranceCreative> creativeList) {
        List<SysDict> sysDictList = sysDictService.listAvailable(CommonConstant.SysDict.CREATIVE_TAG, null);
        //得到最终的标签列表
        List<SysDict> tagList =
                sysDictList.stream().filter(p -> p.getParentDictValue().equals(CommonConstant.SysDict.CREATIVE_TAG) == false).collect(Collectors.toList());

        HashMap<String, Integer> tagCreativeNum = new HashMap<>(tagList.size());
        HashMap<String, List<String>> tagCreativeMap = new HashMap<>(tagList.size());
        for (SysDict sysDict : tagList) {
            tagCreativeNum.put(sysDict.getDictValue(), 0);
            tagCreativeMap.put(sysDict.getDictValue(), new ArrayList<>());
        }
        for (SceneDspEntranceCreative sceneDspEntranceCreative : creativeList) {
            if (StringUtils.isNotBlank(sceneDspEntranceCreative.getTags())) {
                String[] tagsArray = sceneDspEntranceCreative.getTags().split(CommonConstant.BaseCharacter.COMMA);
                for (String tag : tagsArray) {
                    if (tagCreativeNum.get(tag) != null) {
                        tagCreativeNum.put(tag, tagCreativeNum.get(tag) + 1);
                        List<String> sceneDspEntranceCreatives = tagCreativeMap.get(tag);
                        sceneDspEntranceCreatives.add(String.valueOf(sceneDspEntranceCreative.getId()));
                        tagCreativeMap.put(tag, sceneDspEntranceCreatives);
                    }
                }
            }
        }

        JedisPool jedisPool = (JedisPool)jedisUpdater.getClientOriginal();
        for (SysDict sysDict : tagList) {
            String tagName = sysDict.getDictName();
            String tag = sysDict.getDictValue();

            try (Jedis jedis = jedisPool.getResource()) {
                Long tagCreativeCacheNum = jedis.llen(CreativesKey.tagRelationCreative(tag).generateKey());

                Integer tagCreativeDbNum = tagCreativeNum.get(tag);
                if(tagCreativeDbNum == null || tagCreativeDbNum.intValue() != tagCreativeCacheNum.intValue()){
                    //报警
                    noticeInfo.append(tagName).append("：素材和标签关系数据异常，即将同步数据缓存。").append("\r\n");

                    Pipeline pipeline = jedis.pipelined();
                    //这里直接删除key
                    pipeline.del(CreativesKey.tagRelationCreative(tag).generateKey());

                    //同步数据
                    List<String> creativeIdList = tagCreativeMap.get(tag);
                    if (CollectionUtils.isNotEmpty(creativeIdList)) {
                        String[] nad = creativeIdList.toArray(new String[creativeIdList.size()]);
                        pipeline.lpush(CreativesKey.tagRelationCreative(tag).generateKey(), nad);
                    }

                    pipeline.syncAndReturnAll();
                }
            }
        }
    }

    private void checkCreativeSync(StringBuffer noticeInfo, List<SceneDspEntranceCreative> creativeList) {
        JedisPool jedisPool = (JedisPool)jedisUpdater.getClientOriginal();
        Long creativeCacheSize = 0L;
        try (Jedis jedis = jedisPool.getResource()) {
            creativeCacheSize = jedis.hlen(CreativesKey.creatives().generateKey());
        }

        int creativeDbSize = creativeList.size();
        if(creativeCacheSize.intValue() != creativeDbSize){
            //报警
            noticeInfo.append("素材库数据异常,数据库数量比缓存多:" + (creativeCacheSize.intValue() - creativeDbSize)).append("\r\n");

            //同步数据
            try (Jedis jedis = jedisPool.getResource()) {
                Pipeline pipeline = jedis.pipelined();
                //这里直接删除key
                pipeline.del(CreativesKey.creatives().generateKey());
                //同步
                for (SceneDspEntranceCreative sceneDspEntranceCreative : creativeList) {
                    SceneSettingCacheVO.MediaEntranceCreative mediaEntranceCreative = new SceneSettingCacheVO.MediaEntranceCreative();
                    BeanUtils.copyProperties(sceneDspEntranceCreative, mediaEntranceCreative);
                    mediaEntranceCreative.setCreativeId(sceneDspEntranceCreative.getId());

                    pipeline.hset(CreativesKey.creatives().generateKey(), String.valueOf(mediaEntranceCreative.getCreativeId()),
                            JsonHelper.convert(mediaEntranceCreative));
                }
                pipeline.syncAndReturnAll();
            }
        }
    }

    @Override
    public Boolean update(UserVo userVo, SceneDspEntranceCreative creative) {
        SceneDspEntranceCreative oldCreative = findByIdWithNotNull(creative.getId());
        Assert.isTrue(SceneDspEntranceCreativeEnum.CreativeSource.CREATIVE.getType() == oldCreative.getCreativeSource(), "该文件并非素材库图片。");
        boolean oldOpen = Constants.Opened.OPENED.getType() == oldCreative.getOpened();
        String oldTags = oldCreative.getTags();

        boolean newOpen = Constants.Opened.OPENED.getType() == creative.getOpened();
        String newTags = creative.getTags();

        oldCreative.setCreativeName(creative.getCreativeName());
        oldCreative.setOpened(creative.getOpened());
        oldCreative.setTags(creative.getTags());
        oldCreative.setModifyTime(new Date());
        oldCreative.setModifyUser(userVo.getUsername());
        updateById(oldCreative);

        List<String> oldTagList = new ArrayList<>();
        List<String> newTagList = new ArrayList<>();
        if (StringUtils.isNotBlank(oldTags)) {
            oldTagList = new ArrayList<>(Arrays.asList(oldTags.split(CommonConstant.BaseCharacter.COMMA)));
        }
        if (StringUtils.isNotBlank(newTags)) {
            newTagList = new ArrayList<>(Arrays.asList(newTags.split(CommonConstant.BaseCharacter.COMMA)));
        }

        String creativeId = String.valueOf(creative.getId());
        if (oldOpen) {
            if (newOpen){
                if (oldTags.equals(newTags) == false) {
                    //交集
                    oldTagList.retainAll(newTagList);

                    List<String> oldTagListTemp = new ArrayList<>(Arrays.asList(oldTags.split(CommonConstant.BaseCharacter.COMMA)));
                    //剩下的就是移除的
                    oldTagListTemp.removeAll(oldTagList);
                    //剩下的就是需要要添加的
                    newTagList.removeAll(oldTagListTemp);
                    //更新素材到标签
                    JedisPool jedisPool = (JedisPool)jedisUpdater.getClientOriginal();
                    try (Jedis jedis = jedisPool.getResource()) {
                        Pipeline pipeline = jedis.pipelined();
                        for (String tag : oldTagListTemp) {
                            pipeline.lrem(CreativesKey.tagRelationCreative(tag).generateKey(), 0, creativeId);
                        }
                        for (String tag : newTagList) {
                            pipeline.lpush(CreativesKey.tagRelationCreative(tag).generateKey(), creativeId);
                        }
                        pipeline.syncAndReturnAll();
                    }
                }
            } else {
                if (CollectionUtils.isNotEmpty(oldTagList)) {
                    //移除所有的老标签里面的关系
                    JedisPool jedisPool = (JedisPool)jedisUpdater.getClientOriginal();
                    try (Jedis jedis = jedisPool.getResource()) {
                        Pipeline pipeline = jedis.pipelined();
                        for (String tag : oldTagList) {
                            pipeline.lrem(CreativesKey.tagRelationCreative(tag).generateKey(), 0, creativeId);
                        }
                        pipeline.syncAndReturnAll();
                    }
                }

                //移除所有广告位里个性筛选里面有这个素材ID的配置
                List<SceneSetting> sceneSettingList =
                        sceneSettingMapper.getContainsIdSceneSettingConfig(SceneDspEnum.ScreenType.PERSONALITY_SCREEN.getType(), oldCreative.getId());
                if (CollectionUtils.isNotEmpty(sceneSettingList)){
                    for (SceneSetting sceneSetting : sceneSettingList) {
                        String screenValue = sceneSetting.getScreenValue();
                        if (StringUtils.isBlank(screenValue)){
                            continue;
                        }
                        String[] creativeIds = screenValue.split(CommonConstant.BaseCharacter.COMMA);
                        ArrayList<String> list = new ArrayList<>(Arrays.asList(creativeIds));
                        list.remove(creativeId);
                        String lastValue = StringUtils.join(list.toArray(), CommonConstant.BaseCharacter.COMMA);
                        sceneSetting.setScreenValue(lastValue);
                    }
                    sceneSettingService.updateBatchById(sceneSettingList, 100);
                }

            }
        } else {
            if (newOpen){
                if (CollectionUtils.isNotEmpty(newTagList)) {
                    //更新素材到标签
                    JedisPool jedisPool = (JedisPool)jedisUpdater.getClientOriginal();
                    try (Jedis jedis = jedisPool.getResource()) {
                        Pipeline pipeline = jedis.pipelined();

                        for (String tag : newTagList) {
                            pipeline.lpush(CreativesKey.tagRelationCreative(tag).generateKey(), creativeId);
                        }
                        pipeline.syncAndReturnAll();
                    }
                }

            }
        }

        SceneSettingCacheVO.MediaEntranceCreative mediaEntranceCreative = new SceneSettingCacheVO.MediaEntranceCreative();
        BeanUtils.copyProperties(oldCreative, mediaEntranceCreative);
        mediaEntranceCreative.setCreativeId(oldCreative.getId());
        //素材写入缓存
        jedisUpdater.hupdate(CreativesKey.creatives(), creativeId, JsonHelper.convert(mediaEntranceCreative));

        return Boolean.TRUE;
    }

    @Override
    public Boolean add(UserVo userVo, CreativeDTO creativeDTO) {
        List<CreativeDTO.Creative> addCreatives = creativeDTO.getCreatives();
        Set<String> md5Set = addCreatives.stream().map(CreativeDTO.Creative::getMd5).collect(Collectors.toSet());
        QueryWrapper<SceneDspEntranceCreative> wrapper = new QueryWrapper<>();
        wrapper.eq(SceneDspEntranceCreative.CREATIVE_SOURCE, SceneDspEntranceCreativeEnum.CreativeSource.CREATIVE.getType());
        wrapper.in(SceneDspEntranceCreative.MD5, md5Set);
        List<SceneDspEntranceCreative> existList = super.list(wrapper);
        if (CollectionUtils.isNotEmpty(existList)) {
            StringBuffer stringBuffer = new StringBuffer("已存在相同的文件：");
            for (SceneDspEntranceCreative sceneDspEntranceCreative : existList) {
                stringBuffer.append(sceneDspEntranceCreative.getCreativeName()).append(";");
            }
            throw new BusinessException(stringBuffer.toString());
        }
        ArrayList<SceneDspEntranceCreative> sceneDspEntranceCreatives = new ArrayList<>();
        for (CreativeDTO.Creative creative : addCreatives) {
            SceneDspEntranceCreative sceneDspEntranceCreative = new SceneDspEntranceCreative();
            BeanUtils.copyProperties(creative, sceneDspEntranceCreative);
            sceneDspEntranceCreative.setId(null);
            sceneDspEntranceCreative.setTags(creativeDTO.getTags());
            sceneDspEntranceCreative.setOpened(Constants.Opened.CLOSED.getType());
            sceneDspEntranceCreative.setCreativeSource(SceneDspEntranceCreativeEnum.CreativeSource.CREATIVE.getType());
            sceneDspEntranceCreative.setCreateTime(new Date());
            sceneDspEntranceCreative.setCreateUser(userVo.getUsername());

            sceneDspEntranceCreatives.add(sceneDspEntranceCreative);
        }
        saveBatch(sceneDspEntranceCreatives, 100);
        return Boolean.TRUE;
    }

    @Override
    public List<SceneDspEntranceCreativeVO> getList(String positionId, String date) {
        SceneSetting sceneSetting = sceneSettingService.findOneByOneParam(SceneSetting.POSITION_ID, positionId);
        if (sceneSetting == null) {
            return Collections.emptyList();
        }

        SceneDspEnum.CreativeSource creativeSource = SceneDspEnum.CreativeSource.get(sceneSetting.getCreativeSource());
        if(SceneDspEnum.CreativeSource.MANUAL.equals(creativeSource)) {
            QueryWrapper<SceneDspEntranceCreative> wrapper = new QueryWrapper<>();
            wrapper.eq(SceneDspEntranceCreative.POSITION_ID, positionId);
            wrapper.eq(SceneDspEntranceCreative.CREATIVE_SOURCE, SceneDspEntranceCreativeEnum.CreativeSource.MANUAL.getType());
            List<SceneDspEntranceCreative> list = super.list(wrapper);
            if (CollectionUtils.isEmpty(list)) {
                return Collections.emptyList();
            }
            if (StringUtils.isBlank(date)){
                date = DateUtil.dateTo8String(new Date());
            }
            List<SceneDspEntranceCreativeVO> sceneDspEntranceCreativeVOList = new ArrayList<>(list.size());
            for (SceneDspEntranceCreative sceneDspEntranceCreative : list) {
                SceneDspEntranceCreativeVO sceneDspEntranceCreativeVO = new SceneDspEntranceCreativeVO();
                BeanUtils.copyProperties(sceneDspEntranceCreative, sceneDspEntranceCreativeVO);

                //从datapark-facade获取pv,uv
                SspCreativeDaily sspCreativeDailyQuery = new SspCreativeDaily();
                sspCreativeDailyQuery.setThedate(date);
                sspCreativeDailyQuery.setCreativeId(String.valueOf(sceneDspEntranceCreative.getId()));
                List<SspCreativeDaily> creativeDataList = dataparkWebIntegration.getCreativeData(sspCreativeDailyQuery);
                if (CollectionUtils.isNotEmpty(creativeDataList)) {
                    SspCreativeDaily sspCreativeDaily = creativeDataList.get(0);
                    sceneDspEntranceCreativeVO.setClickPv(Long.valueOf(sspCreativeDaily.getClickPv()));
                    sceneDspEntranceCreativeVO.setViewPv(Long.valueOf(sspCreativeDaily.getExposurePv()));
                }

                sceneDspEntranceCreativeVOList.add(sceneDspEntranceCreativeVO);
            }
            return sceneDspEntranceCreativeVOList;
        } else if (SceneDspEnum.CreativeSource.CREATIVE.equals(creativeSource)){
            SceneDspEnum.ScreenType screenType = SceneDspEnum.ScreenType.get(sceneSetting.getSceneType());
            if (SceneDspEnum.ScreenType.PERSONALITY_SCREEN.equals(screenType)){
                QueryWrapper<SceneDspEntranceCreative> wrapper = new QueryWrapper<>();
                wrapper.eq(SceneDspEntranceCreative.CREATIVE_SOURCE, SceneDspEntranceCreativeEnum.CreativeSource.CREATIVE.getType());
                wrapper.in(SceneDspEntranceCreative.ID, Arrays.asList(sceneSetting.getScreenValue().split(CommonConstant.BaseCharacter.COMMA)));
                List<SceneDspEntranceCreative> list = super.list(wrapper);
                if (CollectionUtils.isEmpty(list)) {
                    return Collections.emptyList();
                }
                List<SceneDspEntranceCreativeVO> sceneDspEntranceCreativeVOList = new ArrayList<>(list.size());
                for (SceneDspEntranceCreative sceneDspEntranceCreative : list) {
                    SceneDspEntranceCreativeVO sceneDspEntranceCreativeVO = new SceneDspEntranceCreativeVO();
                    BeanUtils.copyProperties(sceneDspEntranceCreative, sceneDspEntranceCreativeVO);

                    sceneDspEntranceCreativeVOList.add(sceneDspEntranceCreativeVO);
                }
                return sceneDspEntranceCreativeVOList;
            }
        }

        return Collections.emptyList();
    }

    @Override
    public Integer getShortlistCreativeCount(UserVo user, String positionId, List<String> tags) {
        PositionFacadeVO positionFacadeVO = positionIntegration.findByPositionId(positionId);
        String[] sizeArray = positionFacadeVO.getPositionSize().split("\\"+ CommonConstant.BaseCharacter.ASTERISK);
        if (sizeArray.length < 2) {
            throw new BusinessException("当前广告位尺寸异常："+ positionFacadeVO.getPositionSize());
        }
        Integer width = null;
        Integer height = null;
        try {
            //pagesize规范：列表 width * height maximumKb    接口实际存储 height * width maximumKb
            width = Integer.valueOf(sizeArray[0]);
            height = Integer.valueOf(sizeArray[1].split("\\s+")[0]);
        }catch (Exception e){
            throw new BusinessException("当前广告位尺寸异常："+ positionFacadeVO.getPositionSize());
        }
        Page page = new Page(1, -1);
        //设置最大单页限制数量，默认 500 条，-1 不受限制
        SceneDspEntranceCreativeQueryDTO dto = new SceneDspEntranceCreativeQueryDTO();
        dto.setHeight(height);
        dto.setWidth(width);
        dto.setTags(tags);
        dto.setCreativeSource(SceneDspEntranceCreativeEnum.CreativeSource.CREATIVE.getType());
        dto.setOpened(Constants.Opened.OPENED.getType());
        Page<CreativeListVO> creativeListVOPage = getBaseMapper().pageBySearch(page, dto);

        return creativeListVOPage.getRecords().size();
    }
}
