package com.bxm.mccms.common.core.service.impl;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.bxm.acl.facade.model.UserVo;
import com.bxm.mccms.common.core.entity.SceneActivity;
import com.bxm.mccms.common.core.entity.SceneDspEntranceCreative;
import com.bxm.mccms.common.core.entity.SceneSetting;
import com.bxm.mccms.common.core.mapper.SceneSettingMapper;
import com.bxm.mccms.common.core.service.ISceneActivityService;
import com.bxm.mccms.common.core.service.ISceneDspEntranceCreativeService;
import com.bxm.mccms.common.core.service.ISceneSettingService;
import com.bxm.mccms.common.helper.enums.RoleEnum;
import com.bxm.mccms.common.helper.exception.McCmsException;
import com.bxm.mccms.common.helper.util.DateUtil;
import com.bxm.mccms.common.helper.util.UserRoleUtil;
import com.bxm.mccms.common.integration.acl.AclUserIntegration;
import com.bxm.mccms.common.model.position.SceneSettingDTO;
import com.bxm.mccms.common.model.position.SceneSettingListVO;
import com.bxm.mccms.common.model.position.SceneSettingQueryDTO;
import com.bxm.mccms.facade.constant.Constants;
import com.bxm.mccms.facade.enums.SceneDspEntranceCreativeEnum;
import com.bxm.mccms.facade.enums.SceneDspEnum;
import com.bxm.mcssp.common.enums.position.CustomPositionTypeEnum;
import com.bxm.mcssp.common.enums.position.PositionSceneTypeEnum;

/**
 * <p>
 * 广告位场景设置 服务实现类
 * </p>
 *
 * @author zhengwangeng
 * @since 2020-05-28
 */
@Service
public class SceneSettingServiceImpl extends BaseServiceImpl<SceneSettingMapper, SceneSetting> implements ISceneSettingService {

    @Autowired
    private AclUserIntegration aclUserIntegration;
    @Autowired
    private ISceneDspEntranceCreativeService sceneDspEntranceCreativeService;
    @Autowired
    private ISceneActivityService sceneActivityService;

    @Override
    public Page<SceneSettingListVO> pageBySearch(UserVo user, Page page, SceneSettingQueryDTO dto) {
        String mjCode = dto.getMjCode();

        // 当登录的mj/bd和传过来参数mj/bd不一致时不查
        if (UserRoleUtil.isMj(user)) {
            if (org.apache.commons.lang3.StringUtils.isNotBlank(mjCode) && !mjCode.equals(user.getUsername())) {
                return new Page<>();
            }
            mjCode = user.getUsername();
        }

        dto.setMjCode(mjCode);

        List<Integer> positionScenes = new ArrayList<>();
        if (dto.getPositionScene() != null) {
            positionScenes.add(dto.getPositionScene());
        } else {
            positionScenes.addAll(PositionSceneTypeEnum.getTypeListByCustomPositionTypeEnum(CustomPositionTypeEnum.get(dto.getCustomPositionType())));
        }
        dto.setPositionScenes(positionScenes);

        Page<SceneSettingListVO> voPage = getBaseMapper().pageBySearch(page, dto);
        List<SceneSettingListVO> records = voPage.getRecords();
        if (CollectionUtils.isNotEmpty(records)) {
            // 获取 BD 字典信息
            Map<String, String> mjMap = aclUserIntegration.queryUserByRoleCode(RoleEnum.SHANG_YE_HUA_YUN_YING_ZHUAN_YUAN.getCode(), false);
            for (SceneSettingListVO vo : records) {
                vo.setMjName(mjMap.getOrDefault(vo.getMjCode(), StringUtils.EMPTY));
            }
        }
        return voPage;
    }

    @Override
    public Boolean moveSceneActivityToTable() {
        List<SceneActivity> sceneActivityList = sceneActivityService.list();
        if (CollectionUtils.isNotEmpty(sceneActivityList)) {
            throw new McCmsException("活动数据已经迁移过了，无需再迁移！");
        }
        List<SceneSetting> sceneSettingList = findListByOneParam(SceneSetting.SCENE_TYPE,
                SceneDspEnum.SceneType.SCENE_TYPE_INSPIRE_VIDEO.getType());
        if (CollectionUtils.isEmpty(sceneSettingList)) {
            return true;
        }
        ArrayList<SceneActivity> sceneActivities = new ArrayList<>(sceneSettingList.size());
        for (SceneSetting sceneSetting : sceneSettingList) {
            SceneActivity sceneActivity = new SceneActivity();
            sceneActivity.setPositionId(sceneSetting.getPositionId());
            sceneActivity.setActivityId(sceneSetting.getActivityId());
            sceneActivity.setActivityUrl(sceneSetting.getActivityUrl());
            sceneActivity.setType(1);
            sceneActivity.setStartDate(com.bxm.mcssp.common.util.DateUtil.StringToDate("2020-01-01"));
            sceneActivity.setEndDate(com.bxm.mcssp.common.util.DateUtil.StringToDate("2099-01-01"));
            sceneActivity.setOpened(1);
            sceneActivity.setCreateUser("admin");
            sceneActivity.setCreateTime(new Date());
            sceneActivities.add(sceneActivity);
        }
        if (CollectionUtils.isNotEmpty(sceneActivities)) {
            sceneActivityService.saveBatch(sceneActivities);
        }
        return true;
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Boolean saveConfig(UserVo userVo, SceneSettingDTO dto) {
        String positionId = dto.getPositionId();
        SceneSetting sceneSetting = findOneByOneParam(SceneSetting.POSITION_ID, positionId);

        if (dto.getH5Url() != null) {
            dto.setH5Url(dto.getH5Url().trim());
        }
        SceneDspEnum.CreativeSource oldCreativeSource = null;
        //保存场景
        if (sceneSetting == null) {
            sceneSetting = new SceneSetting();
            oldCreativeSource = SceneDspEnum.CreativeSource.get(dto.getCreativeSource());
            BeanUtils.copyProperties(dto, sceneSetting);
            super.save(sceneSetting);
        } else {
            oldCreativeSource = SceneDspEnum.CreativeSource.get(sceneSetting.getCreativeSource());
            BeanUtils.copyProperties(dto, sceneSetting);
            super.updateById(sceneSetting);
        }

        //保存媒体入口素材
        saveCreative(userVo, oldCreativeSource, sceneSetting, dto, positionId);

        SceneDspEnum.SceneType sceneType = SceneDspEnum.SceneType.get(sceneSetting.getSceneType());
        //互动激励视频,SDK复合活动才需要配置活动
        if (SceneDspEnum.SceneType.SCENE_TYPE_INSPIRE_VIDEO == sceneType
                || SceneDspEnum.SceneType.SCENE_TYPE_SDK_COMPOUND_ACTIVITY == sceneType) {
            //保存活动
            saveActivitie(userVo, dto, positionId);
        } else {
            QueryWrapper<SceneActivity> queryWrapper = new QueryWrapper<>();
            queryWrapper.eq(SceneActivity.POSITION_ID, positionId);
            sceneActivityService.remove(queryWrapper);
        }

        return Boolean.TRUE;
    }

    private void saveActivitie(UserVo userVo, SceneSettingDTO dto, String positionId) {
        validatedActivityArgs(dto);

        List<SceneSettingDTO.SceneActivity> sceneActivities = dto.getSceneActivities();
        for (SceneSettingDTO.SceneActivity activity : sceneActivities) {
            activity.setPositionId(positionId);
        }

        List<SceneSettingDTO.SceneActivity> updateObjs = sceneActivities.stream().filter(p ->
                p.getId() != null).collect(Collectors.toList());
        List<SceneSettingDTO.SceneActivity> addObjs = sceneActivities.stream().filter(p ->
                p.getId() == null).collect(Collectors.toList());

        //之前保存的老的信息要排除下
        List<SceneActivity> existList = sceneActivityService.findListByOneParam(SceneActivity.POSITION_ID, positionId);

        if(CollectionUtils.isNotEmpty(existList)) {
            List<Long> oldIds = existList.stream().map(SceneActivity::getId).collect(Collectors.toList());
            List<Long> needUpdateIds = updateObjs.stream().map(SceneSettingDTO.SceneActivity::getId).collect(Collectors.toList());
            //移除需要更新的ID，剩下的就是需要删除的
            oldIds.removeAll(needUpdateIds);
            if (oldIds.size() > 0) {
                sceneActivityService.removeByIds(oldIds);
            }
        }

        if (CollectionUtils.isNotEmpty(addObjs)) {
            List<SceneActivity> needSaveList = new ArrayList<>(addObjs.size());
            for (SceneSettingDTO.SceneActivity addObj : addObjs) {
                SceneActivity sceneActivity = new SceneActivity();
                BeanUtils.copyProperties(addObj, sceneActivity);
                //活动地址为完整的url
                sceneActivity.setActivityUrl(sceneActivity.getActivityUrl() + StringUtils.defaultIfBlank(sceneActivity.getDiyArgs(), StringUtils.EMPTY));
                sceneActivity.setCreateUser(userVo.getUsername());
                sceneActivity.setCreateTime(new Date());
                needSaveList.add(sceneActivity);
            }
            sceneActivityService.saveBatch(needSaveList);
        }
        if (CollectionUtils.isNotEmpty(updateObjs)) {
            List<SceneActivity> needUpdateList = new ArrayList<>(updateObjs.size());
            for (SceneSettingDTO.SceneActivity updateObj : updateObjs) {
                SceneActivity sceneActivity = new SceneActivity();
                //BeanUtils.copyProperties(updateObj, sceneActivity);
                //只能修改流量占比
                sceneActivity.setId(updateObj.getId());
                sceneActivity.setRateOfFlow(updateObj.getRateOfFlow());
                sceneActivity.setPositionConfig(updateObj.getPositionConfig());
                sceneActivity.setOpened(updateObj.getOpened());
                sceneActivity.setModifyUser(userVo.getUsername());
                sceneActivity.setModifyTime(new Date());
                needUpdateList.add(sceneActivity);
            }
            sceneActivityService.updateBatchById(needUpdateList);
        }
    }

    private void validatedActivityArgs(SceneSettingDTO dto) {
        List<SceneSettingDTO.SceneActivity> openedSceneActivityList = dto.getSceneActivities().stream().filter(p ->
                Constants.Opened.OPENED.getType() == p.getOpened()).collect(Collectors.toList());
        if (openedSceneActivityList.size() < 1) {
            throw new McCmsException("活动至少要开启一个！");
        }
        //校验测试活动时间是否重合
        List<SceneSettingDTO.SceneActivity> testActivityList = openedSceneActivityList.stream().filter(p ->
                SceneSettingDTO.TYPE_TEST == p.getType()).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(testActivityList) && testActivityList.size() > 1) {
            validatedActivityDateOverlap(testActivityList);

            //流量占比
            for (SceneSettingDTO.SceneActivity sceneActivity : testActivityList) {
                if (sceneActivity.getRateOfFlow() != null && (sceneActivity.getRateOfFlow() < 0 || sceneActivity.getRateOfFlow() > 100)) {
                    throw new McCmsException("流量占比只能在0-100之间。");
                }
            }
        }
        //校验直接投放活动时间是否重合
        List<SceneSettingDTO.SceneActivity> launchActivityList = openedSceneActivityList.stream().filter(p ->
                SceneSettingDTO.TYPE_LAUNCH == p.getType()).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(launchActivityList) && launchActivityList.size() > 1) {
            validatedActivityDateOverlap(launchActivityList);
        }

    }

    private void validatedActivityDateOverlap(List<SceneSettingDTO.SceneActivity> openedSceneActivityList) {
        //校验重复的时间区间
        ArrayList<String> dateList = new ArrayList<>();
        String split = "#";
        for (SceneSettingDTO.SceneActivity sceneActivity : openedSceneActivityList) {
            dateList.add(com.bxm.mcssp.common.util.DateUtil.dateTo8String(sceneActivity.getStartDate()) + split +
                    com.bxm.mcssp.common.util.DateUtil.dateTo8String(sceneActivity.getEndDate())
            );
        }
        List<String> errorList = DateUtil.validatedDateOverlap(dateList, split);
        if (CollectionUtils.isNotEmpty(errorList)) {
            throw new McCmsException(errorList.toString());
        }
    }

    private void saveCreative(UserVo userVo, SceneDspEnum.CreativeSource oldCreativeSource, SceneSetting sceneSetting, SceneSettingDTO dto, String positionId) {
        SceneDspEnum.CreativeSource newCreativeSource = SceneDspEnum.CreativeSource.get(dto.getCreativeSource());

        if (sceneSetting == null) {
            //新增场景配置
            if (SceneDspEnum.CreativeSource.MANUAL.equals(newCreativeSource)) {
                saveManualCreative(userVo, dto, positionId);
            }
        } else {
            //修改场景配置
            if (oldCreativeSource.equals(newCreativeSource) == false) {
                if (SceneDspEnum.CreativeSource.MANUAL.equals(newCreativeSource)) {
                    saveManualCreative(userVo, dto, positionId);
                } else {
                    //使用素材库，则删除原来手动上传的素材
                    QueryWrapper<SceneDspEntranceCreative> queryWrapper = new QueryWrapper<>();
                    queryWrapper.eq(SceneDspEntranceCreative.POSITION_ID, positionId);
                    queryWrapper.eq(SceneDspEntranceCreative.CREATIVE_SOURCE, SceneDspEntranceCreativeEnum.CreativeSource.MANUAL.getType());
                    sceneDspEntranceCreativeService.remove(queryWrapper);
                }
            } else if (oldCreativeSource.equals(newCreativeSource) && SceneDspEnum.CreativeSource.MANUAL.equals(newCreativeSource)){
                saveManualCreative(userVo, dto, positionId);
            }

        }

    }

    private void saveManualCreative(UserVo userVo, SceneSettingDTO dto, String positionId) {
        List<SceneSettingDTO.MediaEntranceCreative> mediaEntranceCreatives = dto.getMediaEntranceCreatives();
        for (SceneSettingDTO.MediaEntranceCreative mediaEntranceCreative : mediaEntranceCreatives) {
            mediaEntranceCreative.setPositionId(positionId);
        }

        List<SceneSettingDTO.MediaEntranceCreative> updateObjs = mediaEntranceCreatives.stream().filter(p ->
                p.getId() != null).collect(Collectors.toList());
        List<SceneSettingDTO.MediaEntranceCreative> addObjs = mediaEntranceCreatives.stream().filter(p ->
                p.getId() == null).collect(Collectors.toList());

        //之前保存的老的信息要排除下
        QueryWrapper<SceneDspEntranceCreative> sceneDspEntranceCreativeQueryWrapper = new QueryWrapper<>();
        sceneDspEntranceCreativeQueryWrapper.eq(SceneDspEntranceCreative.POSITION_ID, positionId);
        sceneDspEntranceCreativeQueryWrapper.eq(SceneDspEntranceCreative.CREATIVE_SOURCE, SceneDspEntranceCreativeEnum.CreativeSource.MANUAL.getType());
        List<SceneDspEntranceCreative> existList = sceneDspEntranceCreativeService.list(sceneDspEntranceCreativeQueryWrapper);

        if(CollectionUtils.isNotEmpty(existList)) {
            List<Long> oldIds = existList.stream().map(SceneDspEntranceCreative::getId).collect(Collectors.toList());
            List<Long> needUpdateIds = updateObjs.stream().map(SceneSettingDTO.MediaEntranceCreative::getId).collect(Collectors.toList());
            //移除需要更新的ID，剩下的就是需要删除的
            oldIds.removeAll(needUpdateIds);
            if (oldIds.size() > 0) {
                sceneDspEntranceCreativeService.removeByIds(oldIds);
            }
        }

        if (CollectionUtils.isNotEmpty(addObjs)) {
            List<SceneDspEntranceCreative> needSaveList = new ArrayList<>(addObjs.size());
            for (SceneSettingDTO.MediaEntranceCreative addObj : addObjs) {
                SceneDspEntranceCreative sceneDspEntranceCreative = new SceneDspEntranceCreative();
                BeanUtils.copyProperties(addObj, sceneDspEntranceCreative);
                sceneDspEntranceCreative.setCreativeSource(SceneDspEntranceCreativeEnum.CreativeSource.MANUAL.getType());
                sceneDspEntranceCreative.setCreateUser(userVo.getUsername());
                sceneDspEntranceCreative.setCreateTime(new Date());
                needSaveList.add(sceneDspEntranceCreative);
            }
            sceneDspEntranceCreativeService.saveBatch(needSaveList);
        }
        if (CollectionUtils.isNotEmpty(updateObjs)) {
            List<SceneDspEntranceCreative> needUpdateList = new ArrayList<>(updateObjs.size());
            for (SceneSettingDTO.MediaEntranceCreative addObj : updateObjs) {
                SceneDspEntranceCreative sceneDspEntranceCreative = new SceneDspEntranceCreative();
                BeanUtils.copyProperties(addObj, sceneDspEntranceCreative);
                sceneDspEntranceCreative.setCreativeSource(SceneDspEntranceCreativeEnum.CreativeSource.MANUAL.getType());
                sceneDspEntranceCreative.setModifyUser(userVo.getUsername());
                sceneDspEntranceCreative.setModifyTime(new Date());
                needUpdateList.add(sceneDspEntranceCreative);
            }
            sceneDspEntranceCreativeService.updateBatchById(needUpdateList);
        }
    }
}
