package com.bxm.mccms.common.core.service.impl;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.bxm.mccms.common.core.entity.PositionShieldStrategyRef;
import com.bxm.mccms.common.core.entity.ShieldStrategy;
import com.bxm.mccms.common.core.mapper.ShieldStrategyMapper;
import com.bxm.mccms.common.core.service.IPositionShieldStrategyRefService;
import com.bxm.mccms.common.core.service.IShieldStrategyService;
import com.bxm.mccms.common.helper.exception.McCmsException;
import com.bxm.mccms.common.integration.ssp.position.PositionIntegration;
import com.bxm.mccms.common.model.position.PositionRefShieldStrategyVO;
import com.bxm.mccms.common.model.position.ShieldStrategyRelationPositionVO;
import com.bxm.mccms.common.pushable.ShieldStrategyPushable;
import com.bxm.mcssp.common.entity.IDAndNameVO;
import com.bxm.mcssp.facade.model.position.PositionFacadeDTO;
import com.bxm.mcssp.facade.model.position.PositionFacadeQueryDTO;
import com.bxm.mcssp.facade.model.position.PositionFacadeVO;

/**
 * <p>
 * 屏蔽策略 服务实现类
 * </p>
 *
 * @author zhengwangeng
 * @since 2020-01-11
 */
@Service
public class ShieldStrategyServiceImpl extends ServiceImpl<ShieldStrategyMapper, ShieldStrategy> implements IShieldStrategyService {

    @Autowired
    private IPositionShieldStrategyRefService positionShieldStrategyRefService;
    @Autowired
    private PositionIntegration positionIntegration;

    @Autowired
    private ShieldStrategyPushable shieldStrategyPushable;

    @Override
    public List<IDAndNameVO> getIdAndNameList(List<Long> ids) {
        return baseMapper.getIdAndNameList(ids);
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public boolean save(ShieldStrategy t) {
        int rows = getBaseMapper().insert(t);
        //消息推送
        shieldStrategyPushable.push(getById(t.getId()));
        return rows > 0;
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public boolean updateById(ShieldStrategy t) {
        ShieldStrategy old = getById(t.getId());
        if(null == old){
            throw new McCmsException("数据不存在");
        }
        int rows = getBaseMapper().updateById(t);
        //消息推送
        shieldStrategyPushable.push(getById(t.getId()));
        return rows > 0;
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public boolean removeById(Serializable id) {
        boolean result = super.removeById(id);
        if (result) {
            //删除所有的引用
            QueryWrapper<PositionShieldStrategyRef> positionShieldStrategyRefWrapper = new QueryWrapper<>();
            positionShieldStrategyRefWrapper.eq(PositionShieldStrategyRef.SHIELD_STRATEGY_ID, id);
            positionShieldStrategyRefService.remove(positionShieldStrategyRefWrapper);
            //消息推送
            shieldStrategyPushable.delete(id);
        }
        return result;
    }

    @Override
    public IPage<ShieldStrategyRelationPositionVO> getRelationPosition(Page page, Long id) {
        QueryWrapper<PositionShieldStrategyRef> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq(PositionShieldStrategyRef.SHIELD_STRATEGY_ID, id);
        IPage refPage = positionShieldStrategyRefService.page(page, queryWrapper);

        List<PositionShieldStrategyRef> refPageRecords = refPage.getRecords();
        if(CollectionUtils.isEmpty(refPageRecords)){
           return new Page<>();
        }
        List<String> positionIdList = refPageRecords.stream().map(n -> n.getPositionId()).distinct().collect(Collectors.toList());
        PositionFacadeQueryDTO query = new PositionFacadeQueryDTO();
        query.setPositionIds(positionIdList);
        query.setCurrent(page.getCurrent());
        query.setSize(page.getSize());

        // 调用app服务,获取广告位列表
        IPage<PositionFacadeVO> positionPage = positionIntegration.page(query);
        List<PositionFacadeVO> positionList = positionPage.getRecords();
        if (CollectionUtils.isEmpty(positionList)) {
            return new Page<>();
        }
        Map<String, PositionFacadeVO> positionMap = positionList.stream().collect(HashMap::new, (k, v) -> k.put(v.getPositionId(), v), HashMap::putAll);
        List<ShieldStrategyRelationPositionVO> resultList = new ArrayList<>(refPageRecords.size());
        for (PositionShieldStrategyRef ref : refPageRecords) {
            ShieldStrategyRelationPositionVO vo = new ShieldStrategyRelationPositionVO();
            resultList.add(vo);
            vo.setId(ref.getId());
            PositionFacadeVO positionFacadeVO = positionMap.get(ref.getPositionId());
            if (positionFacadeVO != null){
                vo.setPositionName(positionFacadeVO.getPositionName());
                vo.setAppName(positionFacadeVO.getAppName());
            }
        }
        page.setRecords(resultList);
        return page;
    }

    @Override
    public List<PositionRefShieldStrategyVO> getRefShieldStrategy(String positionId) {
        QueryWrapper<PositionShieldStrategyRef> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq(PositionShieldStrategyRef.POSITION_ID, positionId);
        List<PositionShieldStrategyRef> refPageRecords = positionShieldStrategyRefService.list(queryWrapper);

        if(CollectionUtils.isEmpty(refPageRecords)){
            return Collections.emptyList();
        }
        List<Long> shieldStrategyIdList = refPageRecords.stream().map(n -> n.getShieldStrategyId()).distinct().collect(Collectors.toList());

        List<ShieldStrategy> shieldStrategyList = baseMapper.selectBatchIds(shieldStrategyIdList);
        if (CollectionUtils.isEmpty(shieldStrategyList)) {
            return Collections.emptyList();
        }
        Map<Long, ShieldStrategy> shieldStrategyMap = shieldStrategyList.stream().collect(HashMap::new, (k, v) -> k.put(v.getId(), v), HashMap::putAll);
        List<PositionRefShieldStrategyVO> resultList = new ArrayList<>(refPageRecords.size());
        for (PositionShieldStrategyRef ref : refPageRecords) {
            PositionRefShieldStrategyVO vo = new PositionRefShieldStrategyVO();
            resultList.add(vo);
            vo.setId(ref.getId());
            ShieldStrategy shieldStrategy = shieldStrategyMap.get(ref.getShieldStrategyId());
            if (shieldStrategy != null){
                vo.setStrategyName(shieldStrategy.getStrategyName());
            }
        }
        return resultList;
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public boolean removeRelationPositionById(Long id) {
        PositionShieldStrategyRef positionShieldStrategyRef = positionShieldStrategyRefService.getById(id);
        if (null != positionShieldStrategyRef){
            boolean result = positionShieldStrategyRefService.removeById(id);
            if (result){
                //同步修改广告位
                String positionId = positionShieldStrategyRef.getPositionId();
                PositionFacadeVO positionFacadeVO = positionIntegration.findByPositionId(positionId);
                if (null != positionFacadeVO) {
                    PositionFacadeDTO dto = new PositionFacadeDTO();
                    BeanUtils.copyProperties(positionFacadeVO, dto);

                    QueryWrapper<PositionShieldStrategyRef> queryWrapper = new QueryWrapper<>();
                    queryWrapper.eq(PositionShieldStrategyRef.POSITION_ID, positionId);
                    List<PositionShieldStrategyRef> refPageRecords = positionShieldStrategyRefService.list(queryWrapper);

                    if(CollectionUtils.isEmpty(refPageRecords)){
                        dto.setShieldStrategyIds(StringUtils.EMPTY);
                    }else {
                        StringBuffer stringBuffer = new StringBuffer();
                        for (PositionShieldStrategyRef refPageRecord : refPageRecords) {
                            stringBuffer.append(refPageRecord.getShieldStrategyId()).append(",");
                        }
                        dto.setShieldStrategyIds(stringBuffer.deleteCharAt(stringBuffer.length() - 1).toString());
                    }
                    positionIntegration.update(dto);
                }
            }
        }
        return Boolean.TRUE;
    }
}
