package com.bxm.mccms.common.core.service.impl;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.bxm.adx.facade.model.task.TaskCounter;
import com.bxm.mccms.common.core.entity.Task;
import com.bxm.mccms.common.core.entity.TaskRules;
import com.bxm.mccms.common.core.mapper.TaskMapper;
import com.bxm.mccms.common.core.service.ITaskRulesService;
import com.bxm.mccms.common.core.service.ITaskService;
import com.bxm.mccms.common.helper.constant.CommonConstant;
import com.bxm.mccms.common.helper.exception.McCmsException;
import com.bxm.mccms.common.integration.adx.TaskIntegration;
import com.bxm.mccms.common.model.task.TaskDTO;
import com.bxm.mccms.common.model.task.TaskDetailVO;
import com.bxm.mccms.common.model.task.TaskListVO;
import com.bxm.mccms.common.model.task.TaskQueryDTO;
import com.bxm.mccms.common.pushable.TaskPushable;
import com.bxm.mccms.facade.enums.RuleTypeEnum;
import com.bxm.mccms.facade.enums.TaskPauseReasonEnum;
import com.bxm.mccms.facade.enums.TaskStatusEnum;

import lombok.extern.slf4j.Slf4j;

/**
 * <p>
 * 任务 服务实现类
 * </p>
 *
 * @author zhengwangeng
 * @since 2020-03-25
 */
@Slf4j
@Service
public class TaskServiceImpl extends ServiceImpl<TaskMapper, Task> implements ITaskService {

    @Autowired
    private ITaskRulesService taskRulesService;
    @Autowired
    private TaskPushable taskPushable;
    @Autowired
    private TaskIntegration taskIntegration;

    @Override
    public Page<TaskListVO> page(Page page, TaskQueryDTO query) {
        Page<TaskListVO> voPage = getBaseMapper().page(page, query);
        List<TaskListVO> records = voPage.getRecords();
        if (CollectionUtils.isNotEmpty(records)) {
            List<Long> taskIdList = records.stream().map(TaskListVO::getId).collect(Collectors.toList());
            String taskIdStr = StringUtils.join(taskIdList.toArray(), CommonConstant.BaseCharacter.COMMA);
            List<TaskCounter> taskCounterList = taskIntegration.list(taskIdStr);

            if (CollectionUtils.isNotEmpty(taskCounterList)) {
                Map<Long, TaskCounter> positionCalcConfigMap = taskCounterList.stream().collect(Collectors.toMap(TaskCounter::getId, TaskCounter -> TaskCounter));
                for (TaskListVO task : records) {
                    TaskCounter taskCounter = positionCalcConfigMap.get(task.getId());
                    if (taskCounter != null) {
                        task.setTodayAcquires(taskCounter.getTodayAcquires());
                        task.setTodayClicks(taskCounter.getTodayClicks());
                        task.setTodayViews(taskCounter.getTodayViews());
                        task.setTodayReadyWakes(taskCounter.getTodayReadyWakes());
                        task.setTodayWakes(taskCounter.getTodayWakes());
                        task.setTodaySuccessWakes(taskCounter.getTodaySucceedWakes());
                    }
                }
            }
        }
        return voPage;
    }

    @Override
    public TaskDetailVO get(Long id) {
        Task task = super.getById(id);
        if (task == null) {
            throw new McCmsException("数据不存在！");
        }
        TaskDetailVO vo = new TaskDetailVO();
        BeanUtils.copyProperties(task, vo);

        QueryWrapper<TaskRules> taskQueryWrapper = new QueryWrapper<>();
        taskQueryWrapper.eq(TaskRules.TASK_ID, task.getId());
        List<TaskRules> taskRules = taskRulesService.list(taskQueryWrapper);
        if (CollectionUtils.isNotEmpty(taskRules)) {
            for (TaskRules taskRule : taskRules) {
                RuleTypeEnum ruleTypeEnum = RuleTypeEnum.get(taskRule.getRuleType());
                if (ruleTypeEnum == null) {
                    continue;
                }
                switch (ruleTypeEnum) {
                    case REGION:
                        vo.setRegion(taskRule.getRuleValue());
                        break;
                    case POSITION:
                        vo.setPosition(taskRule.getRuleValue());
                        break;
                    case FREQUENCYS:
                        vo.setFrequencys(taskRule.getRuleValue());
                        break;
                    case APP:
                        vo.setApps(taskRule.getRuleValue());
                        break;
                    case DEVICE_BRAND:
                        vo.setDeviceBrands(taskRule.getRuleValue());
                        break;
                    case TIME_SLOT:
                        vo.setTimeSlots(taskRule.getRuleValue());
                        break;
                    default:
                        break;
                }
            }
        }

        return vo;
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Task add(TaskDTO dto) {
        super.save(dto);
        //保存规则
        saveTaskRule(dto);
        return dto;
    }

    private void saveTaskRule(TaskDTO dto) {
        if (StringUtils.isNotBlank(dto.getRegion())) {
            taskRulesService.save(dto.getId(), RuleTypeEnum.REGION, dto.getRegion());
        }
        if (StringUtils.isNotBlank(dto.getPosition())) {
            taskRulesService.save(dto.getId(), RuleTypeEnum.POSITION, dto.getPosition());
        }
        if (StringUtils.isNotBlank(dto.getFrequencys())) {
            taskRulesService.save(dto.getId(), RuleTypeEnum.FREQUENCYS, dto.getFrequencys());
        }
        if (StringUtils.isNotBlank(dto.getApps())) {
            taskRulesService.save(dto.getId(), RuleTypeEnum.APP, dto.getApps());
        }
        if (StringUtils.isNotBlank(dto.getDeviceBrands())) {
            taskRulesService.save(dto.getId(), RuleTypeEnum.DEVICE_BRAND, dto.getDeviceBrands());
        }
        if (StringUtils.isNotBlank(dto.getTimeSlots())) {
            taskRulesService.save(dto.getId(), RuleTypeEnum.TIME_SLOT, dto.getTimeSlots());
        }
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Boolean update(TaskDTO dto) {
        Task task = super.getById(dto.getId());
        if (task == null) {
            throw new McCmsException("数据不存在！");
        }
        //SDK需求2.3.2追加小需求：耍呗任务开启的状态下可以编辑内容，并且直接生效
//        if (TaskStatusEnum.PAUSE.equals(TaskStatusEnum.get(task.getStatus())) == false) {
//            throw new McCmsException("只有暂停状态下的任务才能修改！");
//        }
        dto.setStatus(task.getStatus());
        dto.setPauseReason(task.getPauseReason());
        super.updateById(dto);

        //删除原来的规则
        QueryWrapper<TaskRules> taskQueryWrapper = new QueryWrapper<>();
        taskQueryWrapper.eq(TaskRules.TASK_ID, dto.getId());
        taskRulesService.remove(taskQueryWrapper);

        //保存新的规则
        saveTaskRule(dto);
        return true;
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Boolean updatePriority(TaskDTO dto) {
        Task task = super.getById(dto.getId());
        if (task == null) {
            throw new McCmsException("数据不存在！");
        }
        if (TaskStatusEnum.PAUSE.equals(TaskStatusEnum.get(task.getStatus())) == false) {
            throw new McCmsException("只有暂停状态下的任务才能修改！");
        }

        task.setPriority(dto.getPriority());
        super.updateById(task);
        return true;
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Boolean updateStatus(TaskDTO dto) {
        Task task = super.getById(dto.getId());
        if (task == null) {
            throw new McCmsException("数据不存在！");
        }
        if (TaskStatusEnum.STOP.equals(TaskStatusEnum.get(task.getStatus()))) {
            throw new McCmsException("停止状态下的任务无法修改！");
        }

        if (TaskStatusEnum.RUNING.equals(TaskStatusEnum.get(dto.getStatus()))) {
            QueryWrapper<Task> taskQueryWrapper = new QueryWrapper<>();
            taskQueryWrapper.ne(Task.ID, task.getId());
            taskQueryWrapper.eq(Task.STATUS, TaskStatusEnum.RUNING.getValue());
            taskQueryWrapper.eq(Task.PRIORITY, task.getPriority());
            List<Task> list = super.list(taskQueryWrapper);
            if (CollectionUtils.isNotEmpty(list)) {
                throw new McCmsException("已存在优先级为 "+ task.getPriority() +" 的任务，无法开启！");
            }

            task.setPauseReason(null);
        } else {
            //人工暂停
            task.setPauseReason(TaskPauseReasonEnum.MANUAL.getValue());
        }

        task.setStatus(dto.getStatus());
        super.updateById(task);

        return true;
    }


    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Boolean autoUpdateTaskStatus() {
        return Boolean.TRUE;

        //QueryWrapper<Task> taskQueryWrapper = new QueryWrapper<>();
        //taskQueryWrapper.ne(Task.STATUS, TaskStatusEnum.STOP.getValue());
        //List<Task> list = super.list(taskQueryWrapper);
        //if (CollectionUtils.isNotEmpty(list)) {
        //    return Boolean.TRUE;
        //}
        //
        //// 判断有效时间
        //Date date = new Date();
        //for (Task task : list) {
        //    task.setStatus(TaskStatusEnum.RUNING.getValue());
        //    task.setPauseReason(null);
        //
        //    // 判断当前时间是否小于开始有效时间
        //    if (DateUtil.compareOnlyDate(date, task.getStartDate()) < 0) {
        //        task.setStatus(TaskStatusEnum.PAUSE.getValue());
        //        task.setPauseReason(TaskStatusEnum.PAUSE_DATE_START);
        //        log.info("getId={}, getPauseReason={}", task.getId(), task.getPauseReason());
        //        return task;
        //    }
        //
        //    // 判断当前时间是否大于结束有效时间
        //    if (DateUtil.compareOnlyDate(date, task.getEndDate()) > 0) {
        //        task.setStatus(TaskStatusEnum.PAUSE.getValue());
        //        task.setPauseReason(TaskStatusEnum.PAUSE_DATE_END);
        //        log.info("getId={}, getPauseReason={}", task.getId(), task.getPauseReason());
        //        return task;
        //    }
        //}

    }
}
