package com.bxm.mccms.common.integration.ssp.position;

import java.util.Collections;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.cloud.openfeign.EnableFeignClients;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.bxm.mccms.common.helper.constant.CommonConstant;
import com.bxm.mccms.common.helper.util.PageFactory;
import com.bxm.mcssp.common.entity.IDAndNameVO;
import com.bxm.mcssp.facade.model.position.PositionAuditFacadeVO;
import com.bxm.mcssp.facade.model.position.PositionFacadeDTO;
import com.bxm.mcssp.facade.model.position.PositionFacadeQueryDTO;
import com.bxm.mcssp.facade.model.position.PositionFacadeVO;
import com.bxm.mcssp.facade.service.PositionFacadeService;
import com.google.common.base.Joiner;

import lombok.extern.slf4j.Slf4j;

/**
 * <p>
 * 广告位 远程调用服务类
 * </p>
 *
 * @author zhengwangeng
 * @since 2019-12-22
 */
@Slf4j
@Service
@EnableFeignClients("com.bxm.mcssp.facade")
public class PositionIntegration {

    @Resource
    private PositionFacadeService positionFacadeService;

    /**
     * ID和名称数据
     * @param keyword
     * @return
     */
    public List<IDAndNameVO> list(String keyword) {
        ResponseEntity<List<IDAndNameVO>> resultModel = positionFacadeService.list(keyword);
        if (HttpStatus.OK.equals(resultModel.getStatusCode())) {
            return resultModel.getBody();
        }
        log.error("开发者后台异常，查询开发者列表失败！-->{}", resultModel.getStatusCodeValue());
        return Collections.emptyList();
    }

    /**
     * 分页查询
     *
     * @param dto 查询条件
     * @return
     */
    public IPage<PositionFacadeVO> page(PositionFacadeQueryDTO dto) {
        ResponseEntity<Page<PositionFacadeVO>> resultModel = positionFacadeService.page(dto);
        if (HttpStatus.OK.equals(resultModel.getStatusCode())) {
            return resultModel.getBody();
        }
        log.error("开发者后台异常，查询广告位列表失败！-->{}", resultModel.getStatusCodeValue());
        return PageFactory.noData();
    }

    /**
     * 根据id查询
     * @param id id
     * @return
     */
    public PositionFacadeVO get(Long id) {
        ResponseEntity<PositionFacadeVO> resultModel = positionFacadeService.get(id);
        if (HttpStatus.OK.equals(resultModel.getStatusCode())) {
            return resultModel.getBody();
        }
        log.error("开发者后台异常，查询开发者信息失败！-->{}", resultModel.getStatusCodeValue());
        return null;
    }

    /**
     * 修改
     *
     * @param dto 修改参数
     * @return
     */
    public Boolean update(PositionFacadeDTO dto) {
        ResponseEntity<Boolean> resultModel = positionFacadeService.update(dto);
        if (HttpStatus.OK.equals(resultModel.getStatusCode())) {
            return resultModel.getBody();
        }
        log.error("开发者后台异常，修改开发者信息失败！-->{}", resultModel.getStatusCodeValue());
        return Boolean.FALSE;
    }

    /**
     * 从开发者后台调用查询广告位 全部
     *
     * @param dto 查询条件
     * @return
     */
    public List<PositionFacadeVO> getAllList(PositionFacadeQueryDTO dto) {
        dto.setCurrent(1);
        dto.setSize(-1);
        return page(dto).getRecords();
    }


    /**
     * 根据 positionId 查询广告位
     *
     * @param positionId 广告位ID
     * @return
     */
    public PositionFacadeVO findByPositionId(String positionId) {
        ResponseEntity<PositionFacadeVO> resultModel = positionFacadeService.findByPositionId(positionId);
        if (HttpStatus.OK.equals(resultModel.getStatusCode())) {
            return resultModel.getBody();
        }
        log.error("开发者后台异常，根据positionId查询广告位失败！-->{}", resultModel.getStatusCodeValue());
        return null;
    }

    /**
     * 根据 positionId 查询广告位
     * 这里会返回包含不同广告位类型的配置信息
     * @param positionId 广告位ID
     * @return
     */
    public PositionFacadeVO findAllInfoByPositionId(String positionId) {
        ResponseEntity<PositionFacadeVO> resultModel = positionFacadeService.findAllInfoByPositionId(positionId);
        if (HttpStatus.OK.equals(resultModel.getStatusCode())) {
            return resultModel.getBody();
        }
        log.error("开发者后台异常，根据positionId查询广告位失败！-->{}", resultModel.getStatusCodeValue());
        return null;
    }

    /**
     * 根据广告位ID集合查询广告位
     *
     * @param positionIds 广告位ID集合字符串，逗号分隔
     * @return
     */
    public List<PositionFacadeVO> findListByPositionIds(List<String> positionIds) {
        if (CollectionUtils.isEmpty(positionIds)) {
            return Collections.emptyList();
        }
        String positionIdsStr = Joiner.on(CommonConstant.BaseCharacter.COMMA).skipNulls().join(positionIds);
        ResponseEntity<List<PositionFacadeVO>> resultModel = positionFacadeService.findListByPositionIds(positionIdsStr);
        if (HttpStatus.OK.equals(resultModel.getStatusCode())) {
            return resultModel.getBody();
        }
        log.error("开发者后台异常，根据positionIds集合批量查询广告位失败！-->{}", resultModel.getStatusCodeValue());
        return Collections.emptyList();
    }


    /**
     * 根据主键集合查询广告位
     *
     * @param positionIds 广告位主键集合
     * @return
     */
    public List<PositionFacadeVO> findByIds(List<Long> positionIds) {
        if (CollectionUtils.isEmpty(positionIds)) {
            return Collections.emptyList();
        }
        String idsStr = Joiner.on(CommonConstant.BaseCharacter.COMMA).skipNulls().join(positionIds);
        ResponseEntity<List<PositionFacadeVO>> resultModel = positionFacadeService.findByIds(idsStr);
        if (HttpStatus.OK.equals(resultModel.getStatusCode())) {
            return resultModel.getBody();
        }
        log.error("开发者后台异常，根据主键集合查询广告位失败！-->{}", resultModel.getStatusCodeValue());
        return Collections.emptyList();
    }


    /**
     * 条件获取所有的广告位ID
     * @param dto
     * @return
     */
    public List<String> getAllPositionIds(PositionFacadeQueryDTO dto) {
        ResponseEntity<List<String>> resultModel = positionFacadeService.getAllPositionIds(dto);
        if (HttpStatus.OK.equals(resultModel.getStatusCode())) {
            return resultModel.getBody();
        }
        log.error("开发者后台异常，条件获取所有的广告位ID失败！-->{}", resultModel.getStatusCodeValue());
        return Collections.emptyList();
    }

    /**
     * 运营后台查询审核页面的 广告位列表
     *
     * @param developerKeyword 邮箱/手机/公司名称
     * @param appKeyword 应用ID/应用名称
     * @param positionKeyword 广告位ID/广告位名称
     * @param status   状态
     * @param mjCode   媒介code
     * @param current  页码
     * @param size 页面大小
     * @return
     */
    public IPage<PositionAuditFacadeVO> getAuditPage(String developerKeyword, String appKeyword, String positionKeyword,
                                                     Integer status, Long  developerId, String mjCode, Integer current,
                                                     Integer size) {
        ResponseEntity<Page<PositionAuditFacadeVO>> resultModel = positionFacadeService.getAuditPage(developerKeyword,
                appKeyword, positionKeyword, status, developerId, mjCode, current, size);
        if (HttpStatus.OK.equals(resultModel.getStatusCode())) {
            return resultModel.getBody();
        }
        log.error("开发者后台异常，查询审核页面广告位列表失败！-->{}", resultModel.getStatusCodeValue());
        return PageFactory.noData();
    }

    /**
     * 批量审核广告位
     *
     * @param idsStr       id字符串， 逗号分隔
     * @param auditPass    审核通过
     * @param mj         是否是媒介审核
     * @param refuseReason 拒绝原因
     * @param reviewRefuseIds 拒绝原因ID列表
     * @param modifyUser     审核人
     * @return
     */
    public List<Long> batchAudit(String idsStr, Boolean auditPass, Boolean mj, String refuseReason, String reviewRefuseIds, String modifyUser) {
        ResponseEntity<List<Long>> resultModel = positionFacadeService.batchAudit(idsStr, auditPass, mj, refuseReason, reviewRefuseIds, modifyUser);
        if (HttpStatus.OK.equals(resultModel.getStatusCode())) {
            return resultModel.getBody();
        }
        log.error("开发者后台异常，批量审核广告位失败！-->{}", resultModel.getStatusCodeValue());
        return Collections.EMPTY_LIST;
    }


    /**
     * 广告位 开启/关闭
     *
     * @param positionId 广告位ID
     * @param closedFlag  是否开关， true开，false关
     * @param modifyUser   修改者
     * @return 
     */
    public Boolean positionSwitch(String positionId, Boolean closedFlag, String modifyUser) {
        ResponseEntity<Boolean> resultModel = positionFacadeService.positionSwitch(positionId, closedFlag, modifyUser);
        if (HttpStatus.OK.equals(resultModel.getStatusCode())) {
            return resultModel.getBody();
        }
        log.error("开发者后台异常，开启/关闭广告位失败！-->{}", resultModel.getStatusCodeValue());
        return Boolean.FALSE;
    }
}
