package com.bxm.mccms.common.manager.position.impl;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Objects;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.bxm.acl.facade.model.UserVo;
import com.bxm.mccms.common.core.entity.PositionCalcConfig;
import com.bxm.mccms.common.core.entity.PositionShieldStrategyRef;
import com.bxm.mccms.common.core.service.IPositionCalcConfigService;
import com.bxm.mccms.common.core.service.IPositionDspPosService;
import com.bxm.mccms.common.core.service.IPositionShieldStrategyRefService;
import com.bxm.mccms.common.helper.constant.CommonConstant;
import com.bxm.mccms.common.helper.enums.DspStaticEnum;
import com.bxm.mccms.common.helper.enums.RoleEnum;
import com.bxm.mccms.common.helper.exception.McCmsException;
import com.bxm.mccms.common.helper.util.PageFactory;
import com.bxm.mccms.common.helper.util.UserRoleUtil;
import com.bxm.mccms.common.integration.ssp.position.PositionIntegration;
import com.bxm.mccms.common.manager.position.PositionService;
import com.bxm.mccms.common.model.position.PositionAuditDTO;
import com.bxm.mccms.common.model.position.PositionFlowDTO;
import com.bxm.mccms.common.model.position.PositionFlowVO;
import com.bxm.mcssp.common.enums.AuditResultsEnum;
import com.bxm.mcssp.common.enums.position.CustomPositionTypeEnum;
import com.bxm.mcssp.common.enums.position.PositionSceneTypeEnum;
import com.bxm.mcssp.facade.model.position.PositionAuditFacadeVO;
import com.bxm.mcssp.facade.model.position.PositionFacadeDTO;
import com.bxm.mcssp.facade.model.position.PositionFacadeQueryDTO;
import com.bxm.mcssp.facade.model.position.PositionFacadeVO;
import com.google.common.base.Joiner;
import com.google.common.collect.Lists;

/**
 * <p>
 * 广告位相关接口实现类
 * </p>
 *
 * @author zhengwangeng
 * @since 2019-12-24
 */
@Service
public class PositionServiceImpl implements PositionService {

    @Autowired
    private PositionIntegration positionIntegration;

    @Autowired
    private IPositionCalcConfigService positionCalcConfigService;
    @Autowired
    private IPositionShieldStrategyRefService positionShieldStrategyRefService;
    @Autowired
    private IPositionDspPosService positionDspPosService;


    @Override
    public IPage<PositionFacadeVO> page(UserVo user, PositionFacadeQueryDTO dto) {
        String mjCode = dto.getMjCode();
        String bdCode = dto.getBdCode();

        // 当登录的mj/bd和传过来参数mj/bd不一致时不查
        if (UserRoleUtil.isMj(user)) {
            if (StringUtils.isNotBlank(mjCode) && !mjCode.equals(user.getUsername())) {
                return new Page<>();
            }
            mjCode = user.getUsername();
        }
        if (UserRoleUtil.isBd(user)) {
            if (StringUtils.isNotBlank(bdCode) && !bdCode.equals(user.getUsername())) {
                return new Page<>();
            }
            bdCode = user.getUsername();
        }
        dto.setMjCode(mjCode);
        dto.setBdCode(bdCode);

        //查询审核通过的
        dto.setStatus(AuditResultsEnum.PASS.getStatus());

        if (dto.getPositionScene() == null) {
            dto.setPositionScenes(PositionSceneTypeEnum.getTypeListByCustomPositionTypeEnum(CustomPositionTypeEnum.get(dto.getCustomPositionType())));
        }
        return positionIntegration.page(dto);
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Boolean update(UserVo user, PositionFacadeDTO dto) {
        dto.setModifyUser(user.getUsername());
        Boolean result = positionIntegration.update(dto);
        if (result) {
            PositionFacadeVO positionFacadeVO = positionIntegration.get(dto.getId());
            //广告位合作计费模式配置
            addPositionCalcConfig(user, positionFacadeVO);

            // 广告位屏蔽策略应用关系更新
            //删除旧的
            QueryWrapper<PositionShieldStrategyRef> positionShieldStrategyRefWrapper = new QueryWrapper<>();
            positionShieldStrategyRefWrapper.eq(PositionShieldStrategyRef.POSITION_ID, positionFacadeVO.getPositionId());
            positionShieldStrategyRefService.remove(positionShieldStrategyRefWrapper);

            //保存新增的
            String shieldStrategyIds = dto.getShieldStrategyIds();
            if (StringUtils.isNotBlank(shieldStrategyIds)) {
                String[] shieldStrategyIdArray = shieldStrategyIds.split(",");
                if (shieldStrategyIdArray.length > 0) {
                    List<PositionShieldStrategyRef> addList = new ArrayList<>(shieldStrategyIdArray.length);
                    for (String shieldStrategyId : shieldStrategyIdArray) {
                        PositionShieldStrategyRef ref = new PositionShieldStrategyRef();
                        ref.setPositionId(positionFacadeVO.getPositionId());
                        ref.setShieldStrategyId(Long.valueOf(shieldStrategyId));
                        addList.add(ref);
                    }
                    positionShieldStrategyRefService.saveBatch(addList, 20);
                }
            }
        }
        return result;
    }

    private void addPositionCalcConfig(UserVo user, PositionFacadeVO positionFacadeVO) {
        QueryWrapper<PositionCalcConfig> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq(PositionCalcConfig.POSITION_ID, positionFacadeVO.getPositionId());
        PositionCalcConfig positionCalcConfig = positionCalcConfigService.getOne(queryWrapper);
        // 广告位合作计费模式配置 不存在则新增
        if (null == positionCalcConfig) {
            positionCalcConfig = new PositionCalcConfig();
            Date modifyTime = new Date();
            positionCalcConfig.setPositionId(positionFacadeVO.getPositionId());

            positionCalcConfig.setCooperationType(positionFacadeVO.getCooperationType());
            positionCalcConfig.setDivideInto(positionFacadeVO.getDivideInto());
            positionCalcConfig.setBiddingType(positionFacadeVO.getBiddingType());
            positionCalcConfig.setBasePrice(positionFacadeVO.getBasePrice());
            positionCalcConfig.setStartDate(positionFacadeVO.getStartDate());
            positionCalcConfig.setEndDate(positionFacadeVO.getEndDate());

            positionCalcConfig.setOldCooperationType(positionCalcConfig.getCooperationType());
            positionCalcConfig.setOldDivideInto(positionCalcConfig.getDivideInto());
            positionCalcConfig.setOldBiddingType(positionCalcConfig.getBiddingType());
            positionCalcConfig.setOldBasePrice(positionCalcConfig.getBasePrice());
            positionCalcConfig.setOldStartDate(positionCalcConfig.getStartDate());
            positionCalcConfig.setOldEndDate(positionCalcConfig.getEndDate());

            positionCalcConfig.setCreateUser(user.getUsername());
            positionCalcConfig.setCreateTime(modifyTime);
            positionCalcConfigService.save(positionCalcConfig);
        }
    }

    @Override
    public IPage<PositionAuditFacadeVO> getAuditPage(UserVo user, String developerKeyword, String appKeyword,
                                                     String positionKeyword, Integer status, Long developerId,
                                                     String mjCode, Integer current, Integer size) {
        if (user.getRoleCodes() == null) {
            user.setRoleCodes(Collections.EMPTY_LIST);
        }
        //媒介经理和风控可以看到所有的
        Boolean lookAll = user.getRoleCodes().contains(RoleEnum.SHANG_YE_HUA_YUN_YING_JING_LI.getCode()) || user.getRoleCodes().contains(RoleEnum.RISK_CONTROL.getCode());
        if (lookAll == false) {
            mjCode = user.getUsername();
        }
        return positionIntegration.getAuditPage(developerKeyword, appKeyword, positionKeyword, status, developerId, mjCode, current, size);
    }

    @Override
    public Boolean batchAudit(UserVo user, PositionAuditDTO dto) {
        if (user.getRoleCodes() == null) {
            user.setRoleCodes(Collections.emptyList());
        }
        Boolean mj = user.getRoleCodes().contains(RoleEnum.SHANG_YE_HUA_YUN_YING_ZHUAN_YUAN.getCode()) || user.getRoleCodes().contains(RoleEnum.SHANG_YE_HUA_YUN_YING_JING_LI.getCode());
        Boolean fk = user.getRoleCodes().contains(RoleEnum.RISK_CONTROL.getCode());
        mj = mj && fk ? false : mj;
        if (!mj && !fk) {
            throw new McCmsException("当前用户角色并无审核操作权限！");
        }
        String idsStr = Joiner.on(CommonConstant.BaseCharacter.COMMA).skipNulls().join(dto.getIds());

        //媒介/媒介经理是初审，否则就是终审（风控）
        boolean auditPass = AuditResultsEnum.PASS.getStatus().equals(dto.getState());
        List<Long> successIds = positionIntegration.batchAudit(idsStr, auditPass, mj,
                dto.getRefuseReason(), dto.getReviewRefuseIds(), user.getUsername());
        if (successIds == null || successIds.size() == 0) {
            throw new McCmsException("审核失败！请确认选择的广告位所在的媒体和开发者财务信息是否通过审核。");
        }
        if (fk && auditPass) {
            //是风控审核通过（终审通过）
            for (Long id : successIds) {
                PositionFacadeVO positionFacadeVO = positionIntegration.get(id);
                //广告位合作计费模式配置
                addPositionCalcConfig(user, positionFacadeVO);

                //锁屏的广告位，需要增加锁屏DSP
                PositionSceneTypeEnum positionSceneTypeEnum =
                        PositionSceneTypeEnum.get(positionFacadeVO.getPositionScene());
                if (PositionSceneTypeEnum.LOCK_SCREEN == positionSceneTypeEnum) {
                    positionDspPosService.addDsp(positionFacadeVO.getPositionId(), DspStaticEnum.LOCK_SCREET_DSP,
                            positionFacadeVO.getPositionId());
                }
            }
        }

        return Boolean.TRUE;
    }




    @Override
    public IPage<PositionFlowVO> pageForFlow(Page page, PositionFlowDTO dto) {
        PositionFacadeQueryDTO query = new PositionFacadeQueryDTO();
        query.setAppId(dto.getAppId());
        query.setPositionKeyword(dto.getPositionName());
        //审核通过的广告位
        query.setStatus(AuditResultsEnum.PASS.getStatus());
        query.setCurrent(page.getCurrent());
        query.setSize(page.getSize());
        query.setOrders(page.orders());
        
        // 调用app服务,获取广告位列表
        IPage<PositionFacadeVO> voPage = positionIntegration.page(query);
        List<PositionFacadeVO> vos = voPage.getRecords();
        if (CollectionUtils.isEmpty(vos)) {
            return PageFactory.noData();
        }
        List<PositionFlowVO> records = Lists.newArrayListWithCapacity(vos.size());
        for (PositionFacadeVO vo : vos) {
            PositionFlowVO record = new PositionFlowVO();
            record.setId(vo.getId());
            record.setPositionId(vo.getPositionId());
            record.setPositionName(vo.getPositionName());
            record.setAppName(vo.getAppName());
            record.setMonetizedType(vo.getMonetizedType());
            record.setPositionType(vo.getPositionType());
            record.setPositionScene(vo.getPositionScene());
            record.setPositionSize(vo.getPositionSize());
            record.setDeveloperType(vo.getDeveloperType());
            record.setMjName(vo.getMjName());
            record.setClosedFlag(vo.getClosedFlag());
            records.add(record);
        }

        IPage<PositionFlowVO> resp = new Page<>(voPage.getCurrent(), voPage.getSize(),voPage.getTotal());
        resp.setRecords(records);
        return resp;
    }

    @Override
    public boolean updateFlow(String positionId, Integer closedFlag, String username) {
        positionIntegration.positionSwitch(positionId, Objects.equals(closedFlag, 1), username);
        return Boolean.TRUE;
    }
    
    @Override
    public BigDecimal getBasePrice(String positionId) {
        PositionFacadeVO vo = positionIntegration.findByPositionId(positionId);
        if(null == vo){
            return null;
        }
        return vo.getBasePrice();
    }

    @Override
    public List<PositionFacadeVO> listForSceneDsp(String positionId, Integer positionScene, Integer informationFlowTemplate) {
        PositionFacadeVO positionFacadeVO = positionIntegration.findByPositionId(positionId);
        if (positionFacadeVO == null) {
            return Collections.emptyList();
        }
        PositionFacadeQueryDTO positionFacadeQueryDTO = new PositionFacadeQueryDTO();
        positionFacadeQueryDTO.setAppId(positionFacadeVO.getAppId());
        positionFacadeQueryDTO.setStatus(AuditResultsEnum.PASS.getStatus());
        positionFacadeQueryDTO.setPositionScene(positionScene);
        positionFacadeQueryDTO.setInformationFlowTemplate(informationFlowTemplate);
        positionFacadeQueryDTO.setNotInIds(Arrays.asList(positionFacadeVO.getId().intValue()));
        return positionIntegration.getAllList(positionFacadeQueryDTO);
    }
}
