package com.bxm.messager.common.helper.exception;

import com.mysql.jdbc.exceptions.jdbc4.MySQLIntegrityConstraintViolationException;
import feign.FeignException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.TypeMismatchException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.env.Environment;
import org.springframework.dao.DataAccessException;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.validation.BindException;
import org.springframework.web.HttpMediaTypeNotAcceptableException;
import org.springframework.web.HttpMediaTypeNotSupportedException;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.bind.ServletRequestBindingException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.multipart.support.MissingServletRequestPartException;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.ConstraintViolationException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 异常处理<br/>
 *
 * @author kerry.jiang
 * @date 2019/12/16 15:41
 */
@Slf4j
@ControllerAdvice
public class GlobalExceptionHandler {

    private final static Pattern PATTERN_FEIGN = Pattern.compile("^.*; content:[\\r|\\n]*(.*)$");

    @Autowired
    private Environment environment;

    /**
     * http状态码：400.</br>
     */
    @ExceptionHandler(value = {
            BindException.class,
            HttpMessageNotReadableException.class,
            MissingServletRequestPartException.class,
            TypeMismatchException.class,
            ConstraintViolationException.class,

            IllegalStateException.class,
            IllegalArgumentException.class,
            UnsupportedOperationException.class,
            HttpMediaTypeNotAcceptableException.class,
            HttpMediaTypeNotSupportedException.class
    })
    @ResponseBody
    public Object handleBadRequestException(HttpServletRequest request,
                                            HttpServletResponse response, Exception e) {
        log.error("异常信息：" + e.getMessage(), e);
        return ResponseEntity.badRequest().body(e.getMessage());
    }

    /**
     * spring相关异常
     *
     * @param e 异常
     * @return
     */
    @ExceptionHandler(HttpRequestMethodNotSupportedException.class)
    @ResponseBody
    public ResponseEntity<String> httpRequestMethodNotSupported(HttpRequestMethodNotSupportedException e) {
        return ResponseEntity.badRequest().body("请求方法不支持，请确认！");
    }

    @ExceptionHandler(MissingServletRequestParameterException.class)
    @ResponseBody
    public ResponseEntity<String> handleRequestParameterException(MissingServletRequestParameterException e) {
        return ResponseEntity.badRequest().body(e.getParameterName() + "：请检查参数是否正确！");
    }

    @ExceptionHandler(MethodArgumentNotValidException.class)
    @ResponseBody
    public ResponseEntity<String> handleMethodArgumentNotValidException(MethodArgumentNotValidException e) {
        return ResponseEntity.badRequest().body(e.getBindingResult().getFieldErrors().get(0).getDefaultMessage());
    }

    @ExceptionHandler(ServletRequestBindingException.class)
    @ResponseBody
    public ResponseEntity<String> handleRequestBindingException(ServletRequestBindingException e) {
        return ResponseEntity.badRequest().body("请检查参数是否正确！");
    }

    /**
     * 服务器内部错误
     * http状态码：500.</br>
     */
//    @ExceptionHandler(value = {Exception.class})
//    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
//    @ResponseBody
//    public Object handleOtherException(HttpServletRequest request, HttpServletResponse response, Exception e) {
//        log.error("异常信息：" + e.getMessage(), e);
//        List<String> activeProfiles = Arrays.asList(environment.getActiveProfiles());
//        if (activeProfiles.contains(CommonConstant.Environment.PROFILES_DEV)
//                || activeProfiles.contains(CommonConstant.Environment.PROFILES_TEST)) {
//            return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR)
//                    .body(StringUtils.defaultString(e.getMessage(), "后台错误，请查看后台日志！"));
//        }
//        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body("响应失败");
//    }

    /**
     * 主键重复
     * http状态码：400.</br>
     */
    @ExceptionHandler(value = {MySQLIntegrityConstraintViolationException.class, DuplicateKeyException.class})
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    @ResponseBody
    public Object handleDuplicateKeyException(HttpServletRequest request, HttpServletResponse response, Exception e) {
        return ResponseEntity.status(HttpStatus.BAD_REQUEST).body("保存失败，存在重复的唯一值");
    }

    @ExceptionHandler(DataAccessException.class)
    @ResponseBody
    public ResponseEntity<String> dataAccess(DataAccessException e) {
        return ResponseEntity.badRequest().body("数据操作异常");
    }

    /**
     * 自定义异常.</br>
     * http状态码：400.</br>
     */
    @ExceptionHandler(value = {MessagerException.class})
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    @ResponseBody
    public Object handleMcException(HttpServletRequest request, HttpServletResponse response, Exception e) {
        return ResponseEntity.badRequest().body(e.getMessage());
    }

    /**
     * facade异常.</br>
     * http状态码：400.</br>
     */
    @ExceptionHandler(value = {FeignException.class})
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    @ResponseBody
    public Object handleFeignException(HttpServletRequest request, HttpServletResponse response, Exception e) {
        String message = null;
        Matcher matcher = PATTERN_FEIGN.matcher(e.getMessage());
        if (matcher.find()) {
            message = matcher.group(1);
        } else {
            message = e.getMessage();
        }
        return ResponseEntity.badRequest().body(message);
    }
}