package com.bxm.fossicker.activity.lottery.service.impl;

import com.bxm.fossicker.activity.domain.lottery.LotteryParticipantMapper;
import com.bxm.fossicker.activity.domain.lottery.LotteryPhaseMapper;
import com.bxm.fossicker.activity.facade.model.LotteryDTO;
import com.bxm.fossicker.activity.lottery.config.LotteryJoinStatus;
import com.bxm.fossicker.activity.lottery.config.LotteryPhaseStatus;
import com.bxm.fossicker.activity.lottery.service.LotteryDrawService;
import com.bxm.fossicker.activity.lottery.service.LotteryParticipantService;
import com.bxm.fossicker.activity.lottery.service.LotteryPhaseService;
import com.bxm.fossicker.activity.lottery.timer.DrawChanceResetTimer;
import com.bxm.fossicker.activity.model.dto.lottery.*;
import com.bxm.fossicker.activity.model.param.lottery.LotteryDrawParam;
import com.bxm.fossicker.activity.model.param.lottery.LotteryPhaseQueryParam;
import com.bxm.fossicker.activity.model.param.lottery.LotteryQueryParam;
import com.bxm.fossicker.activity.model.vo.lottery.JoinHistoryBean;
import com.bxm.fossicker.activity.model.vo.lottery.LotteryParticipantBean;
import com.bxm.fossicker.activity.model.vo.lottery.LotteryPhaseBean;
import com.bxm.fossicker.activity.model.vo.lottery.LotteryVirtaulUserBean;
import com.bxm.fossicker.activity.service.config.ActivityProperties;
import com.bxm.fossicker.base.facade.PointReportFacadeService;
import com.bxm.fossicker.base.facade.param.PointParam;
import com.bxm.fossicker.user.facade.UserInfoFacadeService;
import com.bxm.fossicker.user.facade.VirtualUserFacadeService;
import com.bxm.fossicker.user.facade.dto.UserInfoDto;
import com.bxm.fossicker.user.model.entity.VirtualUserBean;
import com.bxm.fossicker.vo.PageWarper;
import com.alibaba.fastjson.JSON;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.schedule.ScheduleService;
import com.bxm.newidea.component.schedule.builder.OnceTaskBuilder;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.DateUtils;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.RandomUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Service;

import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static com.bxm.fossicker.activity.constants.ActivityRedisKeyConstant.*;

@SuppressWarnings("AlibabaUndefineMagicConstant")
@Service
@Slf4j
public class LotteryParticipantServiceImpl extends BaseService implements LotteryParticipantService,
        ApplicationContextAware {

    /**
     * 用户参与历史的缓存记录
     */
    private final static Long HISTORY_EXPIRED_SECONDS = 2 * 3600 * 24L;

    /**
     * 单个活动用户最多参与次数
     */
    private final static int MAX_JOIN_TIMES = 3;

    private final ActivityProperties activityProperties;

    private final DistributedLock distributedLock;

    private final LotteryPhaseService lotteryPhaseService;

    private final LotteryParticipantMapper lotteryParticipantMapper;

    private final RedisStringAdapter redisStringAdapter;

    private final RedisHashMapAdapter redisHashMapAdapter;

    private final UserInfoFacadeService userInfoFacadeService;

    private final LotteryPhaseMapper lotteryPhaseMapper;

    private final VirtualUserFacadeService virtualUserFacadeService;

    private LotteryDrawService lotteryDrawService;

    @Autowired(required = false)
    private ScheduleService scheduleService;

    private final DrawChanceResetTimer drawChanceResetTimer;

    private ApplicationContext applicationContext;

    private final PointReportFacadeService pointReportFacadeService;

    @Autowired
    public LotteryParticipantServiceImpl(ActivityProperties activityProperties,
                                         DistributedLock distributedLock,
                                         LotteryPhaseMapper lotteryPhaseMapper,
                                         LotteryPhaseService lotteryPhaseService,
                                         LotteryParticipantMapper lotteryParticipantMapper,
                                         RedisStringAdapter redisStringAdapter,
                                         RedisHashMapAdapter redisHashMapAdapter,
                                         UserInfoFacadeService userInfoFacadeService,
                                         VirtualUserFacadeService virtualUserFacadeService,
                                         DrawChanceResetTimer drawChanceResetTimer,
                                         PointReportFacadeService pointReportFacadeService) {
        this.activityProperties = activityProperties;
        this.distributedLock = distributedLock;
        this.lotteryPhaseService = lotteryPhaseService;
        this.lotteryParticipantMapper = lotteryParticipantMapper;
        this.redisStringAdapter = redisStringAdapter;
        this.redisHashMapAdapter = redisHashMapAdapter;
        this.userInfoFacadeService = userInfoFacadeService;
        this.lotteryPhaseMapper = lotteryPhaseMapper;
        this.virtualUserFacadeService = virtualUserFacadeService;
        this.drawChanceResetTimer = drawChanceResetTimer;
        this.pointReportFacadeService = pointReportFacadeService;
    }

    private LotteryDrawService getLotteryDrawService() {
        if (this.lotteryDrawService == null) {
            this.lotteryDrawService = applicationContext.getBean(LotteryDrawService.class);
        }
        return this.lotteryDrawService;
    }

    @Override
    public LotteryPhaseJoinDTO addParticipant(LotteryDrawParam param) {

        LotteryPhaseDetailDTO phase = lotteryPhaseService.loadCache(param.getPhaseId());

        //非虚拟用户验证是否具备抽奖资格
        LotteryPhaseJoinDTO res = validate(param, phase);
        if (null != res) {
            log.debug("参与失败：{},请求参数：{}", JSON.toJSONString(res), param);
            return res;
        }

        //参加活动
        String resource = String.valueOf(param.getPhaseId());
        String requestId = nextSequence().toString();

        if (distributedLock.lock(resource, requestId)) {
            //增加参与次数
            String lotteryCode = addHistory(param, phase, resource, requestId);

            return LotteryPhaseJoinDTO.builder()
                    .status(LotteryJoinStatus.SUCCESS.getCode())
                    .tooltipMsg("获得一张夺宝劵")
                    .lotteryCode(lotteryCode)
                    .build();
        } else {
            log.warn("重复点击参加活动：{}", param);
            return LotteryPhaseJoinDTO.builder()
                    .status(LotteryJoinStatus.SUCCESS.getCode())
                    .lotteryCode("")
                    .build();
        }

    }

    @SuppressWarnings("CodeBlock2Expr")
    @Override
    public void addVirtualUser(Long phaseId) {
        KeyGenerator key = LOTTERY_TEMP_VIRTUAL_KEY.copy().appendKey(phaseId).appendKey(getHalfHourKey());

        List<LotteryVirtaulUserBean> virtualUsers;

        if (redisStringAdapter.hasKey(key)) {
            TypeReference<List<LotteryVirtaulUserBean>> typeReference = new TypeReference<List<LotteryVirtaulUserBean>>() {
            };
            virtualUsers = redisStringAdapter.get(key, typeReference);
        } else {
            //创建半个小时以后的虚拟用户
            //剩余分钟数，进行取数权重的降低
            int currentMin = 30 - (DateUtils.getField(new Date(), Calendar.MINUTE) % 30);

            int randomNum = RandomUtils.nextInt(activityProperties.getVirtualMinNum() * currentMin / 30,
                    activityProperties.getVirtualMaxNum() * currentMin / 30);

            if (randomNum < 1) {
                randomNum = 1;
            }

            List<VirtualUserBean> sourceVirtualUsers = virtualUserFacadeService.listVirtualUser(randomNum);

            virtualUsers = sourceVirtualUsers.stream().map(user -> {
                return LotteryVirtaulUserBean.builder()
                        .addTime(getRandomDate())
                        .headImg(user.getHeadImg())
                        .sex(user.getSex())
                        .nickname(user.getNickname())
                        .id(user.getId())
                        .build();
            }).collect(Collectors.toList());

            redisStringAdapter.set(key, virtualUsers, 60 * 30);
        }

        //根据时间排序
        virtualUsers.sort((u1, u2) -> {
            return (int) (u1.getAddTime().getTime() - u2.getAddTime().getTime());
        });

        List<LotteryVirtaulUserBean> afterJoin = Lists.newArrayList();
        afterJoin.addAll(virtualUsers);

        //将小于当前时间的虚拟数据添加到参与用户中
        virtualUsers.forEach(user -> {
            if (DateUtils.before(user.getAddTime())) {
                addParticipant(LotteryDrawParam.builder()
                        .virtaulUser(user)
                        .virtual(true)
                        .phaseId(phaseId)
                        .userId(user.getId())
                        .lastVirtual(false)
                        .build());

                afterJoin.remove(user);
            }
        });

        redisStringAdapter.set(key, afterJoin, 60 * 30);
    }

    /**
     * 获取半个小时内的随机时间
     */
    private Date getRandomDate() {
        Date now = new Date();

        Calendar calendar = Calendar.getInstance();
        calendar.setTime(now);

        Calendar afterHalfHour = Calendar.getInstance();
        afterHalfHour.setTime(now);
        afterHalfHour.set(Calendar.SECOND, 0);
        afterHalfHour.set(Calendar.MILLISECOND, 0);

        int minute = afterHalfHour.get(Calendar.MINUTE);

        if (minute >= 30) {
            afterHalfHour.set(Calendar.MINUTE, 0);
            afterHalfHour.add(Calendar.HOUR, 1);
        } else {
            afterHalfHour.set(Calendar.MINUTE, 30);
        }

        int diffMillSeconds = (int) (afterHalfHour.getTimeInMillis() - calendar.getTimeInMillis());

        calendar.add(Calendar.MILLISECOND, RandomUtils.nextInt(0, diffMillSeconds));

        return calendar.getTime();
    }

    private String getHalfHourKey() {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(new Date());

        int minute = calendar.get(Calendar.MINUTE);

        String key = calendar.get(Calendar.DAY_OF_MONTH) + "" + calendar.get(Calendar.HOUR_OF_DAY);
        if (minute >= 30) {
            key += "30";
        } else {
            key += "00";
        }

        return key;
    }

    private LotteryPhaseJoinDTO validate(LotteryDrawParam param, LotteryPhaseDetailDTO phase) {
        //判断用户参与次数是否足够，一个活动仅可参与三次
        if (!param.getVirtual()) {
            int joinTimes = getTodayJoinTimes(param.getUserId(), param.getPhaseId());
            if (joinTimes >= MAX_JOIN_TIMES) {
                return LotteryPhaseJoinDTO.builder()
                        .status(LotteryJoinStatus.MAX_TIMES.getCode())
                        .tooltipMsg("每期活动每天只能参加三次")
                        .build();
            }

            //判断用户剩余次数是否足够
            long chances = getChances(param.getUserId());
            if (chances <= 0) {
                return LotteryPhaseJoinDTO.builder()
                        .tooltipMsg("本阶段的抽奖次数已用完")
                        .status(LotteryJoinStatus.LACK.getCode())
                        .nextChanceSeconds(getNextChanesSeconds(param.getUserId()))
                        .build();
            }
        }

        long joinNum = getJoinNum(phase.getPhaseId());

        //活动状态不为进行中或者参与人数已达到临界值（比开奖人数少一条），则不允许新增加
        boolean joinCondition = !param.getLastVirtual() &&
                (phase.getStatus() != LotteryPhaseStatus.GOING.getCode()
                        || phase.getConditionNum() - joinNum <= 1);
        if (joinCondition) {
            LotteryPhaseJoinDTO res = LotteryPhaseJoinDTO.builder()
                    .status(LotteryJoinStatus.FINISH.getCode())
                    .build();

            if (phase.getLastPhaseId() != null) {
                res.setTooltipMsg("来晚咯，活动已结束，参加下一期吧");
                res.setLastPhaseId(phase.getLastPhaseId());
            } else {
                res.setTooltipMsg("来晚咯，活动已结束，参加其他抽奖吧");
            }
            return res;
        }

        return null;
    }

    /**
     * 添加用户参与历史记录
     */
    private String addHistory(LotteryDrawParam param,
                              LotteryPhaseDetailDTO phase,
                              String resource,
                              String requestId) {
        //扣除抽奖次数
        if (!param.getVirtual()) {
            decrementChances(param.getUserId());
        }
        //增加参与人数
        long joinNum = execJoinCache(param);

        //根据当前参与人数构建奖券号码
        String joinCode = createLotteryCode(joinNum);

        log.debug("参与参数：{},奖券号码:{}，参与人数：{}",
                param,
                joinCode,
                joinNum);

        //更新活动当前人数信息
        LotteryPhaseBean modifyPhase = LotteryPhaseBean.builder()
                .id(param.getPhaseId())
                .participantNum((int) joinNum)
                .build();
        lotteryPhaseMapper.updateByPrimaryKeySelective(modifyPhase);

        //保存用户参与记录
        LotteryParticipantBean entity;
        if (param.getVirtual()) {
            entity = LotteryParticipantBean.builder()
                    .id(nextId())
                    .code(joinCode)
                    .createTime(param.getVirtaulUser().getAddTime())
                    .phaseId(param.getPhaseId())
                    .userId(param.getUserId())
                    .userName(param.getVirtaulUser().getNickname())
                    .userHead(param.getVirtaulUser().getHeadImg())
                    .userType(param.getVirtual())
                    .build();

        } else {
            UserInfoDto userInfo = userInfoFacadeService.getUserById(param.getUserId());
            entity = LotteryParticipantBean.builder()
                    .id(nextId())
                    .code(joinCode)
                    .createTime(new Date())
                    .phaseId(param.getPhaseId())
                    .userId(param.getUserId())
                    .userName(userInfo.getNickName())
                    .userHead(userInfo.getHeadImg())
                    .userType(param.getVirtual())
                    .build();
        }
        lotteryParticipantMapper.insert(entity);

        //参与人数更新，缓存更新
        lotteryPhaseService.cleanCache(phase.getPhaseId());

        distributedLock.unlock(resource, requestId);

        //活动人数达到设置的开奖人数条件(比人员少1)，触发开奖
        log.debug("活动参与情况，参与人数：{}，预计开奖人数：{},请求参数：{}",
                joinNum,
                phase.getConditionNum(),
                param);
        if (!param.getLastVirtual() && joinNum == (phase.getConditionNum() - 1)) {
            getLotteryDrawService().doDraw(param.getPhaseId(), phase.getTitle());
        }

        //参与成功后上报数据埋点
        report(param);

        return joinCode;
    }

    /**
     * 用户参与活动成功后上报数据埋点
     *
     * @param param 相关参数
     */
    private void report(LotteryDrawParam param) {
        if (param.getVirtual()) {
            return;
        }
        pointReportFacadeService.add(PointParam.build(param)
                .e("3034")
                .ev("102." + param.getPhaseId())
                .put("uid", String.valueOf(param.getUserId()))
        );
    }

    private long getJoinNum(Long phaseId) {
        KeyGenerator key = LOTTERY_PHASE_NUM_KEY.copy().appendKey(phaseId);
        return redisStringAdapter.getLong(key);
    }

    private long execJoinCache(LotteryDrawParam param) {
        String phaseIdStr = String.valueOf(param.getPhaseId());
        //抽奖人数增加
        KeyGenerator key = LOTTERY_PHASE_NUM_KEY.copy().appendKey(param.getPhaseId());
        long joinNum = redisStringAdapter.increment(key);

        //用户历史参与抽奖记录增加，虚拟用户不处理
        if (!param.getVirtual()) {
            KeyGenerator userJoinKey = LOTTERY_JOIN_HISTORY_KEY.copy().appendKey(param.getUserId());
            Integer times = redisHashMapAdapter.get(userJoinKey, phaseIdStr, Integer.class);
            if (times == null) {
                times = 0;
            }
            times += 1;
            redisHashMapAdapter.put(userJoinKey, phaseIdStr, times);

            //用户今日参与次数增加
            KeyGenerator todayJoinKey = buildTodayJoinKey(param.getUserId());
            times = redisHashMapAdapter.get(todayJoinKey, phaseIdStr, Integer.class);
            if (times == null) {
                times = 0;
            }
            times += 1;
            redisHashMapAdapter.put(todayJoinKey, phaseIdStr, times);
            redisHashMapAdapter.expire(todayJoinKey, DateUtils.getCurSeconds());
        }

        return joinNum;
    }

    /**
     * 奖券号码构建规则
     */
    private String createLotteryCode(Long participantNum) {
        long codeValue = activityProperties.getCodePrefix() + participantNum;
        return String.valueOf(codeValue);
    }

    /**
     * 扣除用户一次抽奖次数
     */
    private void decrementChances(Long userId) {
        KeyGenerator key = LOTTERY_DRAW_CHANCE_KEY.copy().appendKey(userId);
        if (redisStringAdapter.hasKey(key)) {
            KeyGenerator nextChanceSeconds = LOTTERY_NEXT_CHANCE_KEY.copy().appendKey(userId);

            long afterDecrement = redisStringAdapter.decrement(key);

            if (afterDecrement == 0) {
                //抽奖次数用完，设置重置时间
                long resetSeconds = getChangeExpiredSeconds();
                long expiredSeconds = resetSeconds - (System.currentTimeMillis() / 1000);
                redisStringAdapter.set(nextChanceSeconds, resetSeconds, expiredSeconds);

                //用户抽奖机会重置的推送
                pushNotify(userId, resetSeconds);
                return;
            }

            //扣为了负数，表示超过次数，判断是否存在重置时间，如果存在则不做处理
            if (afterDecrement < 0 && redisStringAdapter.hasKey(nextChanceSeconds)) {
                log.error("调用了扣除抽奖次数方法，但是次数已为负数，逻辑错误，用户ID：{}", userId);
                return;
            }
            //扣除后直接退出
            return;
        }

        //初次调用扣除方法
        long chances = activityProperties.getMaxChances() - 1;
        redisStringAdapter.set(key, chances);
        redisStringAdapter.expire(key, DateUtils.getCurSeconds());
    }

    private void pushNotify(Long userId, long nextChanceSeconds) {
        log.debug("用户[{}]抽奖次数已用完，距离下次抽奖：{}", userId, nextChanceSeconds);
        if (nextChanceSeconds < activityProperties.getLoopSeconds()) {
            log.info("零点的抽奖机会重置不产生推送,userId:{},nextChanceSeconds:{}",
                    userId,
                    nextChanceSeconds);
            return;
        }
        if (null != scheduleService) {
            Date fireDate = new Date(nextChanceSeconds * 1000);
            scheduleService.push(OnceTaskBuilder.builder(fireDate, drawChanceResetTimer)
                    .callbackParam(userId)
                    .taskName("resetChances_" + userId)
                    .author("liujia")
                    .description("通知用户抽奖次数已重置")
                    .build());
        }
    }

    /**
     * 获取用户剩余的参与抽奖次数
     */
    private long getChances(Long userId) {
        KeyGenerator key = LOTTERY_DRAW_CHANCE_KEY.copy().appendKey(userId);
        KeyGenerator nextChanceSeconds = LOTTERY_NEXT_CHANCE_KEY.copy().appendKey(userId);

        if (redisStringAdapter.hasKey(key)) {
            //判断是否存在重置时间，如果存在表示正在倒计时，返回0
            if (redisStringAdapter.hasKey(nextChanceSeconds)) {
                return 0;
            }

            long chances = redisStringAdapter.getLong(key);
            if (chances > 0) {
                return chances;
            }
        }

        redisStringAdapter.set(key, activityProperties.getMaxChances());
        redisStringAdapter.expire(key, DateUtils.getCurSeconds());
        return activityProperties.getMaxChances();
    }

    /**
     * 下次抽奖的秒数
     */
    private long getNextChanesSeconds(Long userId) {
        KeyGenerator key = LOTTERY_NEXT_CHANCE_KEY.copy().appendKey(userId);
        long expiredTime = redisStringAdapter.getLong(key);

        if (expiredTime == 0) {
            return expiredTime;
        }

        return expiredTime - (System.currentTimeMillis() / 1000);
    }

    /**
     * 用户抽奖次数的过期时间
     * 四小时过期，但是不超过今日零点
     *
     * @return 过期的截止时间
     */
    private long getChangeExpiredSeconds() {
        long maxLoopSeconds = activityProperties.getLoopSeconds();
        long lastSeconds = DateUtils.getCurSeconds();
        return Math.min(maxLoopSeconds, lastSeconds) + (System.currentTimeMillis() / 1000);
    }

    /**
     * 获取用户今日参与活动的次数
     *
     * @param userId  用户ID
     * @param phaseId 活动ID
     * @return 已参与次数
     */
    private Integer getTodayJoinTimes(Long userId, Long phaseId) {
        KeyGenerator key = buildTodayJoinKey(userId);
        Integer times = redisHashMapAdapter.get(key, phaseId.toString(), Integer.class);
        if (times == null) {
            return 0;
        }
        return times;
    }

    private KeyGenerator buildTodayJoinKey(Long userId) {
        return LOTTERY_JOIN_TODAY_KEY.copy()
                .appendKey(DateUtils.formatDate(new Date()))
                .appendKey(userId);
    }

    @Override
    public Map<String, Integer> loadJoinHistoryMap(Long userId, Long phaseId) {
        KeyGenerator key = LOTTERY_JOIN_HISTORY_KEY.copy().appendKey(userId);

        Map<String, Integer> res = Maps.newHashMap();
        String phaseIdStr = null == phaseId ? null : String.valueOf(phaseId);

        if (redisHashMapAdapter.hasKey(key)) {
            Map<String, Integer> tempMap = Maps.newHashMap();

            List<JoinHistoryBean> histories = lotteryParticipantMapper.getUserJoinHistory(userId);
            histories.forEach(item -> tempMap.put(String.valueOf(item.getPhaseId()), item.getCount()));

            if (tempMap.size() == 0) {
                tempMap.put("-1", -1);
            }

            redisHashMapAdapter.putAll(key, tempMap);
            redisHashMapAdapter.expire(key, HISTORY_EXPIRED_SECONDS);

            if (null != phaseId) {
                res.put(phaseIdStr, tempMap.get(phaseIdStr));
            } else {
                res.putAll(tempMap);
            }
        } else {
            if (null != phaseId) {
                int count = redisHashMapAdapter.get(key, phaseIdStr, Integer.class);
                res.put(phaseIdStr, count);
            } else {
                res = redisHashMapAdapter.entries(key, Integer.class);
            }
        }

        return res;
    }

    @Override
    public PageWarper<LotteryParticipantDTO> getParticipantList(LotteryPhaseQueryParam param) {
        if (null == param) {
            param = new LotteryPhaseQueryParam();
        }

        List<LotteryParticipantDTO> result = lotteryParticipantMapper.getParticipantByPage(param);
        return new PageWarper<>(result);
    }

    @Override
    public LotteryChanceDTO getUserChance(Long userId) {
        return LotteryChanceDTO.builder()
                .maxChances((int) activityProperties.getMaxChances())
                .nextChanceSeconds(getNextChanesSeconds(userId))
                .residueChances((int) getChances(userId))
                .build();
    }

    @Override
    public PageWarper<LotteryHistoryDTO> getHistoryPhaseList(LotteryQueryParam param) {
        List<LotteryHistoryDTO> result = lotteryParticipantMapper.getUserHistoryByPage(param);

        Map<String, Integer> userJoinHistories = loadJoinHistoryMap(param.getUserId(), null);

        for (LotteryDTO item : result) {
            Integer joinTimes = userJoinHistories.get(String.valueOf(item.getPhaseId()));
            if (null != joinTimes && joinTimes > 0) {
                item.setTimes(joinTimes);
            }
        }

        return new PageWarper<>(result);
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }
}















