package com.bxm.fossicker.activity.lottery.service.impl;

import com.bxm.fossicker.activity.domain.lottery.LotteryMapper;
import com.bxm.fossicker.activity.lottery.config.LotteryStatus;
import com.bxm.fossicker.activity.lottery.service.LotteryPhaseService;
import com.bxm.fossicker.activity.lottery.service.LotteryService;
import com.bxm.fossicker.activity.model.param.lottery.LotteryQueryPageParam;
import com.bxm.fossicker.activity.model.vo.lottery.LotteryBean;
import com.bxm.fossicker.vo.PageWarper;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.vo.Message;
import com.google.common.base.Preconditions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.Date;
import java.util.List;
import java.util.Objects;

import static com.bxm.fossicker.activity.constants.ActivityRedisKeyConstant.LOTTERY_INFO_MAP_KEY;

@Service
public class LotteryServiceImpl extends BaseService implements LotteryService {

    private LotteryMapper lotteryMapper;

    private LotteryPhaseService lotteryPhaseService;

    private RedisHashMapAdapter redisHashMapAdpter;

    @Autowired
    public LotteryServiceImpl(LotteryMapper lotteryMapper,
                              LotteryPhaseService lotteryPhaseService,
                              RedisHashMapAdapter redisHashMapAdpter) {
        this.lotteryMapper = lotteryMapper;
        this.lotteryPhaseService = lotteryPhaseService;
        this.redisHashMapAdpter = redisHashMapAdpter;
    }

    @Override
    public Message addLottery(LotteryBean entity) {
        if (null == entity.getId()) {
            entity.setId(nextId());
        }

        entity.setCreateTime(new Date());
        entity.setDeleteFlag(false);
        entity.setPhaseCount(0);
        //默认是未开始状态
        entity.setStatus(LotteryStatus.DISABLE.getCode());

        if (null == entity.getOrder()) {
            entity.setOrder(0);
        }

        reorder(entity.getId(), entity.getOrder());

        int res = lotteryMapper.insert(entity);

        return Message.build(res);
    }

    /**
     * 重新排序，方便运营随便输入排序
     *
     * @param lotteryId 当前操作的活动
     * @param order     当前活动的排序
     */
    private void reorder(Long lotteryId, int order) {
        //判断当前排序是否存在活动
        List<Long> existsLotteryList = lotteryMapper.getLotteryByOrder(order);

        if (!CollectionUtils.isEmpty(existsLotteryList)) {
            if (existsLotteryList.size() == 1) {
                if (existsLotteryList.stream().anyMatch(code -> code.equals(lotteryId))) {
                    return;
                }
            }

            //将排序在当前编辑顺序后的活动顺序+1
            lotteryMapper.updateOrder(order);
        }
    }

    @Override
    public Message modify(LotteryBean entity) {
        Preconditions.checkArgument(null != entity);
        Preconditions.checkArgument(null != entity.getId());

        reorder(entity.getId(), entity.getOrder());

        //更新活动基础信息
        Message message = Message.build(lotteryMapper.updateByPrimaryKeySelective(entity));
        //变更当前处于启用状态的活动期，变更活动需要的参与人数
        message.append(lotteryPhaseService.modify(entity));

        cleanCache(entity.getId());


        return message;
    }

    private void cleanCache(Long lotteryId) {
        redisHashMapAdpter.remove(LOTTERY_INFO_MAP_KEY, lotteryId.toString());
    }

    @Override
    public Message remove(Long lotteryId, Long operator) {
        LotteryBean entity = LotteryBean.builder()
                .deleteFlag(true)
                .deleter(operator)
                .id(lotteryId)
                .deleteTime(new Date())
                .build();

        return Message.build(lotteryMapper.updateByPrimaryKeySelective(entity));
    }

    @Override
    public Message change(Long lotteryId, Byte status) {
        Preconditions.checkArgument(null != lotteryId);
        if (status == null) {
            status = 0;
        }

        LotteryBean lottery = LotteryBean.builder().id(lotteryId).status(status).build();
        Message message = Message.build(lotteryMapper.updateByPrimaryKeySelective(lottery));

        cleanCache(lotteryId);

        //活动变更为启动状态时,继续新一期活动
        if (status == LotteryStatus.ENABLE.getCode()) {
            lotteryPhaseService.resumePhase(loadCache(lotteryId));
        } else {
            //禁用时，取消活动
            lotteryPhaseService.cancel(lottery);
        }

        return message;
    }

    @Override
    public LotteryBean loadCache(Long lotteryId) {
        String idStr = Objects.toString(lotteryId);
        LotteryBean lottery = redisHashMapAdpter.get(LOTTERY_INFO_MAP_KEY, idStr, LotteryBean.class);

        if (null == lottery) {
            lottery = lotteryMapper.selectByPrimaryKey(lotteryId);
            if (null == lottery) {
                lottery = LotteryBean.builder().id(lotteryId).build();
            }
            redisHashMapAdpter.put(LOTTERY_INFO_MAP_KEY, idStr, lottery);
        }
        return lottery;
    }

    @Override
    public PageWarper<LotteryBean> getByPage(LotteryQueryPageParam param) {
        if (null == param) {
            param = new LotteryQueryPageParam();
        }
        List<LotteryBean> result = lotteryMapper.getLotteryByPage(param);

        return new PageWarper<>(result);
    }
}
