package com.bxm.fossicker.activity.service.advert.impl;

import com.bxm.fossicker.activity.model.dto.AdvertDto;
import com.bxm.fossicker.activity.model.enums.AdvertPositionEnum;
import com.bxm.fossicker.activity.model.param.AdvertParam;
import com.bxm.fossicker.activity.service.advert.AdvertisementFilter;
import com.bxm.fossicker.activity.service.advert.AdvertisementFilterChainService;
import com.bxm.fossicker.activity.service.advert.annotation.AdverFilter;
import com.bxm.fossicker.activity.service.advert.bo.AdvertisementFilterParam;
import com.bxm.fossicker.user.facade.UserInfoFacadeService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * AdvertisementFilter 执行链
 *
 * @authoer Gonzo
 * @date 2019-08-21 10:44
 * <p>
 */
@Component
public class AdvertFilterChainServiceImpl implements ApplicationContextAware, AdvertisementFilterChainService {

    private Map<AdvertPositionEnum, List<AdvertisementFilter>> advertisements = Maps.newHashMap();

    /**
     * 默认符合所有过滤规则的filter
     */
    private List<AdvertisementFilter> defaultFilters = Lists.newArrayList();

    @Resource
    private UserInfoFacadeService userInfoFacadeService;

    @Override
    public void filter(List<AdvertDto> advertDtos, AdvertParam advertParam) {

        if (CollectionUtils.isEmpty(advertDtos)) {
            return;
        }

        AdvertisementFilterParam param = AdvertisementFilterParam.builder()
                .advertParam(advertParam)
                .userInfoFacadeService(userInfoFacadeService)
                .position(advertParam.getPosition()).build();

        // 先执行默认的
        if (null != defaultFilters) {
            for (AdvertisementFilter advertisementFilter : defaultFilters) {
                if (!advertisementFilter.filter(advertDtos, param)) {
                    return;
                }
            }
        }

        // 再执行指定的
        List<AdvertisementFilter> advertisementFilters = advertisements.get(AdvertPositionEnum.getByPosition(param.getPosition()));
        if (null != advertisementFilters) {
            for (AdvertisementFilter advertisementFilter : advertisementFilters) {
                if (!advertisementFilter.filter(advertDtos, param)) {
                    return;
                }
            }
        }
    }


    /**
     * 初始化 advertisements
     *
     * @param applicationContext
     * @throws BeansException
     */
    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        Collection<AdvertisementFilter> advertisementFilterMap = applicationContext.getBeansOfType(AdvertisementFilter.class).values();

        advertisementFilterMap.forEach(p -> {
            AdverFilter adverFilter = AnnotationUtils.getAnnotation(AopUtils.getTargetClass(p), AdverFilter.class);

            if (null != adverFilter) {
                addFilter(adverFilter.value(), p);
            }
        });
    }


    /**
     * 根据position 进行初始化map
     *
     * @param positions
     * @param filter
     */
    private void addFilter(AdvertPositionEnum[] positions, AdvertisementFilter filter) {

        if (null == positions || positions.length == 0) {
            defaultFilters.add(filter);
        }

        for (AdvertPositionEnum position : positions) {
            List<AdvertisementFilter> advertisementFilters = advertisements.get(position);

            if (null == advertisementFilters) {
                advertisementFilters = Lists.newArrayList();
                advertisements.put(position, advertisementFilters);
            }

            advertisementFilters.add(filter);
        }

    }
}
