package com.bxm.fossicker.activity.service.advert.impl;

import com.bxm.fossicker.activity.config.AdvertConfig;
import com.bxm.fossicker.activity.enums.AdvertSceneEnum;
import com.bxm.fossicker.activity.model.dto.AdvertSceneDTO;
import com.bxm.fossicker.activity.model.dto.AdvertStrategyConfigDTO;
import com.bxm.fossicker.activity.model.dto.AdvertStrategyDTO;
import com.bxm.fossicker.activity.model.param.AdvertSceneParam;
import com.bxm.fossicker.activity.model.vo.AdvertSceneInfo;
import com.bxm.fossicker.activity.service.advert.AdvertStrategyService;
import com.bxm.fossicker.activity.service.config.ActivityProperties;
import com.bxm.fossicker.user.facade.UserInfoFacadeService;
import com.bxm.fossicker.vo.BasicParam;
import com.alibaba.fastjson.JSONArray;
import com.bxm.newidea.component.tools.StringUtils;
import com.google.common.collect.Maps;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Map;

@Service
public class AdvertStrategyServiceImpl implements AdvertStrategyService {

    private final ActivityProperties activityProperties;

    @Autowired
    private AdvertConfig advertConfig;

    @Autowired
    private UserInfoFacadeService userInfoFacadeService;

    /**
     * 解析结果缓存
     */
    private Map<String, List<AdvertStrategyDTO>> cacheMap;

    /**
     * 最后一次解析时间
     */
    private long lastParseTime;

    /**
     * 默认匹配的广告获取策略
     */
    private List<AdvertStrategyDTO> defaultMatch;

    /**
     * 解析间隔
     */
    private static final long PARSE_INTERVAL = 5 * 60 * 1000;

    /**
     * 默认的广告策略名称，配置到version字段上
     */
    private static final String DEFAULT_STRATEGY_NAME = "default";

    @Autowired
    public AdvertStrategyServiceImpl(ActivityProperties activityProperties) {
        this.activityProperties = activityProperties;
    }

    @Override
    public List<AdvertStrategyDTO> getAdvertStrategy(BasicParam basicParam) {
        //每次解析间隔获取一次远程的配置
        if (System.currentTimeMillis() - lastParseTime > PARSE_INTERVAL) {
            parse();
        }

        List<AdvertStrategyDTO> matchRes = cacheMap.get(basicParam.getCurVer());

        if (null == matchRes) {
            matchRes = defaultMatch;
        }

        return matchRes;
    }

    @SuppressWarnings("checkstyle:WhitespaceAround")
    @Override
    public AdvertSceneDTO getAdvertForScene(AdvertSceneParam param) {

        String sceneInfoStr = advertConfig.getSceneKeys().get(AdvertSceneEnum.getSceneCode(param.getPosition()));
        if(StringUtils.isBlank(sceneInfoStr)){
            return null;
        }

        List<AdvertSceneInfo> sceneInfoList = JSONArray.parseArray(sceneInfoStr,AdvertSceneInfo.class);

        if(CollectionUtils.isEmpty(sceneInfoList)){
            return null;
        }

        //是否是新用户
        boolean isNoviceUser =  userInfoFacadeService.isNoviceUser(param.getUserId());

        AdvertSceneDTO advertSceneDTO = new AdvertSceneDTO();
        //匹配广告信息
        for (AdvertSceneInfo advertSceneInfo : sceneInfoList) {
            if(!isNoviceUser){
                if(advertSceneInfo.getIsNoviceUser()==0 && param.getPlatform()==advertSceneInfo.getPlatform()){
                    BeanUtils.copyProperties(advertSceneInfo,advertSceneDTO);
                    break;
                }
            }else{
                if(advertSceneInfo.getIsNoviceUser()==1 && param.getPlatform() == advertSceneInfo.getPlatform()){
                    BeanUtils.copyProperties(advertSceneInfo,advertSceneDTO);
                }
            }
        }
        return advertSceneDTO;
    }

    /**
     * 解析配置的数据，方便获取
     */
    private void parse() {
        String config = activityProperties.getAdvertStrategyConfigList();
        defaultMatch = null;
        cacheMap = Maps.newHashMap();

        if (StringUtils.isNotBlank(config)) {
            List<AdvertStrategyConfigDTO> advertStrategyConfigList = JSONArray.parseArray(config, AdvertStrategyConfigDTO.class);

            int configSize = advertStrategyConfigList.size();

            if (configSize > 0) {
                advertStrategyConfigList.forEach(strategy -> cacheMap.put(strategy.getVersion(), strategy.getConfig()));

                //获取默认的配置，如果未配置则将最后一条作为配置返回
                if (cacheMap.containsKey(DEFAULT_STRATEGY_NAME)) {
                    defaultMatch = cacheMap.get(DEFAULT_STRATEGY_NAME);
                } else {
                    defaultMatch = advertStrategyConfigList.get(configSize - 1).getConfig();
                }
            }

        }

        lastParseTime = System.currentTimeMillis();
    }
}
