package com.bxm.fossicker.activity.service.clock.impl;

import com.bxm.fossicker.activity.facade.TelephoneChargeFacadeService;
import com.bxm.fossicker.activity.model.constant.ActivityRedisKey;
import com.bxm.fossicker.activity.model.dto.ClockInfoDTO;
import com.bxm.fossicker.activity.model.param.ClockActivityParam;
import com.bxm.fossicker.activity.model.param.ClockOperateParam;
import com.bxm.fossicker.activity.model.vo.ClockInfo;
import com.bxm.fossicker.activity.service.clock.ClockActivityService;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Calendar;
import java.util.Date;
import java.util.Objects;

/**
 * @author pf.w
 * @date 2021/5/17 16:36
 **/
@Service
public class ClockActivityServiceImpl implements ClockActivityService {

    @Autowired
    private RedisHashMapAdapter redisHashMapAdapter;

    @Autowired
    private TelephoneChargeFacadeService telephoneChargeFacadeService;

    /**
     * 打卡周期天数
     */
    private static Integer CLOCK_OVER_DAYS = 7;

    @Override
    public ClockInfoDTO getClockInfo(ClockActivityParam param) {
        //获取用户的打卡信息
        ClockInfo clockInfo = getClockInfoFromRedis(param.getUserId());

        //没有用户的打卡信息
        if(Objects.isNull(clockInfo)){
            return ClockInfoDTO.builder()
                    .clockToday(0)
                    .clockLastDays(0)
                    .buttonStatus(1)
                    .build();
        }

        //获取用户最后的打卡时间和当前时间的相差天数
        int days = this.differentDays(new Date(),clockInfo.getClockLastTime());

        //中间有断更 只能重新打卡
        if(days>=2){
            return ClockInfoDTO.builder()
                    .clockToday(0)
                    .clockLastDays(0)
                    .buttonStatus(1)
                    .build();
        }

        //相差天数为0 说明今天已经打卡
        if(days==0){
            return ClockInfoDTO.builder()
                    .clockToday(1)
                    .clockLastDays(clockInfo.getClockLastDays())
                    //比较是否已打满
                    .buttonStatus(Objects.equals(clockInfo.getClockLastDays(),CLOCK_OVER_DAYS) ? 3 : 2)
                    .build();
        }

        if(clockInfo.getClockLastDays()>=CLOCK_OVER_DAYS){
            return ClockInfoDTO.builder()
                    .clockToday(0)
                    .clockLastDays(clockInfo.getClockLastDays())
                    //比较是否已打满
                    .buttonStatus(3)
                    .build();
        }
        return  ClockInfoDTO.builder()
                .clockToday(0)
                .clockLastDays(clockInfo.getClockLastDays())
                .buttonStatus(1)
                .build();
    }

    @Override
    public ClockInfoDTO clockOperateCallBack(ClockOperateParam param) {
        //获取用户的打卡信息
        ClockInfo clockInfo = getClockInfoFromRedis(param.getUserId());

        if(Objects.isNull(clockInfo)){
            redisHashMapAdapter.put(getClockInfoKey(),param.getUserId().toString(),ClockInfo.builder()
                    .clockLastDays(1)
                    .clockLastTime(new Date())
                    .build());

            return this.getClockInfo(param);
        }

        //获取用户最后的打卡时间和当前时间的相差天数
        int days = differentDays(new Date(),clockInfo.getClockLastTime());

        //断更则重新计算
        if(days>=2){

            redisHashMapAdapter.put(getClockInfoKey(),param.getUserId().toString(),ClockInfo.builder()
                    .clockLastDays(1)
                    .clockLastTime(new Date())
                    .build());

            return this.getClockInfo(param);
        }

        //已经打过则不再更新
        if(DateUtils.isSameDay(new Date(),clockInfo.getClockLastTime())){
            return this.getClockInfo(param);
        }

        //更新用户打卡信息
        redisHashMapAdapter.put(getClockInfoKey(),param.getUserId().toString(),ClockInfo.builder()
                .clockLastDays(clockInfo.getClockLastDays()+1)
                .clockLastTime(new Date())
                .build());

        //本次打卡为最后一次
        if((clockInfo.getClockLastDays()+1)==CLOCK_OVER_DAYS){
            //增加打卡话费记录
            telephoneChargeFacadeService.addTelephoneChargeList(param.getUserId(),(byte)3);
            //记录用户完成情况
            redisHashMapAdapter.put(getHasOverClockUserKey(),param.getUserId().toString(),System.currentTimeMillis());
        }

        return this.getClockInfo(param);
    }

    private ClockInfo getClockInfoFromRedis(Long userId){
        ClockInfo clockInfo = redisHashMapAdapter.get(getClockInfoKey(),userId.toString(),ClockInfo.class);
        if(Objects.isNull(clockInfo)){
            return null;
        }
        return clockInfo;
    }

    private KeyGenerator getClockInfoKey(){
        return ActivityRedisKey.CLOCK_ACTIVITY_INFO_KEY.copy();
    }

    private KeyGenerator getHasOverClockUserKey(){
        return ActivityRedisKey.CLOCK_ACTIVITY_INFO_HAS_OVER_KEY.copy();
    }

    private  int differentDays(Date date1,Date date2)
    {
        Calendar cal1 = Calendar.getInstance();
        cal1.setTime(date1);
        Calendar cal2 = Calendar.getInstance();
        cal2.setTime(date2);
        int day1= cal1.get(Calendar.DAY_OF_YEAR);
        int day2 = cal2.get(Calendar.DAY_OF_YEAR);

        int year1 = cal1.get(Calendar.YEAR);
        int year2 = cal2.get(Calendar.YEAR);
        //不同一年
        if(year1 != year2)
        {
            int timeDistance = 0 ;
            for(int i = year1 ; i < year2 ; i ++)
            {
                //闰年
                if(i%4==0 && i%100!=0 || i%400==0)
                {
                    timeDistance += 366;
                }else
                {
                    //不是闰年
                    timeDistance += 365;
                }
            }

            return timeDistance + (day2-day1) ;
        } else
        {
            return Math.abs(day2-day1);
        }
    }

}
