package com.bxm.fossicker.activity.service.debris.cache;

import com.alibaba.fastjson.JSON;
import com.bxm.fossicker.activity.constants.ActivityRedisKeyConstant;
import com.bxm.fossicker.activity.constants.DebrisActivityAwardType;
import com.bxm.fossicker.activity.model.dto.debris.DebrisGrantRuleDTO;
import com.bxm.fossicker.activity.service.config.ActivityDebrisProperties;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.tools.RandomUtils;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Objects;

/**
 * 碎片规则缓存管理类
 *
 * @author wzy
 * @version 1.0
 * @date 2021/4/12 3:40 下午
 */
@Slf4j
@Service
@AllArgsConstructor
public class GrantDebrisCacheManager {

    private final RedisHashMapAdapter redisHashMapAdapter;

    private final ActivityDebrisProperties activityDebrisProperties;

    /**
     * 是否能够获取碎片
     *
     * @param type            奖品类型
     * @param targetDebrisNum 目标获取碎片第几个
     * @param hasWatchNum     已经观看视频的个数
     * @return 是否能够获取碎片
     */
    public boolean hasGrantDebris(Integer type, Integer targetDebrisNum, Integer hasWatchNum) {

        int targetWatchNum = getTargetWatchNum(type, targetDebrisNum);

        log.info("类型：{}, 目标碎片数：{}, 当前已看视频数：{}, 目标视频个数：{}", type,
                targetDebrisNum, hasWatchNum, targetWatchNum);
        if (targetWatchNum <= 0) {
            return false;
        }

        return hasWatchNum >= targetWatchNum;
    }

    /**
     * 删除生成的随机规则
     */
    public void remove(Integer type) {
        redisHashMapAdapter.remove(buildGrantNumCache(type));
    }

    /**
     * 获取指定类型、指定个数需要看多少个视频
     *
     * @param type      指定类型，1：话费 2：虚拟物品
     * @param targetNum 获取第多少个碎片
     * @return 需要看多少个视频
     */
    private int getTargetWatchNum(Integer type, Integer targetNum) {
        if (!redisHashMapAdapter.hasKey(buildGrantNumCache(type))) {
            initGrantDebrisRule(type);
        }
        Integer watchNum = redisHashMapAdapter.get(buildGrantNumCache(type), String.valueOf(targetNum), Integer.class);

        return Objects.isNull(watchNum) ? -1 : watchNum;
    }


    /**
     * 初始化获得碎片规则
     *
     * @param type 奖品类型
     */
    private void initGrantDebrisRule(Integer type) {
        KeyGenerator grantNumCacheKey = buildGrantNumCache(type);

        String debrisRule;
        if (Objects.equals(type, DebrisActivityAwardType.CHARGE.getCode())) {
            debrisRule = activityDebrisProperties.getChargeDebrisRule();
        } else {
            //实际上就是type等于2这种情况，也就是虚拟奖品
            debrisRule = activityDebrisProperties.getVirtualDebrisRule();
        }
        List<DebrisGrantRuleDTO> ruleList = getDebrisRule(debrisRule);

        int targetDebrisCounter = 0;
        for (DebrisGrantRuleDTO ruleItem : ruleList) {
            targetDebrisCounter += RandomUtils.nextInt(ruleItem.getValue().getLeft(), ruleItem.getValue().getRight() + 1);
            redisHashMapAdapter.put(grantNumCacheKey, ruleItem.getKey(), targetDebrisCounter);
        }
    }

    private List<DebrisGrantRuleDTO> getDebrisRule(String debrisRule) {
        return JSON.parseArray(debrisRule, DebrisGrantRuleDTO.class);
    }

    /**
     * 构建授予碎片规则缓存
     *
     * @param type 类型
     * @return 缓存key
     */
    private KeyGenerator buildGrantNumCache(Integer type) {
        return ActivityRedisKeyConstant.GRANT_DEBRIS_RULE_KEY.copy().appendKey(type);
    }
}