package com.bxm.fossicker.activity.service.debris.impl;

import com.bxm.fossicker.activity.constants.DebrisReceiveType;
import com.bxm.fossicker.activity.constants.DebrisStatus;
import com.bxm.fossicker.activity.constants.DistributedLockContant;
import com.bxm.fossicker.activity.domain.debris.ActivityAwardMapper;
import com.bxm.fossicker.activity.domain.debris.ActivityAwardPhaseMapper;
import com.bxm.fossicker.activity.domain.debris.ActivityAwardPhaseRecordMapper;
import com.bxm.fossicker.activity.facade.TelephoneChargeFacadeService;
import com.bxm.fossicker.activity.model.entity.ActivityAwardEntity;
import com.bxm.fossicker.activity.model.entity.ActivityAwardPhaseEntity;
import com.bxm.fossicker.activity.model.entity.ActivityAwardPhaseRecordEntity;
import com.bxm.fossicker.activity.model.param.debris.ActivityExchangeParam;
import com.bxm.fossicker.activity.model.param.debris.ActivityUserPageParam;
import com.bxm.fossicker.activity.model.param.debris.ActivityUserParam;
import com.bxm.fossicker.activity.model.vo.debris.ActivityBarrageVo;
import com.bxm.fossicker.activity.model.vo.debris.ActivityListVo;
import com.bxm.fossicker.activity.model.vo.debris.MyActivityAwardVo;
import com.bxm.fossicker.activity.model.vo.debris.ReceiveDebrisResultVo;
import com.bxm.fossicker.activity.service.config.ActivityDebrisProperties;
import com.bxm.fossicker.activity.service.debris.ActivityDebrisService;
import com.bxm.fossicker.activity.service.debris.cache.GrantDebrisCacheManager;
import com.bxm.fossicker.activity.service.debris.cache.UserDebrisCacheManager;
import com.bxm.fossicker.activity.service.debris.cache.UserLotteryCacheManager;
import com.bxm.fossicker.activity.service.debris.cache.UserTotalLotteryCacheManager;
import com.bxm.fossicker.base.facade.PointReportFacadeService;
import com.bxm.fossicker.base.facade.param.PointParam;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 活动碎片service 实现类
 *
 * @author wzy
 * @version 1.0
 * @date 2021/4/12 3:38 下午
 */
@Slf4j
@Service
@AllArgsConstructor
public class ActivityDebrisServiceImpl implements ActivityDebrisService {

    private final ActivityAwardMapper activityAwardMapper;

    private final ActivityAwardPhaseMapper activityAwardPhaseMapper;

    private final ActivityAwardPhaseRecordMapper activityAwardPhaseRecordMapper;

    private final UserDebrisCacheManager userDebrisCacheManager;

    private final GrantDebrisCacheManager grantDebrisCacheManager;

    private final SequenceCreater sequenceCreater;

    private final ActivityDebrisProperties activityDebrisProperties;

    private final TelephoneChargeFacadeService telephoneChargeFacadeService;

    private final DistributedLock distributedLock;

    private final PointReportFacadeService pointReportFacadeService;

    private final UserLotteryCacheManager userLotteryCacheManager;

    private final UserTotalLotteryCacheManager userTotalLotteryCacheManager;


    @Override
    public List<ActivityListVo> getActivityList(ActivityUserPageParam userPageParam) {
        List<ActivityListVo> activityAwardList = activityAwardMapper.getActivityAwardList(userPageParam);
        activityAwardList.forEach(activityAwardItem ->
                activityAwardItem.setFirstReceive(!getHistoryReceiveActivityDebris(activityAwardItem.getActivityAwardId(), userPageParam.getUserId())));

        return activityAwardList;
    }

    @Override
    public List<ActivityBarrageVo> getBarrageList() {
        List<ActivityAwardEntity> allActivityAwardList = activityAwardMapper.getAllActivityAward();

        return allActivityAwardList.stream().map(activityAwardEntity -> ActivityBarrageVo.builder()
                .awardTitle(activityAwardEntity.getTitle())
                .phone(StringUtils.hideMobile(StringUtils.getRandomPhoneNum()))
                .build()).limit(10).collect(Collectors.toList());
    }

    @Override
    public List<MyActivityAwardVo> getMyAwardList(ActivityUserPageParam userPageParam) {
        List<MyActivityAwardVo> userAwardPhaseList = activityAwardPhaseMapper.getUserAwardPhaseList(userPageParam);

        userAwardPhaseList.forEach(activityAwardItem ->
                activityAwardItem.setFirstReceive(!getHistoryReceiveActivityDebris(activityAwardItem.getActivityAwardId(), userPageParam.getUserId())));

        return userAwardPhaseList;
    }

    @Override
    public Message receiveDebris(ActivityUserParam activityUserParam) {
        //加分布式锁
        String key = buildReceiveDebrisKey(activityUserParam.getActivityAwardId(), activityUserParam.getUserId());
        String requestId = sequenceCreater.nextStringId();

        if (!distributedLock.lock(key, requestId, 2, TimeUnit.SECONDS)) {
            return Message.build(false, "领取碎片操作过于频繁");
        }

        //添加埋点
        pointReportFacadeService.add(buildJoinActivityPointParam(activityUserParam, 1));

        // 判断今天是否已经领取过碎片
        boolean hasHistoryReceive = getHistoryReceiveActivityDebris(activityUserParam.getActivityAwardId(),
                activityUserParam.getUserId());

        //获取奖品和期数信息
        ActivityAwardEntity activityAwardEntity = activityAwardMapper.selectByPrimaryKey(activityUserParam.getActivityAwardId());
        //v2.2.2 去除掉了期数的概念，相当于是用户->奖品的领取情况
        ActivityAwardPhaseEntity userPhaseInfo = activityAwardPhaseMapper.getCurrentPhaseInfo(activityUserParam.getActivityAwardId(), activityUserParam.getUserId());

        // 如果不是首次参与则返回领取信息不增加碎片
        if (hasHistoryReceive) {
            return Message.build(true)
                    .addParam("resultVO", ReceiveDebrisResultVo.builder()
                            .success(true)
                            .targetDebrisNum(userPhaseInfo.getTargetNum())
                            .currentDebrisNum(userPhaseInfo.getCurrentNum())
                            .mainPicture(userPhaseInfo.getAwardPicture())
                            .title(userPhaseInfo.getAwardTitle())
                            .activityAwardId(activityUserParam.getActivityAwardId())
                            .build());

        }

        //处理第一次领取碎片
        ActivityAwardPhaseEntity dbPhaseInfo = handleFirstDebris(activityUserParam, activityAwardEntity, userPhaseInfo);

        //分布式锁解锁
        distributedLock.unlock(key, requestId);

        return Message.build(true).
                addParam("resultVO", ReceiveDebrisResultVo.builder()
                        .success(true)
                        .targetDebrisNum(dbPhaseInfo.getTargetNum())
                        .currentDebrisNum(dbPhaseInfo.getCurrentNum())
                        .mainPicture(dbPhaseInfo.getAwardPicture())
                        .title(dbPhaseInfo.getAwardTitle())
                        .activityAwardId(activityUserParam.getActivityAwardId())
                        .build());
    }


    @Override
    public Message addUserDebris(ActivityUserParam activityUserParam) {

        //加分布式锁
        String key = buildReceiveDebrisKey(activityUserParam.getActivityAwardId(), activityUserParam.getUserId());
        String requestId = sequenceCreater.nextStringId();

        if (!distributedLock.lock(key, requestId, 2, TimeUnit.SECONDS)) {
            return Message.build(false, "领取碎片操作过于频繁");
        }

        //添加埋点
        pointReportFacadeService.add(buildJoinActivityPointParam(activityUserParam, 2));
        //增加抽奖次数
        userTotalLotteryCacheManager.incrementUserLotteryNum(activityUserParam.getUserId());
        userLotteryCacheManager.incrementUserLotteryNum(activityUserParam.getUserId());

        //获取奖品信息和期数信息
        ActivityAwardEntity activityAwardEntity = activityAwardMapper.selectByPrimaryKey(activityUserParam.getActivityAwardId());
        //v2.2.2 去除掉了期数的概念，相当于是用户->奖品的领取情况
        ActivityAwardPhaseEntity userPhaseInfo = activityAwardPhaseMapper
                .getCurrentPhaseInfo(activityUserParam.getActivityAwardId(), activityUserParam.getUserId());

        // 一种错误情况进行记录，第二次领取一定会有领取记录
        if (Objects.isNull(userPhaseInfo)) {
            log.info("调用第二次领取接口，但是未首次领取，活动奖品id：{}, 用户id：{} ", activityUserParam.getActivityAwardId(), activityUserParam.getUserId());

            //如果不存在则创建期数，虽然这种情况几乎不可能出现，执行到这里，说明跳过了第一次领取，初始化第一次领取
            userPhaseInfo = handleFirstDebris(activityUserParam, activityAwardEntity, null);
        }

        //看视频数量+1
        //获取所有看视频数量，是否可以获取新的碎片
        int currentWatchNum = userDebrisCacheManager.increaseWatchNum(activityUserParam.getUserId(), userPhaseInfo.getId());

        //判断是否可以获得新增领取碎片记录
        boolean grantDebris = grantDebrisCacheManager.hasGrantDebris(activityAwardEntity.getType(), userPhaseInfo.getCurrentNum() + 1, currentWatchNum);

        log.info("用户领取碎片结果：{}，用户id：{}, 活动id：{}, 期数id：{}，当前碎片个数：{}, 观看视频次数：{}",
                grantDebris, activityUserParam.getUserId(), activityUserParam.getActivityAwardId(), userPhaseInfo.getId(),
                userPhaseInfo.getCurrentNum(), currentWatchNum);

        // 如果不可领取则返回领取失败
        if (Boolean.FALSE.equals(grantDebris)) {
            //分布式锁解锁
            distributedLock.unlock(key, requestId);

            return Message.build(true)
                    .addParam("resultVO", ReceiveDebrisResultVo.builder()
                            .success(false)
                            .mainPicture(activityAwardEntity.getMainPicture())
                            .title(activityAwardEntity.getTitle())
                            .activityAwardId(activityAwardEntity.getId())
                            .targetDebrisNum(userPhaseInfo.getTargetNum())
                            .currentDebrisNum(userPhaseInfo.getCurrentNum())
                            .build());
        }

        //添加领取记录，增加碎片个数
        addActivityAwardPhaseRecord(activityUserParam, userPhaseInfo, DebrisReceiveType.WATCH_RECEIVE);
        // 增加碎片个数
        activityAwardPhaseMapper.increasePhaseDebrisNum(userPhaseInfo.getId());
        //添加领取成功埋点
        pointReportFacadeService.add(buildAddDebrisPointParam(activityUserParam, 2));

        //分布式锁解锁
        distributedLock.unlock(key, requestId);

        return Message.build(true).
                addParam("resultVO", ReceiveDebrisResultVo.builder()
                        .success(true)
                        .targetDebrisNum(userPhaseInfo.getTargetNum())
                        .currentDebrisNum(userPhaseInfo.getCurrentNum() + 1)
                        .mainPicture(userPhaseInfo.getAwardPicture())
                        .title(userPhaseInfo.getAwardTitle())
                        .activityAwardId(activityUserParam.getActivityAwardId())
                        .build());
    }

    @Override
    public Message conversionDebris(ActivityExchangeParam activityExchangeParam) {

        //加分布式锁
        String key = buildConversionDebrisKey(activityExchangeParam.getPhaseId());
        String requestId = sequenceCreater.nextStringId();

        if (!distributedLock.lock(key, requestId, 2, TimeUnit.SECONDS)) {
            return Message.build(false, "兑换话费操作过于频繁");
        }

        ActivityAwardPhaseEntity phaseEntity =
                activityAwardPhaseMapper.selectAvailablePhase(activityExchangeParam.getPhaseId(), activityExchangeParam.getUserId());

        if (Objects.isNull(phaseEntity)) {
            log.error("兑换失败，期数不存在，奖品id：{}, 期数id：{}，用户id：{}",
                    activityExchangeParam.getActivityAwardId(), activityExchangeParam.getPhaseId(), activityExchangeParam.getUserId());
            return Message.build(true)
                    .addParam("resultVO", false);
        }

        //增加话费记录
        telephoneChargeFacadeService.addTelephoneChargeList(activityExchangeParam.getUserId(), (byte) 2);
        //兑换奖品成功埋点
        pointReportFacadeService.add(buildConversionDebrisPointParam(activityExchangeParam));

        //更改期数兑换状态
        ActivityAwardPhaseEntity updateEntity = new ActivityAwardPhaseEntity();
        updateEntity.setId(phaseEntity.getId());
        updateEntity.setStatus(DebrisStatus.EXCHANGE.getCode());
        updateEntity.setUsedTime(new Date());
        activityAwardPhaseMapper.updateByPrimaryKeySelective(updateEntity);

        log.info("兑换成功，奖品id：{}, 期数id：{}，用户id：{}",
                activityExchangeParam.getActivityAwardId(), activityExchangeParam.getPhaseId(), activityExchangeParam.getUserId());

        //分布式锁解锁
        distributedLock.unlock(key, requestId);
        return Message.build(true).addParam("resultVO", true);
    }

    @Override
    public Boolean getHistoryReceiveDebris(Long userId) {
        return getHistoryReceiveActivityDebris(activityDebrisProperties.getChargeActivityId(), userId);
    }

    @Override
    public void clearYesterdayDebris() {
        activityAwardPhaseMapper.clearExpiredPhase();
    }

    @Override
    public boolean checkPopForDebrisBaskPop(Long userId) {
        return true;
    }


    /**
     * 获取历史是否领取过活动的碎片
     *
     * @param activityAwardId 活动奖品id
     * @param userId          用户id
     * @return 历史是否领取过奖品碎片
     */
    private boolean getHistoryReceiveActivityDebris(Long activityAwardId, Long userId) {

        //v2.2.2 去除掉了期数的概念，相当于是用户->奖品的领取情况
        ActivityAwardPhaseEntity userPhaseInfo = activityAwardPhaseMapper.getCurrentPhaseInfo(activityAwardId, userId);

        if (userPhaseInfo == null) {
            return false;
        }

        List<ActivityAwardPhaseRecordEntity> recordList =
                activityAwardPhaseRecordMapper.getRecordListByPhaseId(userPhaseInfo.getId());

        return !CollectionUtils.isEmpty(recordList);
    }

    /**
     * 构建领取碎片分布式锁
     *
     * @param activityAwardId 活动奖品id
     * @param userId          用户id
     * @return 分布式锁key
     */
    private String buildReceiveDebrisKey(Long activityAwardId, Long userId) {
        return DistributedLockContant.RECEIVE_DEBRIS_KEY.copy().appendKey(activityAwardId).appendKey(userId).gen();
    }

    /**
     * 构建兑换话费分布式锁
     *
     * @param phaseId 期数id
     * @return 分布式锁key
     */
    private String buildConversionDebrisKey(Long phaseId) {
        return DistributedLockContant.CONVERSION_DEBRIS_KEY.copy().appendKey(phaseId).gen();
    }

    private void addActivityAwardPhaseRecord(ActivityUserParam activityUserParam,
                                             ActivityAwardPhaseEntity phaseEntity,
                                             DebrisReceiveType debrisReceiveType) {
        // 新增记录信息，第一次领取直接给一个碎片
        ActivityAwardPhaseRecordEntity recordEntity = new ActivityAwardPhaseRecordEntity();

        recordEntity.setId(sequenceCreater.nextLongId());
        recordEntity.setAwardId(activityUserParam.getActivityAwardId());
        recordEntity.setCreateTime(new Date());
        recordEntity.setSource(debrisReceiveType.getCode());
        recordEntity.setUserId(activityUserParam.getUserId());
        recordEntity.setPhaseId(phaseEntity.getId());

        activityAwardPhaseRecordMapper.insertSelective(recordEntity);
    }

    /**
     * 初始化期数
     *
     * @param activityUserParam   入参
     * @param activityAwardEntity 活动奖品实体类
     * @param initDebrisNum       初始化碎片个数
     * @return 期数实体数据
     */
    private ActivityAwardPhaseEntity initActivityAwardPhase(ActivityUserParam activityUserParam,
                                                            ActivityAwardEntity activityAwardEntity,
                                                            Integer initDebrisNum) {
        //新增期数信息
        ActivityAwardPhaseEntity phaseEntity = new ActivityAwardPhaseEntity();

        phaseEntity.setId(sequenceCreater.nextLongId());
        phaseEntity.setAwardId(activityUserParam.getActivityAwardId());
        phaseEntity.setAwardPicture(activityAwardEntity.getMainPicture());
        phaseEntity.setAwardTitle(activityAwardEntity.getTitle());
        phaseEntity.setCreateTime(new Date());
        phaseEntity.setCurrentNum(initDebrisNum);
        phaseEntity.setTargetNum(activityAwardEntity.getPhaseNum());
        phaseEntity.setType(activityAwardEntity.getType());
        phaseEntity.setUserId(activityUserParam.getUserId());
        phaseEntity.setStatus(DebrisStatus.AVAILABLE.getCode());

        activityAwardPhaseMapper.insertSelective(phaseEntity);
        return phaseEntity;
    }

    /**
     * 进行第一次的碎片领取
     *
     * @param activityUserParam   入参
     * @param activityAwardEntity 活动奖品信息
     * @return 返回期数信息
     */
    private ActivityAwardPhaseEntity handleFirstDebris(ActivityUserParam activityUserParam,
                                                       ActivityAwardEntity activityAwardEntity,
                                                       ActivityAwardPhaseEntity userPhaseInfo) {

        log.info("用户当日首次领取碎片：用户id：{}, 奖品id：{}",
                activityUserParam.getUserId(), activityUserParam.getActivityAwardId());

        ActivityAwardPhaseEntity realPhaseInfo = userPhaseInfo;
        if (Objects.isNull(userPhaseInfo)) {
            // 初始化期数,这里初始化碎片为1，是因为第一次领取是可以直接获得一个碎片
            realPhaseInfo = initActivityAwardPhase(activityUserParam,
                    activityAwardEntity, 1);
        }

        // 增加碎片记录
        addActivityAwardPhaseRecord(activityUserParam, realPhaseInfo, DebrisReceiveType.DIRECT_RECEIVE);

        //添加领取成功埋点
        pointReportFacadeService.add(buildAddDebrisPointParam(activityUserParam, 1));

        //增加抽奖次数
        if (activityDebrisProperties.getFirstReceiveAddLotteryNum()) {
            userLotteryCacheManager.incrementUserLotteryNum(activityUserParam.getUserId());
            //增加总的抽奖次数缓存
            userTotalLotteryCacheManager.incrementUserLotteryNum(activityUserParam.getUserId());
        }

        return realPhaseInfo;
    }


    /**
     * 构建参加活动埋点参数
     *
     * @param activityUserParam 增加碎片参数
     * @param type              1：直接领取 2：看视频领取
     * @return 埋点参数
     */
    private PointParam buildJoinActivityPointParam(ActivityUserParam activityUserParam, Integer type) {
        return PointParam.build(activityUserParam)
                .e("3034")
                .ev("107." + activityUserParam.getActivityAwardId() + "." + type)
                .put("uid", String.valueOf(activityUserParam.getUserId()));
    }

    /**
     * 构建增加碎片埋点参数
     *
     * @param activityUserParam 增加碎片参数
     * @param type              1：直接领取 2：看视频领取
     * @return 埋点参数
     */
    private PointParam buildAddDebrisPointParam(ActivityUserParam activityUserParam, Integer type) {
        return PointParam.build(activityUserParam)
                .e("3034")
                .ev("108." + activityUserParam.getActivityAwardId() + "." + type)
                .put("uid", String.valueOf(activityUserParam.getUserId()));
    }

    /**
     * 构建兑换碎片埋点参数
     *
     * @param activityExchangeParam 兑换碎片参数
     * @return 埋点参数
     */
    private PointParam buildConversionDebrisPointParam(ActivityExchangeParam activityExchangeParam) {
        return PointParam.build(activityExchangeParam)
                .e("3034")
                .ev("109." + activityExchangeParam.getActivityAwardId())
                .put("uid", String.valueOf(activityExchangeParam.getUserId()));
    }
}