package com.bxm.fossicker.activity.service.debris.impl;

import com.bxm.fossicker.activity.constants.DistributedLockContant;
import com.bxm.fossicker.activity.domain.ActivityTelephoneChargeMapper;
import com.bxm.fossicker.activity.domain.debris.ActivityRedPacketMapper;
import com.bxm.fossicker.activity.domain.debris.ActivityRedPacketRecordMapper;
import com.bxm.fossicker.activity.enums.DebrisRedPacketStatusEnum;
import com.bxm.fossicker.activity.model.dto.debris.DebrisRedPacketDTO;
import com.bxm.fossicker.activity.model.entity.ActivityRedPacket;
import com.bxm.fossicker.activity.model.entity.ActivityRedPacketRecord;
import com.bxm.fossicker.activity.model.param.debris.ReceiveRedPacketParam;
import com.bxm.fossicker.activity.model.vo.debris.ReceiveRedPacketResultVo;
import com.bxm.fossicker.activity.model.vo.debris.RedPacketVo;
import com.bxm.fossicker.activity.service.debris.ActivityRedPacketService;
import com.bxm.fossicker.activity.service.debris.cache.UserLotteryCacheManager;
import com.bxm.fossicker.activity.service.debris.cache.UserTotalLotteryCacheManager;
import com.bxm.fossicker.enums.UserGoldFlowTypeEnum;
import com.bxm.fossicker.user.facade.AccountFacadeService;
import com.bxm.fossicker.user.facade.param.GoldRebateParam;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import lombok.AllArgsConstructor;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * 活动红包服务实现类
 *
 * @author wzy
 * @version 1.0
 * @date 2021/5/18 10:02 上午
 */
@Service
@AllArgsConstructor
public class ActivityRedPacketServiceImpl implements ActivityRedPacketService {

    private final ActivityRedPacketMapper activityRedPacketMapper;

    private final ActivityRedPacketRecordMapper activityRedPacketRecordMapper;

    private final UserLotteryCacheManager userLotteryCacheManager;

    private final SequenceCreater sequenceCreater;

    private final DistributedLock distributedLock;

    private final ActivityTelephoneChargeMapper activityTelephoneChargeMapper;

    private final AccountFacadeService accountFacadeService;

    private final UserTotalLotteryCacheManager userTotalLotteryCacheManager;

    @Override
    public List<RedPacketVo> getDebrisRedPacketList(Long userId) {
        List<DebrisRedPacketDTO> allRedPacketList = activityRedPacketMapper.getAllList(userId);

        if (CollectionUtils.isEmpty(allRedPacketList)) {
            return Collections.emptyList();
        }

        DebrisRedPacketDTO lastRedPacketDTO = allRedPacketList.get(allRedPacketList.size() - 1);
        //最高目标抽奖次数
        Integer maxTargetLotteryNum = lastRedPacketDTO.getTargetLotteryNum();
        //当前用户抽奖次数
        Integer currentUserLotteryNum = userLotteryCacheManager.getUserLotteryNum(userId);
        //结果列表
        List<RedPacketVo> resultList = new ArrayList<>();
        //标志后面的红包是否设置锁定状态
        boolean lastReceive = true;
        for (DebrisRedPacketDTO debrisRedPacketDTO : allRedPacketList) {
            RedPacketVo redPacketVo = new RedPacketVo();
            redPacketVo.setRedPacketId(debrisRedPacketDTO.getId());
            //如果没有超过最高红包的目标抽奖次数，则显示当前抽奖次数，超过了则显示最高的（20）
            redPacketVo.setCurrentLotteryNum(currentUserLotteryNum > maxTargetLotteryNum ? maxTargetLotteryNum : currentUserLotteryNum);
            redPacketVo.setTargetLotteryNum(debrisRedPacketDTO.getTargetLotteryNum());

            //上一个没有领取的话，那么直接锁定
            if (!lastReceive) {
                redPacketVo.setStatus(DebrisRedPacketStatusEnum.WAIT_LOCK.getCode());
            } else {
                //如果上一个已经领取，则判断当前是否已经领取
                if (Boolean.FALSE.equals(Objects.isNull(debrisRedPacketDTO.getRecordId()))) {
                    //已领取
                    redPacketVo.setStatus(DebrisRedPacketStatusEnum.RECEIVE.getCode());
                } else {
                    //如果未领取
                    lastReceive = false;
                    //如果当前小于目标
                    redPacketVo.setStatus(DebrisRedPacketStatusEnum.UNLOCK.getCode());
                }
            }

            resultList.add(redPacketVo);
        }

        return resultList;
    }

    @Override
    public Message receiveRedPacket(ReceiveRedPacketParam receiveRedPacketParam) {
        //加分布式锁
        String key = buildReceiveDebrisRedPacketKey(receiveRedPacketParam.getUserId(), receiveRedPacketParam.getRedPacketId());
        String requestId = sequenceCreater.nextStringId();

        if (!distributedLock.lock(key, requestId, 2, TimeUnit.SECONDS)) {
            //分布式锁解锁
            distributedLock.unlock(key, requestId);
            return Message.build(false, "领取红包操作过于频繁");
        }
        ActivityRedPacket activityRedPacket = activityRedPacketMapper.selectByPrimaryKey(receiveRedPacketParam.getRedPacketId());

        if (Objects.isNull(activityRedPacket)) {
            //分布式锁解锁
            distributedLock.unlock(key, requestId);
            return Message.build(false).setMessage("红包不存在");
        }
        //首先查询今天是否已经兑换过了，如果兑换过，则返回已经兑换
        List<ActivityRedPacketRecord> todayRedPacketRecord =
                activityRedPacketRecordMapper.getTodayRecordByRedPacketId(receiveRedPacketParam.getUserId(),
                        receiveRedPacketParam.getRedPacketId());

        //如果不是空，则返回错误信息
        if (Boolean.FALSE.equals(CollectionUtils.isEmpty(todayRedPacketRecord))) {
            //分布式锁解锁
            distributedLock.unlock(key, requestId);
            return Message.build(false).setMessage("今日您已领取该红包");
        }

        //判断是否满足兑换的条件
        Integer targetLotteryNum = activityRedPacket.getTargetLotteryNum();
        //如果不满足条件则返回领取失败
        if (targetLotteryNum > userLotteryCacheManager.getUserLotteryNum(receiveRedPacketParam.getUserId())) {
            //分布式锁解锁
            distributedLock.unlock(key, requestId);
            return Message.build(true).addParam("resultVO", ReceiveRedPacketResultVo.builder().goldCoinNum(activityRedPacket.getGoldCoinNum())
                    .hasReceive(false)
                    .redPacketVoList(getDebrisRedPacketList(receiveRedPacketParam.getUserId())).build());
        } else {
            //插入领取记录
            ActivityRedPacketRecord redPacketRecord = new ActivityRedPacketRecord();
            redPacketRecord.setCreateTime(new Date());
            redPacketRecord.setId(sequenceCreater.nextLongId());
            redPacketRecord.setUserId(receiveRedPacketParam.getUserId());
            redPacketRecord.setRedPacketId(receiveRedPacketParam.getRedPacketId());

            activityRedPacketRecordMapper.insertSelective(redPacketRecord);
            //增加用户金币
            accountFacadeService.goldRebate(GoldRebateParam.builder().amount(BigDecimal.valueOf(activityRedPacket.getGoldCoinNum()))
                    .userGoldFlowType(UserGoldFlowTypeEnum.DEBRIS_RED_PACKET)
                    .relationId(activityRedPacket.getId())
                    .userId(receiveRedPacketParam.getUserId())
                    .build());

            //分布式锁解锁
            distributedLock.unlock(key, requestId);
            return Message.build(true).addParam("resultVO", ReceiveRedPacketResultVo.builder().goldCoinNum(activityRedPacket.getGoldCoinNum())
                    .hasReceive(true)
                    .redPacketVoList(getDebrisRedPacketList(receiveRedPacketParam.getUserId())).build());
        }

    }

    @Override
    public Boolean hasReceiveCharge(Long userId) {
        //3表示的是打卡方式获得
        return activityTelephoneChargeMapper.countByUserIdAndSource(userId, 3) > 0;
    }

    @Override
    public void increaseLotteryNum(Long userId) {
        userLotteryCacheManager.incrementUserLotteryNum(userId);
        //增加总的抽奖次数
        userTotalLotteryCacheManager.incrementUserLotteryNum(userId);
    }

    /**
     * 构建领取碎片分布式锁
     *
     * @param userId      用户id
     * @param redPacketId 红包id
     * @return 分布式锁key
     */
    private String buildReceiveDebrisRedPacketKey(Long userId, Long redPacketId) {
        return DistributedLockContant.RECEIVE_DEBRIS_RED_PACKET_KEY.copy().appendKey(userId).appendKey(redPacketId).gen();
    }
}