package com.bxm.fossicker.activity.service.invite.impl;

import com.bxm.fossicker.activity.domain.UserInviteRelationMapper;
import com.bxm.fossicker.activity.facade.enums.InviteTypeEnum;
import com.bxm.fossicker.activity.facade.model.UserInviteRelationDTO;
import com.bxm.fossicker.activity.model.constant.ActivityRedisKey;
import com.bxm.fossicker.activity.model.param.FriendsListParam;
import com.bxm.fossicker.activity.model.vo.UserInviteRelation;
import com.bxm.fossicker.activity.model.vo.UserInvitedVo;
import com.bxm.fossicker.activity.service.WithdrawService;
import com.bxm.fossicker.activity.service.config.ActivityProperties;
import com.bxm.fossicker.activity.service.invite.InviteRelationService;
import com.bxm.fossicker.activity.service.task.TaskService;
import com.bxm.fossicker.enums.TaskEnum;
import com.bxm.fossicker.order.integration.PushMessageIntegrationService;
import com.bxm.fossicker.user.facade.AccountFacadeService;
import com.bxm.fossicker.user.facade.UserInfoFacadeService;
import com.bxm.fossicker.user.facade.dto.UserInfoDto;
import com.bxm.fossicker.user.facade.enums.CashTypeDetail;
import com.bxm.fossicker.user.facade.param.CashIncrementParam;
import com.bxm.fossicker.user.facade.param.TobeVipParam;
import com.bxm.fossicker.user.facade.vip.VipFacadeService;
import com.alibaba.fastjson.JSON;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import com.fasterxml.jackson.core.type.TypeReference;
import com.github.pagehelper.Page;
import com.github.pagehelper.PageHelper;
import lombok.extern.log4j.Log4j2;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @ClassName InviteRelationServiceImpl
 * @Author XinZhao
 * @Date 2019/7/1 17:35
 * @Version 1.0.0
 **/
@Log4j2
@Service
public class InviteRelationServiceImpl implements InviteRelationService {

    @Autowired
    private SequenceCreater sequenceCreater;

    @Autowired
    private UserInviteRelationMapper userInviteRelationMapper;

    @Autowired
    private UserInfoFacadeService userInfoFacadeService;

    @Autowired
    private AccountFacadeService accountFacadeService;

    @Autowired
    private ActivityProperties activityProperties;

    @Autowired
    private VipFacadeService vipFacadeService;

    @Autowired
    private RedisHashMapAdapter redisHashMapAdapter;

    @Autowired
    private PushMessageIntegrationService pushMessageIntegrationService;

    @Autowired
    private WithdrawService withdrawService;

    /**
     * 不要直接用
     */

    private TaskService taskService;

    private TaskService getTaskService() {
        if (Objects.isNull(taskService)) {
            taskService = SpringContextHolder.getBean(TaskService.class);
        }
        return taskService;
    }

    @Override
    public Boolean hasInviteRelation(Long userId) {
        return userInviteRelationMapper.countByInvitedId(userId) > 0;
    }

    @Override
    public Boolean addRelation(Long userId, Long invitedUserId) {
        // 被邀请人信息
        UserInfoDto userInfo = userInfoFacadeService.getUserById(invitedUserId);
        return null != userInfo
                // 用户信息中设置上级和顶级用户id
                && userInfoFacadeService.setSuperior(invitedUserId, userId)
                && userInviteRelationMapper.addOne(build(userId, userInfo, true));
    }

    @Override
    public Boolean addRelationByPhone(Long userId, String phoneNo) {
        UserInfoDto invitedUserInfo = userInfoFacadeService.getUserForPhone(phoneNo);
        if (null == invitedUserInfo) {
            log.error("建立邀请关系(手机号),查询被邀请者用户信息为空,手机号为:{}", phoneNo);
            return false;
        }
        //若已有邀请关系-不重复添加
        if (null != userInviteRelationMapper.selectByInvitedUserId(invitedUserInfo.getId())) {
            log.error("建立邀请关系(手机号),已有邀请关系,手机号为:{}", phoneNo);
            return false;
        }
        //建立预绑定关系
        userInviteRelationMapper.addOne(build(userId, invitedUserInfo, false));
        return true;
    }

    @Override
    @Transactional
    public Boolean bindRelation(Long userId, Long superiorUserId) {
        UserInfoDto userInfo = userInfoFacadeService.getUserById(userId);
        //绑定关系
        return this.hasInviteRelation(userId) ? Boolean.FALSE : userInviteRelationMapper.addOne(build(superiorUserId, userInfo, true));
    }

    @Override
    @Transactional
    public Boolean bindSuccessRelation(Long userId, Long superiorUserId, String currVer) {
        log.info("绑定邀请关系: userId: {} superiorUserId: {}", userId, superiorUserId);

        if (Objects.isNull(userId) || Objects.isNull(superiorUserId)
                || 0 == userId || 0 == superiorUserId) {
            log.warn("用户id不能为空:{},{}", userId, superiorUserId);
            return false;
        }

        UserInfoDto userInfo = userInfoFacadeService.getUserById(userId);
        if (hasInviteRelation(userId)) {
            log.warn("用户:{}已有邀请关系", userId);
            return false;
        }
        //是否有预绑定关系-有就更新,没有就新增
        if (Objects.isNull(userInviteRelationMapper.selectByInvitedIdAndStatus(userId))) {
            // 没有就新增 - 新增一条未完全绑定的邀请
            userInviteRelationMapper.addOne(build(superiorUserId, userInfo, false));
        }
        //用户信息中设置上级关系
        userInfoFacadeService.setSuperior(userId, superiorUserId);

        //推送一条好友邀请成功弹窗
        pushPopUpInfo(superiorUserId);

        // 给师傅推送一条好友成功接受邀请信息
        pushMessageIntegrationService.pushHasFriendAcceptInvite(superiorUserId);

        //邀请一个好友获得一次提现机会 1.4.1版本需求
        //1.6.0 修改为邀请用户购买黑钻卡后获取一次提现机会
        //withdrawService.add(superiorUserId, ActivityWithdrawType.INVITE);

        // 完成任务
        getTaskService().taskComplete(userId, TaskEnum.INPUT_INVITE.name());

        //2.2.2版本邀请人需要单独计算
        if (com.bxm.newidea.component.tools.StringUtils.isGrateOrEqualThan(currVer, "2.2.2")) {
            redisHashMapAdapter.increment(getNewVersionInviteUserKey(), userId.toString(), 1);
        }

        return true;
    }

    /**
     * 2.2.2版本邀请人数量需要单独计算 作为提现使用
     *
     * @return 邀请好友成功数key
     */
    private KeyGenerator getNewVersionInviteUserKey() {
        return ActivityRedisKey.NEW_VERSION_INVITE_USER_KEY;
    }

    @Override
    public List<UserInvitedVo> getInvitedList(Long userId) {
        return userInviteRelationMapper.queryInvitedByUserId(userId);
    }

    @Override
    public Boolean updateRemind(Long userId, Long invitedId) {
        return userInviteRelationMapper.updateRemindStatus(userId, invitedId) > 0;
    }

    @Override
    @Async
    public void initRemindStatus() {
        userInviteRelationMapper.initRemindStatus();
    }

    @Override
    public Integer getFriendsNum(Long userId) {
        return userInviteRelationMapper.selectFriendsNumByUserId(userId);
    }

    @Override
    public BigDecimal getEarning(Long userId) {
        return userInviteRelationMapper.getEarning(userId);
    }

    @Override
    public Integer getFriendsNumByVersion(Long userId) {
        return userInviteRelationMapper.selectFriendsNumByUserIdAndVersion(userId);
    }

    @Override
    public List<UserInviteRelation> getFriendsList(Long userId) {
        return userInviteRelationMapper.getFriendsListInfo(userId);
    }

    @Override
    public Page<UserInviteRelation> getAllFriendsList(FriendsListParam friendsListParam) {
        return PageHelper.startPage(friendsListParam).doSelectPage(() ->
                userInviteRelationMapper.getAllFriendsListInfo(friendsListParam));
    }

    @Override
    public Boolean rewardOldFriends() {
        //去重后的待推送的用户id以及好友数

        List<UserInviteRelation> oldRelationList = userInviteRelationMapper.selectOldFriends();
        if (CollectionUtils.isEmpty(oldRelationList)) {
            log.error("奖励老用户的名单为空");
            return false;
        }
        Map<Long, Integer> pushUserIdMap = new HashMap<>(oldRelationList.size());

        oldRelationList.forEach(userInviteRelation -> {
            Long userId = userInviteRelation.getUserId();
            rewardOldRelation(userInviteRelation);
            //计算一个用户的好友数
            Integer friendsNum = pushUserIdMap.get(userId);
            if (Objects.isNull(friendsNum)) {
                pushUserIdMap.put(userId, 1);
            } else {
                pushUserIdMap.put(userId, friendsNum + 1);
            }
        });

        //推送推送消息
        pushUserIdMap.forEach((key, value) -> {
            try {
                pushMessageIntegrationService.pushNewVersionPush(key, value);
            } catch (Exception e) {
                log.error("推送数据异常:{}", e);
            }

        });

        return true;

    }

    @Override
    public Integer getSuccessInvitationInfo(Long userId) {
        if (Objects.isNull(userId)) {
            log.error("获取成功邀请好友数,用户id为null");
        }
        KeyGenerator inviteUserMapKey = getInviteUserKey();

        //新版本重新记录邀请人数
        getNewVersionSuccessInviteNum(userId);

        return redisHashMapAdapter.getLong(inviteUserMapKey, userId.toString()).intValue();
    }

    @Override
    public Boolean removeSuccessInvitation(Long userId) {
        if (Objects.isNull(userId)) {
            log.error("删除成功邀请好友数,用户id为null");
        }
        KeyGenerator inviteUserMapKey = getInviteUserKey();

        //新版本删除成功邀请好友数
        removeNewVersionInviteNum(userId);

        redisHashMapAdapter.remove(inviteUserMapKey, userId.toString());
        return true;
    }

    @Override
    public Boolean addOldRelation() {
        List<UserInfoDto> oldRelationUserList = userInfoFacadeService.getOldRelationUser();
        if (CollectionUtils.isEmpty(oldRelationUserList)) {
            return false;
        }

        oldRelationUserList.forEach(this::addOldRelation);

        return true;
    }

    @Override
    public Boolean updateRewardVip(Long userId, Long invitedId, BigDecimal rewardAmt) {
        userInviteRelationMapper.rewardVip(userId, invitedId, rewardAmt, null);
        return true;
    }

    @Override
    public Boolean updateGoldRewardVip(Long userId, Long invitedId, Integer rewardAmt) {
        userInviteRelationMapper.rewardVip(userId, invitedId, null, rewardAmt);
        return true;
    }

    /**
     * 绑定邀请关系
     *
     * @param userId   邀请者用户id
     * @param userInfo 被邀请者用户信息
     * @param sucBind  是否是成功绑定关系(是否登录),还是只是邀请
     * @return
     */
    private UserInviteRelation build(Long userId, UserInfoDto userInfo, boolean sucBind) {
        Date current = new Date();
        return UserInviteRelation.builder()
                .id(sequenceCreater.nextLongId())
                .userId(userId)
                .invitedPhone(userInfo.getPhoneno())
                .status(sucBind ? 1 : 0)
                .invitedId(userInfo.getId())
                .invitedName(userInfo.getNickName())
                .invitedHeadImg(userInfo.getHeadImg())
                .remindStatus(0)
                .inviteVersion(1)
                .createTime(current)
                .modifyTime(current)
                .inviteSuccessTime(current)
                .build();
    }

    @Override
    public Message inviteFinalSuccess(Long invitedId, Long userId) {
        log.info("下发邀请奖励 邀请人: [{}] 被邀请人: [{}]", userId, invitedId);

        // 更新状态
        userInviteRelationMapper.updateBindStatus(userId, invitedId);

        // 奖励登录
        rewardInviteForLogin(userId, invitedId);

        // 奖励vip卡
        rewardInviteForVip(userId, invitedId);

        return Message.build();
    }

    @Override
    public Message updateInviteType(Long userId, Long invitedId, InviteTypeEnum inviteTypeEnum) {
        userInviteRelationMapper.updateInviteType(userId, invitedId, inviteTypeEnum.name());
        return Message.build(true);
    }

    @Override
    public UserInviteRelationDTO getFirstSuccessCashInviteRelationInfo(Long userId) {
        return convert(userInviteRelationMapper.getFirstSuccessCashInviteRelationInfo(userId));
    }

    private UserInviteRelationDTO convert(UserInviteRelation relation) {
        if (Objects.isNull(relation)) {
            return null;
        }

        UserInviteRelationDTO relationDTO = new UserInviteRelationDTO();
        BeanUtils.copyProperties(relation, relationDTO);
        return relationDTO;
    }

    @Override
    public Message inviteFinalSuccessForV2(Long invitedId, Long userId) {
        log.info("下发邀请奖励 邀请人: [{}] 被邀请人: [{}]", userId, invitedId);

        // 更新状态
        userInviteRelationMapper.updateBindStatus(userId, invitedId);

        return Message.build();
    }

    /**
     * 奖励登录的好友邀请-1元
     *
     * @param userId    用户id
     * @param invitedId 被邀请人id
     * @return 请求结果
     */
    private boolean rewardInviteForLogin(Long userId, Long invitedId) {
        KeyGenerator loginRewardKey = ActivityRedisKey.HAVA_REWARD_LOGIN_INVITE;
        TypeReference<List<Long>> typeReference = new TypeReference<List<Long>>() {
        };

        List<Long> invitedIdList = redisHashMapAdapter.get(loginRewardKey, userId.toString(), typeReference);
        if (!CollectionUtils.isEmpty(invitedIdList) && invitedIdList.contains(invitedId)) {
            log.warn("该用户:{}已经获取此用户:{} 的邀请奖励,不再发放奖励", userId, invitedId);
            return false;
        }

        UserInviteRelation userInviteRelation = userInviteRelationMapper.selectByUserAndInvitedId(userId, invitedId);
        if (Objects.isNull(userInviteRelation)) {
            log.warn("未查询到俩用户的邀请关系,邀请人:{}, 被邀请人:{}", userId, invitedId);
            return false;
        }

        // 更新关系表中收益信息
        BigDecimal inviteUserReward = new BigDecimal(activityProperties.getInviteUserReward());
        userInviteRelationMapper.rewardLogin(userId, invitedId, inviteUserReward);

        // 增加邀请人奖励
        CashIncrementParam cashIncrementParam = new CashIncrementParam(userId, inviteUserReward,
                invitedId, CashTypeDetail.INVITE_LOGIN);
        accountFacadeService.cashIncrement(cashIncrementParam);

        // 奖励后添加到已奖励的集合列表中
        if (CollectionUtils.isEmpty(invitedIdList)) {
            //若为空则新建对象,否则add操作会报空指针
            invitedIdList = new ArrayList<>();
        }
        invitedIdList.add(invitedId);
        redisHashMapAdapter.put(loginRewardKey, userId.toString(), invitedIdList);

        return true;
    }

    /**
     * 奖励好友邀请-每成功邀请3名奖励一个vip
     *
     * @param userId 用户id
     * @return
     */
    private boolean rewardInviteForVip(Long userId, Long invitedId) {
        //三位好友的校验
        List<UserInviteRelation> rewardVipUserList = userInviteRelationMapper.getRewardVipUserList(userId);

        // 只有绑定完淘宝、微信之后才算是有效邀请
        rewardVipUserList = rewardVipUserList.stream()
                .filter(p -> {
                    UserInfoDto user = userInfoFacadeService.getUserById(p.getInvitedId());

                    if (Objects.nonNull(user)) {
                        return StringUtils.isNotBlank(user.getWechatNickName())
                                && StringUtils.isNotBlank(user.getTaobaoNickName());
                    }
                    return false;
                })
                .collect(Collectors.toList());

        //可奖励vip的好友list为空或者小于规定奖励vip好友数(3),则不奖励vip
        if (CollectionUtils.isEmpty(rewardVipUserList) || rewardVipUserList.size() < activityProperties.getInviteUserRewardNum()) {
            log.warn("用户:{}所邀请的好友数未达到规定奖励vip数,被邀请用户:{}", userId, invitedId);
            return false;
        }

        List<Long> invitedUserIdList = new ArrayList<>();
        rewardVipUserList.subList(0, 3).forEach(userInviteRelation -> {
            userInviteRelationMapper.updateConsumeStatusById(userInviteRelation.getId());
            invitedUserIdList.add(userInviteRelation.getInvitedId());
        });

        TobeVipParam tobeVipParam = new TobeVipParam();
        tobeVipParam.setUserId(userId);
        tobeVipParam.setType(1);
        tobeVipParam.setRelationId(JSON.toJSONString(invitedUserIdList));

        if (!vipFacadeService.tobeVip(tobeVipParam)) {
            log.warn("创建会员失败,创建信息为:{}", JSON.toJSONString(tobeVipParam));
            return false;
        }

        return true;
    }

    /**
     * 推送弹窗所需信息
     *
     * @param userId 用户id
     * @return
     */
    private boolean pushPopUpInfo(Long userId) {
        KeyGenerator inviteUserMapKey = getInviteUserKey();
        redisHashMapAdapter.increment(inviteUserMapKey, userId.toString(), 1);

        return true;
    }

    /**
     * 获取邀请好友成功数key-用于弹窗
     *
     * @return 邀请好友成功数key
     */
    private KeyGenerator getInviteUserKey() {
        return ActivityRedisKey.INVITE_USER_MAP;
    }

    /**
     * 用户老邀请关系奖励-登录奖励1元
     *
     * @param userInviteRelation 用户邀请邀请关系信息
     * @return 请求结果
     */
    private boolean rewardOldRelation(UserInviteRelation userInviteRelation) {
        Long userId = userInviteRelation.getUserId();
        Long invitedUserId = userInviteRelation.getInvitedId();

        if (Objects.isNull(userId) || Objects.isNull(invitedUserId)) {
            log.error("用户id不能为空:{},{}", userId, invitedUserId);
            return false;
        }

        rewardInviteForLogin(userId, invitedUserId);
        return true;
    }

    /**
     * 增加老用户的绑定关系
     *
     * @param invitedUserInfo 被邀请用户信息
     * @return 请求结果
     */
    private boolean addOldRelation(UserInfoDto invitedUserInfo) {

        if (null == invitedUserInfo) {
            log.error("增加老用户邀请关系,用户信息为空");
            return false;
        }
        //若已有邀请关系-不重复添加
        if (null != userInviteRelationMapper.selectByInvitedUserId(invitedUserInfo.getId())) {
            log.error("增加老用户邀请关系,已有邀请关系,用户id为:{}", invitedUserInfo.getId());
            return false;
        }
        //建立预绑定关系
        UserInviteRelation userInviteRelation = build(invitedUserInfo.getSuperiorUserId(), invitedUserInfo, false);
        userInviteRelation.setInviteVersion(0);
        userInviteRelationMapper.addOne(userInviteRelation);
        return true;
    }

    @Override
    public Integer getNewVersionSuccessInviteNum(Long userId) {
        return redisHashMapAdapter.getLong(getNewVersionInviteUserKey(), userId.toString()).intValue();
    }

    private void removeNewVersionInviteNum(Long userId) {
        redisHashMapAdapter.remove(getNewVersionInviteUserKey(), userId.toString());
    }
}
