package com.bxm.fossicker.activity.service.invite.impl;

import com.bxm.fossicker.activity.config.InviteConfig;
import com.bxm.fossicker.activity.config.TaskConfig;
import com.bxm.fossicker.activity.domain.UserAddressListMapper;
import com.bxm.fossicker.activity.enums.RemindStatusEnum;
import com.bxm.fossicker.activity.facade.InviteFacadeService;
import com.bxm.fossicker.activity.facade.model.UserInviteRelationDTO;
import com.bxm.fossicker.activity.model.dto.InviteListDto;
import com.bxm.fossicker.activity.model.dto.InviteResultDTO;
import com.bxm.fossicker.activity.model.dto.InviterDto;
import com.bxm.fossicker.activity.model.param.BindInviteListParam;
import com.bxm.fossicker.activity.model.param.BindInviteParam;
import com.bxm.fossicker.activity.model.param.RemindParam;
import com.bxm.fossicker.activity.model.vo.UserAddressList;
import com.bxm.fossicker.activity.model.vo.UserInvitedVo;
import com.bxm.fossicker.activity.service.UserAddressListService;
import com.bxm.fossicker.activity.service.invite.InviteRelationService;
import com.bxm.fossicker.activity.service.invite.InviteService;
import com.bxm.fossicker.integration.commodity.service.BrowseFlowFacadeService;
import com.bxm.fossicker.message.enums.SmsTempEnum;
import com.bxm.fossicker.message.facade.SmsFacadeService;
import com.bxm.fossicker.user.facade.UserInfoFacadeService;
import com.bxm.fossicker.user.facade.dto.InviteInfoDTO;
import com.bxm.fossicker.user.facade.dto.UserInfoDto;
import com.bxm.newidea.component.vo.Message;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @ClassName InviteServiceImpl
 * @CopyRright (c) 2019-bxm：杭州微财网络科技有限公司
 * @Author zhaoxin
 * @Date 2019/6/25 15:56
 * @Version 1.0.0
 **/
@Service
@Slf4j
public class InviteServiceImpl implements InviteService, InviteFacadeService {

    @Autowired
    private InviteRelationService inviteRelationService;

    @Autowired
    private UserInfoFacadeService userInfoFacadeService;

    @Autowired
    private BrowseFlowFacadeService browseFlowFacadeService;

    @Autowired
    private TaskConfig taskConfig;

    @Autowired
    private InviteConfig inviteConfig;

    @Autowired
    private UserAddressListService userAddressListService;

    @Autowired
    private UserAddressListMapper userAddressListMapper;

    @Autowired
    private SmsFacadeService smsFacadeService;


    @Override
    public String getInviteCodeByUserId(Long userId) {
        UserInfoDto userInfo = userInfoFacadeService.getUserById(userId);
        if (null == userInfo) {
            log.error("用户[{}]邀请码不存在", userId);
            return "";
        }
        return userInfo.getInviteCode();
    }

    @Override
    public InviteInfoDTO getInviteInfoByUserId(Long userId) {
        return userInfoFacadeService.getInviteInfoByUserId(userId);
    }


    @Override
    public Boolean bindRelation(BindInviteParam bindInviteParam) {

        if (Objects.isNull(bindInviteParam.getUserId()) || Objects.isNull(bindInviteParam.getInviteUserId())) {
            log.error("邀请码绑定关系,用户id不能为空,邀请人id:{}, 被邀请人id:{}", bindInviteParam.getInviteUserId(), bindInviteParam.getUserId());
            return Boolean.FALSE;
        }

        InviteResultDTO checkResult = checkInviteCode(bindInviteParam);
        if (null == checkResult || !checkResult.getSuccess()) {
            return false;
        }
        //绑定邀请关系
        Boolean res = inviteRelationService.bindSuccessRelation(bindInviteParam.getUserId(), bindInviteParam.getInviteUserId(), bindInviteParam.getCurVer());
        if (res) {
            // 绑定关系成功，判断用户是否已经绑定微信和淘宝
            UserInfoDto user = userInfoFacadeService.getUserById(bindInviteParam.getUserId());
            if (Objects.nonNull(user)) {
                // 用户的微信和淘宝已经绑定成功，发送奖励
                // inviteRelationService.inviteFinalSuccess(bindInviteParam.getUserId(), bindInviteParam.getInviteUserId());
                inviteRelationService.inviteFinalSuccessForV2(bindInviteParam.getUserId(), bindInviteParam.getInviteUserId());
            }
        }

        return res;
    }

    @Override
    public InviteResultDTO checkInviteCode(BindInviteParam bindInviteParam) {

        if (Objects.equals(inviteConfig.getInviteCodeSwitch(), Boolean.FALSE)) {
            return new InviteResultDTO("绑定失败");
        }

        //(填写邀请码)被邀请人id
        Long invitedUserId = bindInviteParam.getUserId();
        //邀请人id
        Long inviteUserId = bindInviteParam.getInviteUserId();

        if (Objects.isNull(invitedUserId)) {
            return new InviteResultDTO("用户id必填");
        }

        // 获取邀请者信息
        UserInfoDto userInfoDto;

        //若有邀请者用户id 则根据用户id查询 否则以手机号或邀请码查询
        if (Objects.isNull(inviteUserId)) {
            userInfoDto = userInfoFacadeService.getUserByInviteCodeOrPhone(bindInviteParam.getInputCode());
        } else {
            userInfoDto = userInfoFacadeService.getUserById(inviteUserId);
        }

        // 邀请者信息不存在
        if (Objects.isNull(userInfoDto) || Objects.isNull(userInfoDto.getId())) {
            log.warn("用户[{}]校验邀请关系失败,该邀请码无效", bindInviteParam.getUserId());
            return new InviteResultDTO("邀请码无效，请重新输入");
        }

        // 是否选择的自己
        if (Objects.equals(invitedUserId, userInfoDto.getId())) {
            return new InviteResultDTO("不可以成为自己的好友");
        }

        //邀请人不可为好友，请重新输入
        if (invitedUserId.equals(userInfoDto.getSuperiorUserId()) || invitedUserId.equals(userInfoDto.getTopUserId())) {
            log.warn("用户[{}]邀请的邀请人不可为好友，请重新输入", invitedUserId);
            return new InviteResultDTO("邀请人不可为好友，请重新输入");
        }

        //获取(填写邀请码)被邀请人信息
        UserInfoDto invitedUserInfoDTO = userInfoFacadeService.getUserById(invitedUserId);
        if (Objects.isNull(invitedUserInfoDTO) || Objects.isNull(invitedUserInfoDTO.getId())) {
            return new InviteResultDTO("无被邀请人信息");
        }

        // 1.6.0移除设备校验，因为这个版本开始单单绑定是不会返现的，只有在绑定后且支付会员才会

        // 被邀请人是否已有绑定的邀请关系
//        if (inviteRelationService.hasInviteRelation(bindInviteParam.getUserId())) {
//            return Boolean.FALSE;
//        }

        //(填写邀请码)被邀请人已有邀请人
        if (Objects.nonNull(invitedUserInfoDTO.getSuperiorUserId())) {
            return new InviteResultDTO("你已有邀请人，不可输入");
        }

        //成功返回用户id、头像以及昵称
        return new InviteResultDTO(userInfoDto.getId(), userInfoDto.getNickName(), userInfoDto.getHeadImg());
    }

    @Override
    public Message rewardRemedy(String userIds) {

        if (StringUtils.isBlank(userIds)) {
            return Message.build(false).setMessage("参数有误");
        }

        Arrays.asList(userIds.split(",")).forEach(p -> {
            long id = Long.parseLong(p);
            // 查询用户
            UserInfoDto user = userInfoFacadeService.getUserById(id);

            if (Objects.nonNull(user) && Objects.nonNull(user.getSuperiorUserId())) {
                // 发送奖励
                // inviteRelationService.inviteFinalSuccess(user.getId(), user.getSuperiorUserId());
                inviteRelationService.inviteFinalSuccessForV2(user.getId(), user.getSuperiorUserId());
            }
        });

        return Message.build();
    }

    @Override
    public boolean judgeVipFriendCashReward(Long userId) {
        return userInfoFacadeService.judgeVipFriendCashReward(userId);
    }

    @Override
    public InviteListDto getInvitedList(Long userId) {
        if (log.isDebugEnabled()) {
            log.debug("开始获取用户[{}]邀请成功的好友信息", userId);
        }

        List<UserInvitedVo> userInvitedVoList = inviteRelationService.getInvitedList(userId);
        if (CollectionUtils.isEmpty(userInvitedVoList)) {
            return convertToEmptyDto();
        }

        List<Long> userIdList = userInvitedVoList.stream().map(UserInvitedVo::getUserId).collect(Collectors.toList());
        Map<Long, Integer> totalBrowseMap = browseFlowFacadeService.getTotalBrowseMap(userIdList);
        // 获取用户浏览商品统计列表
        return convert(userInvitedVoList, totalBrowseMap);
    }

    /**
     * @param userInvitedVoList 邀请成功的好友列表
     * @param browseCountVoMap  浏览记录数map key:userId,value:总商品浏览记录数
     */
    private InviteListDto convert(List<UserInvitedVo> userInvitedVoList, Map<Long, Integer> browseCountVoMap) {
        List<InviterDto> inviterDtoList = Lists.newArrayList();
        userInvitedVoList.forEach(userInvitedVo -> inviterDtoList.add(convert(browseCountVoMap, userInvitedVo)));

        return InviteListDto.builder()
                .inviteCount(userInvitedVoList.size())
                .totalAmount(String.valueOf(userInvitedVoList.size() * inviteConfig.getReward()))
                .inviteDtoList(inviterDtoList)
                .build();
    }

    private InviterDto convert(Map<Long, Integer> browseCountVoMap, UserInvitedVo userInvitedVo) {
        // 获取总浏览数、贡献金额
        Integer totalBrowseCount = null == browseCountVoMap || null == browseCountVoMap.get(userInvitedVo.getUserId()) ? 0 : browseCountVoMap.get(userInvitedVo.getUserId());
        return InviterDto.builder()
                .userId(userInvitedVo.getUserId())
                .phone(new StringBuffer(userInvitedVo.getInvitedPhone()).replace(3, 7, "****").toString())
                .browseCount(totalBrowseCount)
                .contributeGold(calculateContributeGold(totalBrowseCount))
                .remind(userInvitedVo.getRemindStatus().equals(RemindStatusEnum.REMINDED.getCode()))
                .build();
    }

    /**
     * 用于计算 邀请页面已邀请好友列表的贡献金币
     * <p>
     * 根据商品浏览量分段字段贡献金币，金币为配置值
     * 0~500次 5400金/次    一阶段金币奖励
     * 501~1500次 3600金/次 二阶段金币奖励
     * 1501及以上 1800金/次  三阶段金币奖励
     *
     * @param totalBrowseCount 总浏览量
     * @return 贡献金币
     */
    private Integer calculateContributeGold(Integer totalBrowseCount) {
        if (totalBrowseCount >= 0 && totalBrowseCount <= 100) {
            return totalBrowseCount * taskConfig.getGoodsViewReward().intValue() * 8;
        }
        if (totalBrowseCount > 100 && totalBrowseCount <= 400) {
            return 100 * taskConfig.getGoodsViewReward().intValue() * 8 +
                    (totalBrowseCount - 100) * taskConfig.getGoodsViewReward().intValue() * 3;
        }
        if (totalBrowseCount > 400 && totalBrowseCount <= 900) {
            return 100 * taskConfig.getGoodsViewReward().intValue() * 8 +
                    300 * taskConfig.getGoodsViewReward().intValue() * 3 +
                    (totalBrowseCount - 400) * taskConfig.getGoodsViewReward().intValue() * 2;
        }
        return 100 * taskConfig.getGoodsViewReward().intValue() * 8 +
                300 * taskConfig.getGoodsViewReward().intValue() * 3 +
                500 * taskConfig.getGoodsViewReward().intValue() * 2 +
                (totalBrowseCount - 900) * taskConfig.getGoodsViewReward().intValue();
    }

    private InviteListDto convertToEmptyDto() {
        return InviteListDto.builder()
                .inviteCount(0)
                .totalAmount("0")
                .inviteDtoList(Lists.newArrayList())
                .build();
    }

    @Override
    public Boolean remind(RemindParam remindParam) {
        if (log.isDebugEnabled()) {
            log.debug("用户[{}]提醒被邀请用户[{}]浏览商品", remindParam.getUserId(), remindParam.getInvitedUserId());
        }

        if (inviteRelationService.updateRemind(remindParam.getUserId(), remindParam.getInvitedUserId())) {
            // 调用短信服务， 类型为提醒
            UserInfoDto userInfoDto = userInfoFacadeService.getUserById(remindParam.getInvitedUserId());

            if (null != userInfoDto) {
                smsFacadeService.sendSms(userInfoDto.getPhoneno(), SmsTempEnum.REIMNDER_SMS, new Object[0]);
                return Boolean.TRUE;
            }
        }

        return Boolean.FALSE;
    }


    @Override
    public Message invite(BindInviteListParam bindInviteListParam) {
        if (log.isDebugEnabled()) {
            log.debug("用户发送邀请短信,data:[{}]", bindInviteListParam);
        }

        UserInfoDto userInfoDto = userInfoFacadeService.getUserByInviteCodeOrPhone(bindInviteListParam.getPhone());

        if (null != userInfoDto) {
            log.error("手机号: [{}] 已注册用户，无法邀请", bindInviteListParam.getPhone());

            return Message.build(false).setMessage("该号码已注册");
        }

        // 从通讯录邀请中选择的邀请用户则用id做邀请
        if (null != bindInviteListParam.getId()) {
            if (userAddressListService.invited(bindInviteListParam.getId())) {
                UserAddressList userAddressList = userAddressListMapper.selectByPrimaryKey(bindInviteListParam.getId());

                if (userAddressList != null) {
                    bindInviteListParam.setPhone(userAddressList.getPhone());
                    return sendInviteSms(bindInviteListParam);
                }
            } else {
                return Message.build(false).setMessage("邀请失败，请勿重复邀请");
            }
        } else if (StringUtils.isNotBlank(bindInviteListParam.getPhone())) {
            // 用户输入邀请号码则用手机号做邀请
            if (userAddressListService.invited(bindInviteListParam.getUserId(), bindInviteListParam.getPhone())) {
                return sendInviteSms(bindInviteListParam);
            } else {
                return Message.build(false).setMessage("邀请失败，请勿重复邀请");
            }
        }
        return Message.build(false).setMessage("邀请失败");
    }

    private Message sendInviteSms(BindInviteListParam bindInviteListParam) {
        return smsFacadeService.sendSms(bindInviteListParam.getPhone(), SmsTempEnum.RELATIVE_INVITE, bindInviteListParam.getUserId());
    }

    @Override
    public List<String> scrollBar() {
        return inviteConfig.getMessage();
    }

    @Override
    public Boolean addRelationBySms(Long userId, String phone) {
        if (Objects.isNull(userId)) {
            log.info("邀请者用户id为空,用户手机号为:{}", phone);
            return false;
        }
        return inviteRelationService.addRelationByPhone(userId, phone);
    }

    @Override
    public UserInviteRelationDTO getFirstSuccessCashInviteRelationInfo(Long userId) {
        return inviteRelationService.getFirstSuccessCashInviteRelationInfo(userId);
    }


}
