package com.bxm.fossicker.activity.service.sign.impl;

import com.bxm.fossicker.activity.domain.ActivitySignLogMapper;
import com.bxm.fossicker.activity.facade.SignFacadeService;
import com.bxm.fossicker.activity.facade.model.SignFacadeDto;
import com.bxm.fossicker.activity.model.DayRewardModel;
import com.bxm.fossicker.activity.model.dto.SignAndInfoDTO;
import com.bxm.fossicker.activity.model.dto.SignDaysDto;
import com.bxm.fossicker.activity.model.dto.SignDto;
import com.bxm.fossicker.activity.model.dto.SignTodayDTO;
import com.bxm.fossicker.activity.model.vo.ActivitySignLog;
import com.bxm.fossicker.activity.service.AccountIntegrationService;
import com.bxm.fossicker.activity.service.sign.ActivitySignService;
import com.bxm.fossicker.activity.service.sign.SignService;
import com.bxm.fossicker.activity.service.config.SignConfig;
import com.bxm.fossicker.user.facade.UserActivitySignFacadeService;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.google.common.collect.Lists;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.*;

/**
 * 签到服务
 *
 * @ClassName SignServiceImpl
 * @Author XinZhao
 * @Date 2019/7/3 11:10
 * @Version 1.0.0
 **/
@Log4j2
@Service
public class SignServiceImpl implements SignService, SignFacadeService {

    private final ActivitySignLogMapper signLogMapper;

    private final SequenceCreater sequenceCreater;

    private final SignConfig signConfig;

    private final AccountIntegrationService accountIntegrationService;


    private ActivitySignLogMapper activitySignLogMapper;

    private ActivitySignService activitySignService;

    private UserActivitySignFacadeService userActivitySignFacadeService;


    @Autowired
    public SignServiceImpl(ActivitySignLogMapper signLogMapper, SequenceCreater sequenceCreater, SignConfig signConfig,
                           AccountIntegrationService accountIntegrationService,
                           ActivitySignLogMapper activitySignLogMapper, ActivitySignService activitySignService,
                           UserActivitySignFacadeService userActivitySignFacadeService) {
        this.signLogMapper = signLogMapper;
        this.sequenceCreater = sequenceCreater;
        this.signConfig = signConfig;
        this.accountIntegrationService = accountIntegrationService;
        this.activitySignLogMapper = activitySignLogMapper;
        this.activitySignService = activitySignService;
        this.userActivitySignFacadeService = userActivitySignFacadeService;
    }

    @Override
    @Transactional
    public SignDto signInfo(Long userId) {
        if (null == userId) {
            return null;
        }

        // 连续签到天数
        Integer signedDays = getSignedDays(userId);
        // 组装返回对象
        return convertToSignDto(signConfig.getDayReward(), signedDays);
    }

    @Override
    public SignDto signInfoSeven(Long userId) {
        if (null == userId) {
            return null;
        }

        // 连续签到天数
        Integer signedDays = getSignedDays(userId);
        // 组装返回对象
        return convertToSignDtoSevenDay(signConfig.getDayReward(), signedDays, userId);
    }

    @Override
    public Integer getSignReward(Long userId) {
        if (null == userId) {
            return null;
        }

        // 连续签到天数
        Integer signedDays = getSignedDays(userId);
        //获取奖励
        DayRewardModel dayRewardModel = signConfig.getDayReward().get(signedDays);

        return dayRewardModel.getReward() + dayRewardModel.getExtraReward();
    }

    @Override
    public Boolean didSign(Long userId, Date date) {

        ActivitySignLog activitySignLog = activitySignLogMapper.queryByUserIdAndSignDate(userId, date);

        return !Objects.isNull(activitySignLog);
    }

    @Override
    public SignFacadeDto getSignInfo(Long userId) {
        if (null == userId) {
            return null;
        }

        SignFacadeDto signResultDTO = new SignFacadeDto();
        // 连续签到天数
        Integer signedDays = getSignedDays(userId);

        //获取对应奖励
        DayRewardModel dayRewardModel = signConfig.getDayReward().get(signedDays);
        Integer rewardGold = dayRewardModel.getReward() + dayRewardModel.getExtraReward();

        /**
         * 去掉连续签到的奖励 version 1.4
         if (signedDays.equals(signConfig.getCircle())) {
         rewardGold += signConfig.getCircleReward();
         // 连续签到天数达到签到周期，奖励额外金币
         }
         */
        //去掉周期签到的奖励 version 1.4.0


        signResultDTO.setSignDays(signedDays);
        signResultDTO.setRewardGold(rewardGold);
        return signResultDTO;
    }

    @Override
    public SignTodayDTO getTodaySignInfo(Long userId) {
        if (null == userId) {
            return null;
        }

        return SignTodayDTO.builder()
                .signDays(activitySignService.getSignedDay(userId))
                .todaySignFlag(isSignToday(userId))
                .build();
    }

    @Override
    public boolean isSignToday(Long userId) {
        Date date = signLogMapper.querySignLastDate(userId);
        boolean signedToday = DateUtils.getDiffSeconed(DateUtils.getClearDate(date), DateUtils.getClearDate(new Date()), Boolean.FALSE) == 0L;
        if (null == date || !signedToday) {
            return false;
        }
        return true;
    }

    @Override
    public boolean signByDay(Long userId) {
        if (null == userId) {
            return false;
        }

        // 连续签到天数
        Integer signedDays = getSignedDays(userId);

        // 今日未签到则签到
        Date date = signLogMapper.querySignLastDate(userId);
        boolean signedToday = DateUtils.getDiffSeconed(DateUtils.getClearDate(date), DateUtils.getClearDate(new Date()), Boolean.FALSE) == 0L;
        if (null == date || !signedToday) {
            if (log.isDebugEnabled()) {
                log.debug("用户[{}]已连续签到[{}]天，今日未签到，执行签到操作", userId, signedDays);
            }
            //签到
            Integer signDay = nextSignDayBySignedDays(signedDays);
            DayRewardModel dayRewardModel = signConfig.getDayReward().get(signDay);
            ActivitySignLog signLog = convertToActivitySignLog(userId, dayRewardModel);
            signLogMapper.addOne(signLog);
            accountIntegrationService.rewardSign(userId, signLog.getReward());
            /*
            去掉连续签到奖励 version 1.4
            if (signDay.equals(signConfig.getCircle())) {
                // 连续签到天数达到签到周期，奖励额外金币
                accountIntegrationService.rewardCircleSign(userId);
                if (log.isDebugEnabled()) {
                    log.debug("用户[{}]已连续签到[{}]天，奖励[{}]金币", userId, signConfig.getCircle(), signConfig.getCircleReward());
                }
            }*/

            //活动签到
            activitySignService.doSign(userId);
            return true;
        }
        // 不符合签到条件，不签到
        return false;
    }

    @Override
    public Integer getTodaySignReward(Long userId) {
        // 连续签到天数
        Integer signedDays = getSignedDays(userId);
        Integer signDay = nextSignDayBySignedDays(signedDays);
        DayRewardModel dayRewardModel = signConfig.getDayReward().get(signDay);
        return dayRewardModel.getReward() + dayRewardModel.getExtraReward();
    }

    @Override
    public boolean isStopSigned(Long userId) {


        Date lastSignDate = signLogMapper.querySignLastDate(userId);
        if (Objects.isNull(lastSignDate)) {
            return true;
        }

        Date lastDayClearDate = DateUtils.getClearDate(DateUtils.addField(new Date(), Calendar.DAY_OF_MONTH, -1));
        //若最近的一次签到记录时间（在前一天的时间）之前,则说明已经断签
        return DateUtils.before(lastSignDate, lastDayClearDate);
    }

    /**
     * dayRewardModels 配置了 16个，0是空的数据，这里需要注意
     *
     * @param dayRewardModels
     * @param signedDays
     * @return
     */
    private SignDto convertToSignDtoSevenDay(List<DayRewardModel> dayRewardModels, Integer signedDays, Long userId) {
        SignDto signDto = new SignDto();
        signDto.setPeriodDays(signConfig.getCircle());
        signDto.setPeriodRewardGold(signConfig.getCircleReward());
        signDto.setSignDays(signedDays);
        signDto.setSignToday(isSignToday(userId));
        List<SignDaysDto> signDaysDtoList = Lists.newArrayList();
        //前台只需要7条数据，把15条数据进行处理- 返回1-7，或者8-14或者9-15
        int begin = signedDays < 7 ? 1 : signedDays < 9 ? 8 : 9;
        for (int i = begin; i < begin + 7; i++) {
            signDaysDtoList.add(convertToSignDaysDto(signedDays, dayRewardModels.get(i)));
        }
        //只给前台返回7条数据-version 1.4
//        signDaysDtoList.removeIf(signDaysDto -> 0 == signDaysDto.getDay());
        signDto.setSignDaysDtoList(signDaysDtoList);
        //明日金币 连续签到天数%周期 = 今天 ，+1=明天，可能存在特殊奖励，也要算进去
        Integer tomorrowGold = dayRewardModels.get(signedDays % 15 + 1).getReward();
        if (dayRewardModels.get(signedDays % 15 + 1).getHasExtra()) {
            tomorrowGold += dayRewardModels.get(signedDays % 15 + 1).getExtraReward();
        }
        signDto.setTomorrowGold(tomorrowGold);

        //如果今天未签，将某个字段设置一个标识
        if (!signDto.isSignToday()) {
            Optional<SignDaysDto> signDtoOptional = signDaysDtoList.stream().filter(e -> e.getStatus().equals(0)).findFirst();
            signDtoOptional.ifPresent(e -> e.setTodayShowSign(true));
        }
        return signDto;
    }

    private SignDto convertToSignDto(List<DayRewardModel> dayRewardModels, Integer signedDays) {
        SignDto signDto = new SignDto();
        signDto.setPeriodDays(signConfig.getCircle());
        signDto.setPeriodRewardGold(signConfig.getCircleReward());
        signDto.setSignDays(signedDays);
        List<SignDaysDto> signDaysDtoList = Lists.newArrayList();
        dayRewardModels.forEach(dayRewardModel -> signDaysDtoList.add(convertToSignDaysDto(signedDays, dayRewardModel)));
        signDaysDtoList.removeIf(signDaysDto -> 0 == signDaysDto.getDay());
        signDto.setSignDaysDtoList(signDaysDtoList);
        return signDto;
    }

    private ActivitySignLog convertToActivitySignLog(Long userId, DayRewardModel dayRewardModel) {
        ActivitySignLog signLog = new ActivitySignLog();
        signLog.setId(sequenceCreater.nextLongId());
        signLog.setUserId(userId);
        // 金币奖励+额外金币奖励
        int reward = dayRewardModel.getReward() + dayRewardModel.getExtraReward();
        signLog.setReward(new BigDecimal(reward));
        return signLog;
    }

    private SignDaysDto convertToSignDaysDto(Integer finalSignedDays, DayRewardModel dayRewardModel) {
        SignDaysDto signDaysDto = new SignDaysDto();
        signDaysDto.setDay(dayRewardModel.getDay());
        signDaysDto.setHasExtraGolds(dayRewardModel.getHasExtra());
        signDaysDto.setGolds(dayRewardModel.getReward());
        signDaysDto.setExtraGolds(dayRewardModel.getExtraReward());
        signDaysDto.setStatus(dayRewardModel.getDay() <= finalSignedDays ? 1 : 0);
        return signDaysDto;
    }

    /**
     * 计算用户连续签到了几天（福利社）
     * <p>
     * 连续签到总天数按签到周期取余则为签到时间
     * 连续签到为0 则返回0天,达到周期签到天数(现在为15天)则返回15天,超过周期签到天数则返回取余的天数
     * 时间范围:0-15天
     *
     * @param userId 用户id
     * @return 用户连续签到了几天
     */
    private Integer getSignedDays(Long userId) {
        // sql查询已按签到时间倒叙
        List<ActivitySignLog> signLogs = signLogMapper.listByUserId(userId);
        Integer signedDays = 0;
        Date currentDate = DateUtils.getClearDate(new Date());
        boolean isSignToday = false;
        for (int i = 0; i < signLogs.size(); i++) {
            Date signDate = DateUtils.getClearDate(signLogs.get(i).getCreateTime());
            if (i == 0) {
                //是否今日签到
                isSignToday = DateUtils.getDiffSeconed(signDate, currentDate, Boolean.FALSE) == 0L;
            }

            int calDays = isSignToday ? -i : -i - 1;
            Date todaySignDate = DateUtils.addField(currentDate, Calendar.DAY_OF_MONTH, calDays);
            boolean isDateSame = DateUtils.getDiffSeconed(signDate, todaySignDate, Boolean.FALSE) == 0L;
            if (isDateSame) {
                signedDays++;
            } else {
                break;
            }
        }
        //多于一个周期之后的
        if (0 == signedDays) {
            return signedDays;
        }

        // 如果刚好在签到周期的最后一天 则且是今天 则返回签到周期的最后一天
        // eg: 今天签到第15天，返回15
        if (0 == signedDays % signConfig.getCircle() && isSignToday) {
            return signConfig.getCircle();
        }

        // 如果不是签到周期的最后一天 或者是签到周期最后一天，但是昨天的时间，就返回余数
        // eg: 今天签到第14天 返回14 % 15 = 14; 昨天签到第15天，返回15 % 15 = 0;
        return signedDays % signConfig.getCircle();
    }

    /**
     * 根据已签到天数获取下一签到的天数
     *
     * @param signedDays 已签到天数
     * @return 下一签到天数 若已签到天数等于周期天数 则下一签到天数为1,否则就是已签到天数+1
     */
    private Integer nextSignDayBySignedDays(Integer signedDays) {

        return signedDays.equals(signConfig.getCircle()) ? 1 : (signedDays + 1);

    }

    @Override
    public SignAndInfoDTO signAndInfo(Long userId) {
        if (null == userId) {
            log.error("请求签到接口失败,用户id为空");
            return null;
        }

        if (!signByDay(userId)) {
            log.error("请求签到接口失败,用户id:{}", userId);
            return null;
        }

        // 连续签到天数
        SignFacadeDto signFacadeDto = getSignInfo(userId);

        if (Objects.isNull(signFacadeDto)) {
            log.error("获取签到后签到信息为空,用户id:{}", userId);
            return null;
        }

        //签到触发更新活动福利缓存
        newbieSignWelfareUpdate(userId, signFacadeDto.getSignDays());

        SignAndInfoDTO signResultDTO = new SignAndInfoDTO();
        signResultDTO.setSignDays(signFacadeDto.getSignDays());
        signResultDTO.setRewardGold(signFacadeDto.getRewardGold());

        return signResultDTO;
    }

    private void newbieSignWelfareUpdate(Long userId, Integer signDays) {
        userActivitySignFacadeService.updateUserActivityCache(userId, signDays);
    }
}
