package com.bxm.fossicker.activity.service.task.action.impl;

import com.bxm.fossicker.activity.enums.TaskStatusEnum;
import com.bxm.fossicker.activity.model.constant.ActivityRedisKey;
import com.bxm.fossicker.activity.service.task.action.ActivityTaskAction;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;

import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * 抽象日常任务action实现
 *
 * @author xin.zhao
 */
@Slf4j
public abstract class AbstractActivityDailyTaskAction implements ActivityTaskAction {

    @Autowired
    private StringRedisTemplate redisTemplate;

    @Autowired
    protected RedisHashMapAdapter redisHashMapAdapter;

    @Autowired
    private DistributedLock distributedLock;

    @Override
    public void complete(Long userId, String taskCode) {
        if (!extraCheckForComplete(userId)) {
            return;
        }
        String key = ActivityRedisKey.buildDailyTaskStatusKey(userId).gen();
        redisTemplate.opsForHash().putIfAbsent(
                key,
                taskCode,
                String.valueOf(TaskStatusEnum.FINISH_UNOBTAIN.getCode())
        );

        // 0点之后失效
        redisTemplate.expire(key, DateUtils.getCurSeconds(), TimeUnit.SECONDS);
    }

    /**
     * 扩展 判断校验任务是否完成
     * <p>
     * 普通日常任务点击即表示完成。特殊日常任务：看视频3次，需进行判断
     *
     * @param userId 用户id
     * @return 任务是否完成
     */
    protected abstract boolean extraCheckForComplete(Long userId);

    @Override
    public Boolean allowReward(Long userId, String taskCode) {
        String resource = StringUtils.join(taskCode, Objects.toString(userId));

        if (!distributedLock.lock(resource, userId.toString(), 2, TimeUnit.SECONDS)) {
            log.warn("获取奖励分布式锁失败，用户id: {}", userId);
            return false;
        }

        KeyGenerator key = ActivityRedisKey.buildDailyTaskStatusKey(userId);
        String result = redisHashMapAdapter.get(key, taskCode, String.class);
        if (StringUtils.isNotBlank(result) && Integer.valueOf(result).equals(TaskStatusEnum.FINISH_UNOBTAIN.getCode())) {
            redisHashMapAdapter.put(key, taskCode, TaskStatusEnum.OBTAINED.getCode());
            return true;
        }

        distributedLock.unlock(resource, userId.toString());
        return false;
    }
}