package com.bxm.fossicker.activity.service.task.impl;

import com.bxm.fossicker.activity.config.DayGoldConfig;
import com.bxm.fossicker.activity.domain.ActivityDayGoldMapper;
import com.bxm.fossicker.activity.domain.ActivityTaskMapper;
import com.bxm.fossicker.activity.enums.DayGoldObtainStatusEnum;
import com.bxm.fossicker.activity.enums.DayGoldRedPacketStatusEnum;
import com.bxm.fossicker.activity.facade.TaskFacadeService;
import com.bxm.fossicker.activity.facade.enums.ActivityFacadeEnum;
import com.bxm.fossicker.activity.facade.model.ActivityTaskFacadeDTO;
import com.bxm.fossicker.activity.facade.model.ActivityUserTaskFacadeDTO;
import com.bxm.fossicker.activity.facade.model.DayGoldFacadeDto;
import com.bxm.fossicker.activity.model.dto.DayGoldDto;
import com.bxm.fossicker.activity.model.dto.DayGoldRedPacketDto;
import com.bxm.fossicker.activity.model.vo.ActivityDayGold;
import com.bxm.fossicker.activity.model.vo.ActivityTask;
import com.bxm.fossicker.activity.model.vo.ActivityUserTask;
import com.bxm.fossicker.activity.service.dispatcher.ActivityActionDispatcher;
import com.bxm.fossicker.activity.service.task.TaskService;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.google.common.collect.Lists;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Objects;

/**
 * @author xin.zhao
 * @date 2019/7/11 11:45
 **/
@Service
@Log4j2
public class TaskFacadeServiceImpl implements TaskFacadeService {

    @Autowired
    private ActivityDayGoldMapper dayGoldMapper;

    @Autowired
    private ActivityTaskMapper taskMapper;

    @Autowired
    private SequenceCreater sequenceCreater;

    @Autowired
    private DayGoldConfig dayGoldConfig;

    private TaskService taskService;

    @Autowired
    private ActivityActionDispatcher activityActionDispatcher;

    private TaskService getTaskService() {
        if (null == taskService) {
            taskService = SpringContextHolder.getBean(TaskService.class);
        }
        return taskService;
    }

    @Override
    public Boolean complete(Long userId, ActivityFacadeEnum activityFacadeEnum) {
        if (null == userId || null == activityFacadeEnum) {
            return false;
        }
        switch (activityFacadeEnum.getType()) {
            // 任务
            case 0:
                getTaskService().taskComplete(userId, activityFacadeEnum.name());
                break;
            // 活动
            case 1:
                activityActionDispatcher.getByActivityName(activityFacadeEnum.name()).reward(userId);
                break;
            default:
                return false;
        }
        return true;
    }

    @Override
    @Async
    @Transactional
    public void createDayGold(Long userId) {
        if (log.isDebugEnabled()) {
            log.debug("用户[{}]创建天天领现金活动", userId);
        }
        if (null != userId && dayGoldMapper.count(userId) == 0) {
            createRedPacket(userId);
        }
    }

    @Override
    public DayGoldFacadeDto dayGoldInfo(Long userId) {
        DayGoldDto dayGoldDto = getTaskService().dayGoldInfo(userId);

        if (Objects.nonNull(dayGoldDto)) {
            BigDecimal goldNum = BigDecimal.ZERO;
            for (DayGoldRedPacketDto dayGoldRedPacketDto : dayGoldDto.getDayGoldRedPacketDtoList()) {
                if (dayGoldRedPacketDto.getStatus() == DayGoldRedPacketStatusEnum.WAIT_OBTAIN.ordinal()) {
                    goldNum = dayGoldRedPacketDto.getGoldNum();
                    break;
                }
            }

            return DayGoldFacadeDto.builder()
                    .goldNum(goldNum.longValue())
                    .unObtain(dayGoldDto.getUnObtain()).build();
        }

        return null;
    }

    @Override
    public ActivityUserTaskFacadeDTO queryUserTask(Long userId, ActivityFacadeEnum activityFacadeEnum) {
        ActivityUserTask activityUserTask = taskMapper.query(userId, activityFacadeEnum.name());

        if (Objects.nonNull(activityUserTask)) {
            ActivityUserTaskFacadeDTO dto = new ActivityUserTaskFacadeDTO();

            BeanUtils.copyProperties(activityUserTask, dto);
            return dto;
        }

        return null;
    }

    @Override
    public ActivityTaskFacadeDTO queryById(Long taskId) {
        ActivityTask activityTask = taskMapper.queryById(taskId);
        if (Objects.nonNull(activityTask)) {

            ActivityTaskFacadeDTO dto = new ActivityTaskFacadeDTO();
            BeanUtils.copyProperties(activityTask, dto);
            return dto;
        }

        return null;
    }


    private void createRedPacket(Long userId) {
        List<ActivityDayGold> dayGoldList = Lists.newArrayList();
        Date currentDate = new Date();

        int day = 1;
        while (day <= 7) {
            dayGoldList.add(
                    ActivityDayGold.builder()
                            .id(sequenceCreater.nextLongId())
                            .userId(userId)
                            .dayNum(day)
                            .status(DayGoldObtainStatusEnum.NOT_OBTAIN.ordinal())
                            .expectDate(DateUtils.addField(DateUtils.getClearDate(currentDate), Calendar.DAY_OF_MONTH, day - 1))
                            .awardType(dayGoldConfig.getType())
                            .awardAmount(new BigDecimal(dayGoldConfig.getAmount().get(day - 1)))
                            .createTime(currentDate)
                            .build()
            );
            day++;
        }
        dayGoldMapper.batchInsert(dayGoldList);
    }
}
