package com.bxm.fossicker.commodity.service;

import com.bxm.fossicker.commodity.common.dto.CommissionAndCoupon;
import com.bxm.fossicker.commodity.common.dto.PlatformCommissionRateInfo;
import com.bxm.fossicker.commodity.common.dto.CommodityCommissionAndCoupon;

/**
 * 商品价格体系服务-将商品价格、优惠券、佣金做统一管理,提供服务给出-解决商品价格不一致等问题
 * @author donghy
 */
public interface CommodityPriceInfoService {


    /**
     * 获取佣金和优惠券信息
     * @param goodsId 商品id
     * @param userId 用户id
     * @return 佣金和优惠券信息
     */
    CommodityCommissionAndCoupon getCommodityCommissionAndCoupon(Long goodsId, Long userId);

    /**
     * 获取佣金和优惠券信息
     * @param goodsId 商品id
     * @param userId 用户id
     * @param newbieNotShowFlag 是否展示新人商品佣金标志 true-不展示 false-展示
     * @return 佣金和优惠券信息
     */
    CommodityCommissionAndCoupon getCommodityCommissionAndCoupon(Long goodsId, Long userId,
                                                                 Boolean newbieNotShowFlag);

    /**
     * 获取商品原价
     * @param goodsId 商品id
     * @return 商品原价
     */
    Double getReservePrice(Long goodsId);

    /**
     * 根据用户id和商品id判断，是否具备新人专享权益
     * @see #isNewbieRightCommodity 是否是新人商品
     * @see #isNewbieRightForUser 是否是新人用户
     * @param userId 用户id
     * @param goodsId 商品id
     * @return true-具备,false-不具备
     */
    Boolean isNewbieRight(Long userId, Long goodsId);

    /**
     * 获取第三方优惠券佣金信息-现只提供给管理后台使用
     * @param goodsId 商品id
     * @author Dong_hy
     * @return 佣金信息
     */
    CommissionAndCoupon getCommissionAndCouponByThird(Long goodsId);

    /**
     * 该用户是否具备新人专享权益-仅校验之前是否购买过商品- 该方法应在用户模块声明
     *
     * @param userId 用户id
     * @return true-具备,false-不具备
     */
    boolean isNewbieRightForUser(Long userId);

    /**
     * 是否新人专享商品
     * @param goodsId 商品id
     * @return 是否新人专享商品
     */
    boolean isNewbieRightCommodity(Long goodsId);

    /**
     * 是否vip 0元购商品
     * @param goodsId
     * @return
     */
    boolean isVipZeroCommodity(Long goodsId);

    /**
     * 查询三方接口，获取优惠券和佣金信息
     * 返回的数据只包含优惠券金额、地址和三方佣金比例
     * @param goodsId
     * @return
     */
    CommissionAndCoupon getCommodityCommissionAndCoupon(Long goodsId);

    /**
     * 查询自身平台的佣金比例信息
     * @param goodsId 商品id
     * @return 自身平台的佣金比例信息
     */
    PlatformCommissionRateInfo getPlatformCommissionRateInfo(Long goodsId);

    /**
     * 获取平台通用佣金
     * @return 平台通用佣金
     */
    String getCommonPurchaseRate();

    /**
     * 佣金值是否符合
     * @param commissionPrice 商品佣金
     * @return true-符合规定, false-不符合规定
     */
    boolean isRuleNewbiePrice(Double commissionPrice);

    /**
     * 校验佣金值 是否符合vip 0元购金额规则
     * 1.6.0版本的计算公式：
     * 补贴金额阈值：
     * 补贴金额=商品券后价*（1-返佣比例）+商品券后价*10%（其中，返佣比例为高佣转链返回的返佣比例）
     * 补贴金额最高6元，大于6元则下架（其中，6元为可配置）
     *
     * @param platformCommissionPrice 商品的平台佣金金额
     * @param discountPrice 券后价
     * @return 校验佣金值 是否符合vip 0元购金额规则
     */
    boolean isRuleVipZeroPrice(Double platformCommissionPrice, Double discountPrice);


}
