package com.bxm.fossicker.commodity.service.commodity.info.source;

import com.alibaba.fastjson.JSON;
import com.bxm.fossicker.commodity.common.enums.LocalCommodityStatusEnum;
import com.bxm.fossicker.commodity.model.constant.CommoditySourceEnum;
import com.bxm.fossicker.commodity.model.dto.CommodityDetailInfoDTO;
import com.bxm.fossicker.commodity.model.dto.CommodityInfoDTO;
import com.bxm.fossicker.commodity.model.param.GetCommodityDetailParam;
import com.bxm.fossicker.commodity.service.commodity.info.CommodityInfoSourceProxyService;
import com.bxm.fossicker.commodity.service.commodity.info.CommodityInfoSourceStrategy;
import com.bxm.fossicker.commodity.service.commodity.info.source.annotation.CommoditySourceStage;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.aop.support.AopUtils;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.context.event.EventListener;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * @author Gonzo
 * @date 2019-09-27 10:27
 * 对外提供功能的代理类
 */
@Service
@Slf4j
public class CommodityInfoSourceProxyServiceImpl implements CommodityInfoSourceProxyService {

    private static final Map<CommoditySourceEnum, CommodityInfoSourceStrategy> STAGE_MAP = Maps.newHashMap();

    @Override
    public CommodityInfoDTO getCommodityDetail(Long commodityId) {
        return getCommodityDetail(GetCommodityDetailParam.builder()
                .commodityId(commodityId)
                // 默认查询优惠券信息 兼容老的版本
                .queryCouponInfo(true)
                .build());
    }

    @Override
    public CommodityInfoDTO getCommodityDetail(GetCommodityDetailParam param) {

        if (log.isDebugEnabled()) {
            log.debug("获取商品信息，请求参数: {}", JSON.toJSONString(param));
        }

        Long commodityId = param.getCommodityId();
        if (Objects.isNull(commodityId)) {
            log.warn("commodityId 为空");
            return null;
        }

        // 优先从缓存中获取
        // 优先从本地找
        CommodityInfoSourceStrategy stage = STAGE_MAP.get(CommoditySourceEnum.LOCAL);

        if (Objects.isNull(stage)) {
            log.warn("数据源类型: {} 的处理类为null", CommoditySourceEnum.LOCAL);
            return null;
        }

        CommodityInfoDTO commodityInfoDTO = stage.getCommodityDetail(param);

        // 如果未指定库内下架则返回空，则判断状态，从三方获取
        if (!param.isIfLocalDownThenReturnNull()) {
            // 下架 或者不存在则从淘宝抓取
            if (Objects.isNull(commodityInfoDTO)
                    || Objects.equals(LocalCommodityStatusEnum.InValid.getStatus(), commodityInfoDTO.getStatus())) {
                // 再从淘宝爬取
                stage = STAGE_MAP.get(CommoditySourceEnum.TAOBAO);

                if (Objects.isNull(stage)) {
                    log.warn("数据源类型: {} 的处理类为null", CommoditySourceEnum.TAOBAO);
                    return null;
                }

                commodityInfoDTO = stage.getCommodityDetail(param);
            }
        }

        return commodityInfoDTO;
    }

    @Override
    public List<CommodityDetailInfoDTO> getCommodityDetailInfo(Long commodityId) {
        // 查找顺序这里可以设计下

        // 优先从本地找
        CommodityInfoSourceStrategy stage = STAGE_MAP.get(CommoditySourceEnum.LOCAL);

        if (Objects.isNull(stage)) {
            log.warn("数据源类型: {} 的处理类为null", CommoditySourceEnum.LOCAL);
            return null;
        }

        List<CommodityDetailInfoDTO> detailInfo = stage.getCommodityDetailInfo(commodityId);

        if (!CollectionUtils.isEmpty(detailInfo)) {
            return detailInfo;
        }

        // 再从淘宝爬取
        stage = STAGE_MAP.get(CommoditySourceEnum.TAOBAO);

        if (Objects.isNull(stage)) {
            log.warn("数据源类型: {} 的处理类为null", CommoditySourceEnum.TAOBAO);
            return null;
        }

        return stage.getCommodityDetailInfo(commodityId);
    }

    @Override
    public CommodityInfoDTO getLocalCommodity(Long commodityId) {
        return getLocalCommodity(GetCommodityDetailParam.builder()
                .commodityId(commodityId)
                // 默认查询优惠券信息 兼容老的版本
                .queryCouponInfo(true)
                .build());
    }

    @Override
    public CommodityInfoDTO getLocalCommodity(GetCommodityDetailParam param) {
        // 从本地查找
        CommodityInfoSourceStrategy stage = STAGE_MAP.get(CommoditySourceEnum.LOCAL);

        if (Objects.isNull(stage)) {
            log.warn("数据源类型: {} 的处理类为null", CommoditySourceEnum.LOCAL);
            return null;
        }

        return stage.getCommodityDetail(param);
    }

    @Override
    public CommodityInfoDTO getThirdpartyCommodity(GetCommodityDetailParam param) {
        // 从三方数据
        CommodityInfoSourceStrategy stage = STAGE_MAP.get(CommoditySourceEnum.TAOBAO);

        if (Objects.isNull(stage)) {
            log.warn("数据源类型: {} 的处理类为null", CommoditySourceEnum.TAOBAO);
            return null;
        }

        return stage.getCommodityDetail(param);
    }

    @Override
    @Async
    public void preheatCommodityInfo(List<Long> commodityIds) {
        if (log.isDebugEnabled()) {
            log.debug("商品信息数据预热，开始");
        }

        commodityIds.parallelStream().forEach(p -> {
            getCommodityDetail(GetCommodityDetailParam.builder()
                    .commodityId(p)
                    .build());
        });

        if (log.isDebugEnabled()) {
            log.debug("商品信息数据预热，结束");
        }

    }

    @EventListener(ContextRefreshedEvent.class)
    public void onApplicationStartedEvent(ContextRefreshedEvent event) {

        Collection<CommodityInfoSourceStrategy> stages = event.getApplicationContext()
                .getBeansOfType(CommodityInfoSourceStrategy.class)
                .values();

        stages.forEach(p -> {
            CommoditySourceStage sourceStage = AnnotationUtils.getAnnotation(AopUtils.getTargetClass(p),
                    CommoditySourceStage.class);

            if (!Objects.isNull(sourceStage)) {
                STAGE_MAP.put(sourceStage.value(), p);
            }
        });
    }
}
