package com.bxm.fossicker.commodity.service.commodity.info.source;

import com.alibaba.fastjson.JSON;
import com.bxm.fossicker.commodity.common.enums.LocalCommodityStatusEnum;
import com.bxm.fossicker.commodity.model.constant.CommoditySourceEnum;
import com.bxm.fossicker.commodity.model.dto.CommodityDetailInfoDTO;
import com.bxm.fossicker.commodity.model.dto.CommodityInfoDTO;
import com.bxm.fossicker.commodity.model.entry.CommodityInfo;
import com.bxm.fossicker.commodity.model.param.GetCommodityDetailParam;
import com.bxm.fossicker.commodity.model.vo.CommodityCouponInfo;
import com.bxm.fossicker.commodity.service.CommodityCouponInfoService;
import com.bxm.fossicker.commodity.service.commodity.info.source.annotation.CommoditySourceStage;
import com.bxm.fossicker.constant.CommonCommodityRedisKey;
import com.bxm.fossicker.integration.param.GetCouponInfoParam;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.List;
import java.util.Objects;

/**
 * @author Gonzo
 * @date 2019-09-26 19:27
 * 本地数据库查询数据
 */
@CommoditySourceStage(CommoditySourceEnum.LOCAL)
@Slf4j
public class LocalCommodityInfoSourceStrategy extends AbstractCommodityInfoSourceStrategy {

    @Autowired
    private CommodityCouponInfoService commodityCouponInfoService;

    @Override
    public CommodityInfoDTO doGetCommodityDetail(GetCommodityDetailParam param) {

        Long commodityId = param.getCommodityId();
        // 获取基本详情
        CommodityInfoDTO commodityInfoDTO = commodityDetail(commodityId);

        if (Objects.nonNull(commodityInfoDTO)) {

            // 如果表明在库内下架则返回null 则判断是否下架
            if (param.isIfLocalDownThenReturnNull()
                    && Objects.equals(commodityInfoDTO.getStatus(), LocalCommodityStatusEnum.InValid.getStatus())) {
                // 下架 返回null
                return null;
            }

            // 未下架

            // 查询优惠券
            // 1.5.0 之后优惠券不会在入库时记录，而是单独的进行查询
            if (param.isQueryCouponInfo()
                    // 上架状态才查询优惠券 
                    && Objects.equals(commodityInfoDTO.getStatus(), LocalCommodityStatusEnum.Valid.getStatus())) {
                if (log.isDebugEnabled()) {
                    log.debug("版本低于1.5.0，既时查询优惠券，商品id: {}", param.getCommodityId());
                }

                CommodityCouponInfo commodityCouponInfo =
                        // 内部会优先查询缓存
                        commodityCouponInfoService.getCommodityCouponInfo(GetCouponInfoParam.builder()
                                .goodsId(Objects.toString(commodityId))
                                // 数据来源
                                .sourceFrom(commodityInfoDTO.getSoruce())
                                // 优惠券链接不替换协议
                                .replaceHttp2TbOpen(false)
                                .build());

                if (Objects.nonNull(commodityCouponInfo)) {
                    commodityInfoDTO.setCouponUrl(commodityCouponInfo.getCouponUrl());
                }
            } else {
                // 1.5.0之后，高佣的优惠券链接不返回客户端判断之后请求接口获取
                // 如果返回了则不会请求高佣接口，这是给库外数据用的判断，所以这里一定要置为null
                commodityInfoDTO.setCouponUrl(null);
            }
        }

        return commodityInfoDTO;
    }

    private CommodityInfoDTO commodityDetail(Long commodityId) {

        if (log.isDebugEnabled()) {
            log.debug("commodityId: {} 从本地获取数据", commodityId);
        }

        // 优先从redis获取
        KeyGenerator keyGenerator = CommonCommodityRedisKey.LOCAL_COMMODITY_INFO.copy()
                .appendKey(Math.abs(commodityId.toString().hashCode()) % 10);

        CommodityInfo commodityInfo = redisHashMapAdapter.get(keyGenerator, commodityId.toString(), CommodityInfo.class);

        // redis中数据不存在，则直接返回，因为同步模块始终会将数据预热。
        if (Objects.isNull(commodityInfo)) {
            if (log.isDebugEnabled()) {
                log.debug("commodityId: {} 对应的本地数据不存在", commodityId);
            }

            return null;
        }

        return convert(commodityInfo);
    }

    @Override
    protected List<CommodityDetailInfoDTO> doGetCommodityDetailInfo(Long commodityId) {

        CommodityInfoDTO commodityInfoDTO = commodityDetail(commodityId);

        if (Objects.isNull(commodityInfoDTO)) {
            return null;
        }

        return commodityInfoDTO.getDetailInfo();
    }


    private CommodityInfoDTO convert(CommodityInfo commodityInfo) {

        String sellerScore = "4.8";
        String logisticsScore = "4.8";
        String commodityScore = "4.8";

        if (StringUtils.isNotBlank(commodityInfo.getSellerScore())) {
            sellerScore = new BigDecimal(commodityInfo.getSellerScore()).setScale(1, RoundingMode.HALF_DOWN).toString();
        }

        if (StringUtils.isNotBlank(commodityInfo.getCommodityScore())) {
            logisticsScore = new BigDecimal(commodityInfo.getCommodityScore()).setScale(1, RoundingMode.HALF_DOWN).toString();
        }

        if (StringUtils.isNotBlank(commodityInfo.getLogisticsScore())) {
            commodityScore = new BigDecimal(commodityInfo.getLogisticsScore()).setScale(1, RoundingMode.HALF_DOWN).toString();
        }

        return CommodityInfoDTO.builder()

                // 平台佣金值
                .platformCommissionPrice(Objects.isNull(commodityInfo.getCommissionPrice()) ? 0.00 :
                        commodityInfo.getCommissionPrice().setScale(2, RoundingMode.HALF_DOWN).doubleValue())
                // 商品价格-到手价,商品原价-优惠券价格-佣金
                .commodityPrice(Objects.isNull(commodityInfo.getCommodityPrice()) ? 0.00 :
                        commodityInfo.getCommodityPrice().setScale(2, RoundingMode.HALF_DOWN).doubleValue())

                // 商品标题
                .commodityTitle(commodityInfo.getCommodityTitle())
                // 优惠券价格
                .couponPrice(Objects.isNull(commodityInfo.getCouponPrice()) ? 0.00 :
                        commodityInfo.getCouponPrice().setScale(2, RoundingMode.HALF_DOWN).doubleValue())
                // 优惠券链接
                .couponUrl(commodityInfo.getCouponUrl())
                // 券后价,商品原价-优惠券价格
                .discountPrice(Objects.isNull(commodityInfo.getDiscountPrice()) ? 0.00 :
                        commodityInfo.getDiscountPrice().setScale(2, RoundingMode.HALF_DOWN).doubleValue())
                // 商品id
                .goodsId(Objects.isNull(commodityInfo.getGoodId()) ? null : Long.parseLong(commodityInfo.getGoodId()))
                // 商品图片list
                .imgUrlList(StringUtils.isNotBlank(commodityInfo.getImgUrlList()) ?
                        JSON.parseArray(commodityInfo.getImgUrlList(), String.class) : Lists.newArrayList())

                // 商品主图
                .mainPic(commodityInfo.getMainPic())
                // 推荐内容
                .recommendContent(commodityInfo.getRecommendContent())
                // 商品原价
                .reservePrice(Objects.isNull(commodityInfo.getReservePrice()) ? 0.00 :
                        commodityInfo.getReservePrice().setScale(2, RoundingMode.HALF_DOWN).doubleValue())
                // 销量
                .salesNum(commodityInfo.getSalesNum())

                // 卖家服务得分
                .sellerScore(sellerScore)
                // 商品描述得分
                .commodityScore(logisticsScore)
                // 物流得分
                .logisticsScore(commodityScore)

                // 店铺标志
                .shopIcon(commodityInfo.getShopIcon())
                // 店铺名称
                .shopName(commodityInfo.getShopName())
                // 店铺类型，1-天猫，0-淘宝
                .shopType(commodityInfo.getShopType().intValue())
                // 店铺链接url
                .shopUrl(commodityInfo.getShopUrl())
                .status(commodityInfo.getStatus())
                // 图文详情
                .detailInfo(StringUtils.isNotBlank(commodityInfo.getDetailInfo())
                        ? JSON.parseArray(commodityInfo.getDetailInfo(), CommodityDetailInfoDTO.class)
                        : Lists.newArrayList())
                .soruce(commodityInfo.getSoruce())
                .build();

    }

}
