package com.bxm.fossicker.commodity.service.commodity.info.source;

import com.bxm.fossicker.commodity.common.dto.CommissionAndCoupon;
import com.bxm.fossicker.commodity.common.utils.CommodityCalUtils;
import com.bxm.fossicker.commodity.model.constant.CommodityRedisKey;
import com.bxm.fossicker.commodity.model.constant.CommoditySourceEnum;
import com.bxm.fossicker.commodity.model.dto.CommodityDetailInfoDTO;
import com.bxm.fossicker.commodity.model.dto.CommodityInfoDTO;
import com.bxm.fossicker.commodity.model.param.GetCommodityDetailParam;
import com.bxm.fossicker.commodity.service.CommodityPriceInfoService;
import com.bxm.fossicker.commodity.service.commodity.info.source.annotation.CommoditySourceStage;
import com.bxm.fossicker.constant.LocalCommoditySourceFrom;
import com.bxm.fossicker.integration.dto.CommodityDetailIntegrationDTO;
import com.bxm.fossicker.integration.dto.CommodityImageIntegrationDTO;
import com.bxm.fossicker.integration.service.CrawlerIntegrationService;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author Gonzo
 * @date 2019-09-27 15:10
 * 从淘宝获取数据
 */
@CommoditySourceStage(CommoditySourceEnum.TAOBAO)
@Slf4j
public class TaobaoCommodityInfoSourceStrategy extends AbstractCommodityInfoSourceStrategy {


    @Autowired
    private CrawlerIntegrationService crawlerIntegrationService;

    @Autowired
    private CommodityPriceInfoService commodityPriceInfoService;

    private static final int EXPIRE_SEC = 60 * 20;


    @Override
    protected CommodityInfoDTO doGetCommodityDetail(GetCommodityDetailParam param ) {
        Long commodityId = param.getCommodityId();
        if (log.isDebugEnabled()) {
            log.debug("commodityId: {} 从爬虫爬取数据", commodityId);
        }

        // 优先从缓存获取
        KeyGenerator key = CommodityRedisKey.TAOBAO_COMMODITY_INFO.copy().appendKey(commodityId);
        CommodityInfoDTO commodityInfoDTO = redisStringAdapter.get(key, CommodityInfoDTO.class);

        if (Objects.isNull(commodityInfoDTO)) {
            //从爬取的信息中取商品详情
            CommodityDetailIntegrationDTO queryResult = crawlerIntegrationService.getCommodityDetail(commodityId);
            if (Objects.isNull(queryResult)) {
                log.warn("commodityId: {} 数据爬取失败", commodityId);

                commodityInfoDTO = new CommodityInfoDTO();
                commodityInfoDTO.setGoodsId(commodityId);

            } else {
                commodityInfoDTO = convert(queryResult);
            }

            //佣金优惠券信息
            CommissionAndCoupon commissionAndCoupon = commodityPriceInfoService.getCommodityCommissionAndCoupon(commodityId);
            Double reservePrice = commodityPriceInfoService.getReservePrice(commodityId);

            // 原价
            commodityInfoDTO.setReservePrice(reservePrice);

            if (!Objects.isNull(commissionAndCoupon)) {

                BigDecimal platformCommissionPrice = BigDecimal.ZERO;

                // 计算平台佣金
                if (!Objects.isNull(commodityInfoDTO.getReservePrice())
                        && !Objects.isNull(commissionAndCoupon.getCouponPrice())
                        && !Objects.isNull(commissionAndCoupon.getCommissionRateByThird())
                        && !Objects.isNull(reservePrice)) {

                    platformCommissionPrice = BigDecimal.valueOf(reservePrice)
                            .subtract(BigDecimal.valueOf(commissionAndCoupon.getCouponPrice()))
                            .multiply(BigDecimal.valueOf(commissionAndCoupon.getCommissionRateByThird()))
                            .setScale(2, BigDecimal.ROUND_DOWN);
                }

                // 券后价
                commodityInfoDTO.setDiscountPrice(CommodityCalUtils.calCommodityPriceWithSubtractingCoupon(commodityInfoDTO.getReservePrice(),
                        commissionAndCoupon.getCouponPrice()));
                // 优惠券金额
                commodityInfoDTO.setCouponPrice(commissionAndCoupon.getCouponPrice());
                // 平台佣金
                commodityInfoDTO.setPlatformCommissionPrice(platformCommissionPrice.doubleValue());
                // 优惠券url
                commodityInfoDTO.setCouponUrl(commissionAndCoupon.getCouponUrl());
            }

            // 图文详情
            List<CommodityDetailInfoDTO> detailInfo = doGetCommodityDetailInfo(commodityId);
            if (!CollectionUtils.isEmpty(detailInfo)) {
                commodityInfoDTO.setDetailInfo(detailInfo);
            }

            // 20分钟后失效
            redisStringAdapter.set(key, commodityInfoDTO, EXPIRE_SEC);
        }

        return commodityInfoDTO;
    }

    @Override
    protected List<CommodityDetailInfoDTO> doGetCommodityDetailInfo(Long commodityId) {
        if (log.isDebugEnabled()) {
            log.debug("commodityId: {} 爬取获取图文详情", commodityId);
        }

        List<CommodityImageIntegrationDTO> commodityImageIntegrationDTOS = crawlerIntegrationService.getDetailList(commodityId);

        if (!CollectionUtils.isEmpty(commodityImageIntegrationDTOS)) {
            return commodityImageIntegrationDTOS.stream().map(this::convert).collect(Collectors.toList());
        }

        return Lists.newArrayList();
    }

    private CommodityDetailInfoDTO convert(CommodityImageIntegrationDTO integrationDTO) {
        CommodityDetailInfoDTO infoDTO = CommodityDetailInfoDTO.builder().build();

        BeanUtils.copyProperties(integrationDTO, infoDTO);

        return infoDTO;
    }

    /**
     * 转换爬虫信息到商品详情信息
     * @param commodityDetailIntegrationDTO 爬虫到的商品详情信息
     * @return 商品详情信息
     */
    private CommodityInfoDTO convert(CommodityDetailIntegrationDTO commodityDetailIntegrationDTO) {
        String mainPic = "";
        if (!CollectionUtils.isEmpty(commodityDetailIntegrationDTO.getImgUrlList())) {
            mainPic = commodityDetailIntegrationDTO.getImgUrlList().get(0);
        }

        return CommodityInfoDTO.builder()
                .goodsId(commodityDetailIntegrationDTO.getCommodityId())
                .commodityTitle(commodityDetailIntegrationDTO.getCommodityTitle())
                .imgUrlList(commodityDetailIntegrationDTO.getImgUrlList())
                .mainPic(mainPic)
                .reservePrice(commodityDetailIntegrationDTO.getReservePrice())
                .salesNum(commodityDetailIntegrationDTO.getSalesNum())
                .shopName(commodityDetailIntegrationDTO.getShopName())
                .shopIcon(commodityDetailIntegrationDTO.getShopIcon())
                .shopUrl(commodityDetailIntegrationDTO.getShopUrl())
                .commodityScore(commodityDetailIntegrationDTO.getCommodityScore())
                .sellerScore(commodityDetailIntegrationDTO.getSellerScore())
                .logisticsScore(commodityDetailIntegrationDTO.getLogisticsScore())
                .shopType(getShopTypeFromTypeCode(commodityDetailIntegrationDTO.getShopType()))
                // 三方数据
                .soruce(LocalCommoditySourceFrom.THIREDPARTY)
                .build();

    }


}
