package com.bxm.fossicker.commodity.service.commodity.list;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.fossicker.commodity.config.CommodityInfoProperties;
import com.bxm.fossicker.commodity.config.CommodityListProperties;
import com.bxm.fossicker.commodity.model.constant.CommodityDetailResultCode;
import com.bxm.fossicker.commodity.model.dto.CommodityExtendInfo;
import com.bxm.fossicker.commodity.model.dto.CommodityGoldInfoDTO;
import com.bxm.fossicker.commodity.model.dto.CommodityListItemResult;
import com.bxm.fossicker.commodity.model.dto.Index;
import com.bxm.fossicker.commodity.model.param.CommodityDetailQueryParam;
import com.bxm.fossicker.commodity.model.param.CommodityListParam;
import com.bxm.fossicker.commodity.model.vo.BaseCommodityListItem;
import com.bxm.fossicker.commodity.service.*;
import com.bxm.fossicker.commodity.service.commodity.info.CommodityInfoSourceProxyService;
import com.bxm.fossicker.commodity.service.strategy.CommodityLabelService;
import com.bxm.fossicker.user.facade.vip.VipFacadeService;
import com.bxm.newidea.component.vo.Message;
import com.google.common.collect.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

public abstract class AbstractCommodityListService<T extends BaseCommodityListItem, P extends CommodityListParam>
        extends CommodityListServiceProxy
        implements CommodityListService<T, P>  {

    protected Logger log = LoggerFactory.getLogger(getClass());

    @Autowired
    protected CommodityInfoSourceProxyService commodityInfoSourceProxyService;

    @Autowired
    private CommodityInfoService commodityInfoService;

    @Autowired
    protected CommodityPoolService commodityPoolService;

    @Autowired
    private BlackCommodityService blackCommodityService;

    @Autowired
    private CommodityPriceInfoService commodityPriceInfoService;

    @Autowired
    private CommodityGoldService commodityGoldService;

    @Autowired
    private VipFacadeService vipFacadeService;

    @Autowired
    protected CommodityListProperties commodityListProperties;

    @Autowired
    private CommodityLabelService commodityLabelService;

    @Autowired
    protected CommodityInfoProperties commodityInfoProperties;

    @Override
    public List<T> commodityList(P p) {
        Message message = checkParam(p);
        if (!message.isSuccess()) {
            log.warn("{} 校验参数失败，错误信息: {}", description(), message.getLastMessage());
            return Lists.newArrayList();
        }


        List<T> ts;
        try {
            // 获取商品列表
            ts = doGetCommodityList(p);

            if (!CollectionUtils.isEmpty(ts)) {
                // 处理商品列表
                ts = afterGetCommodityList(p, ts);
            }
        } catch(Exception e) {
            log.error("获取{} 列表失败", description(), e);
            ts = Lists.newArrayList();
        }
        return ts;
    }

    /**
     * 初始化查询参数
     * @param p p
     * @param goodsId 商品id
     * @return 查询参数
     */
    protected CommodityDetailQueryParam initCommodityDetailQueryParam(P p, Long goodsId) {
        CommodityDetailQueryParam queryParam = new CommodityDetailQueryParam();
        queryParam.setUserId(p.getUserId());
        queryParam.setVipUser(p.getVip());
        queryParam.setNewbieRightUser(p.getNewbieRight());
        queryParam.setGoodsId(goodsId);
        queryParam.setPublicRequest(p.isPublicRequest());

        // 所有列表都需要校验价格信息、进行剔除失效的
        queryParam.setValidateCommissionPriceInfo(true);
        // 所有列表都需要查询新人商品信息
        queryParam.setQueryNewbie(true);

        return queryParam;
    }
    /**
     * 获取商品列表的基本信息
     * 商品失效(佣金为0) -> 返回 commissionValidate = false
     * 库内和区外都没有数据 -> 返回notFound = true
     *
     * 如果 isLocalOnly 为true，则只查询库内，
     * 如果库内为空或者下架状态，则返回 notFound = true
     * @param param param
     * @return 商品信息
     */
    protected CommodityListItemResult commodityListItemBaseInfo(CommodityDetailQueryParam param) {
        // 调用商品信息服务进行商品信息的获取
        return commodityInfoService.getCommodityListItemBaseInfo(param);
    }

    /**
     * 查询标签信息
     * @param commodity commodity
     */
    private void setLabelUrl(BaseCommodityListItem commodity) {
        // 新人不需要标签
        commodity.setLabelUrl(Objects.equals(commodity.getNewbieRight(), Boolean.TRUE)
                ? new ArrayList<>()
                : commodityLabelService.labelTypeFromRedis(commodity.getGoodsId()));
    }
    /**
     * 是否需要获取金币信息
     * @param commodity commodity
     * @param userId userId
     */
    protected void setGoldInfo(BaseCommodityListItem commodity, Long userId) {

        CommodityGoldInfoDTO goldInfoDTO = commodityGoldService.getGoldInfoByCommodityId(commodity.getGoodsId(), userId);
        commodity.setGoldStatus(goldInfoDTO.getDrawFlag());
        commodity.setGoldNum(goldInfoDTO.getBrowseGold());
    }

    /**
     * 校验价格信息 优惠券为0，原价为空
     * 如果商品是新人商品，还会校验是否符合价格规则，否则会进行下架
     * @param commodityExtendInfo ip价格、普通佣金价格、新人展示标志位等信息
     * @param couponPrice reservePrice
     * @param reservePrice reservePrice
     * @param goodsId goodsId
     * @return 校验成功否
     */
    private boolean validateCommission(CommodityExtendInfo commodityExtendInfo, Double reservePrice,
                                       Double couponPrice, Long goodsId) {

        if (Objects.isNull(commodityExtendInfo)
                || Objects.isNull(commodityExtendInfo.getCommissionPrice())
                || Objects.isNull(commodityExtendInfo.getCommodityPrice())
                || Objects.isNull(reservePrice)
                || Objects.isNull(couponPrice)) {
            return false;
        }

        // 佣金为0
        if (Objects.equals(commodityExtendInfo.getCommissionPrice(),  (double) 0)) {
            return false;
        }

        // 优惠券为0
        if (Objects.equals(couponPrice, (double) 0)) {
            return false;
        }

        // 判断新人商品是否符合价格规则
        if (commodityExtendInfo.isNewbieRight()) {
            if (commodityPriceInfoService.isRuleNewbiePrice(commodityExtendInfo.getCommodityPrice())) {
                commodityExtendInfo.setCommissionPrice(commodityExtendInfo.getCommodityPrice());
                commodityExtendInfo.setCommodityPrice(0D);
            } else {
                //商品下架且剔除该商品
                log.info("新人专享商品: {} 折后价超出最大限额，进行下架处理：[{}]", goodsId,
                        JSONObject.toJSONString(commodityExtendInfo));
                commodityPoolService.setNewbieInvalidStatus(goodsId);
                return false;
            }
        }

        return true;
    }

    /**
     * 校验列表信息获取是否成功
     * 不为null 且金额校验成功 且查找成功
     * @return 成功否
     */
    protected boolean checkCommodityListBaseItemSuccess(CommodityListItemResult result) {
        return Objects.equals(result.getResultCode(), CommodityDetailResultCode.SUCCESS)
                && Objects.nonNull(result.getListItem());
    }

    /**
     * 校验参数
     * @param p param
     * @return 成功 or 失败信息
     */
    protected Message checkParam(P p) {

        if (!p.isPublicRequest() && Objects.isNull(p.getUserId())) {
            return Message.build(false).setMessage("用户id为空");
        }

        // 设置基本信息
        // 新人标志位
        if (Objects.isNull(p.getNewbieRight())) {
            p.setNewbieRight(commodityPriceInfoService.isNewbieRightForUser(p.getUserId()));
        }

        // vip
        if (Objects.isNull(p.getVip())) {
            p.setVip(vipFacadeService.isVip(p.getUserId()));
        }

        return Message.build();
    }

    /**
     * 获取对集合进行分页的交表数据
     * @param pageOn pageOn 从1开始
     * @param pageSize pageSize
     * @param totalSize totalSize
     * @return start and end
     */
    protected Index getIndex(int pageOn, int pageSize, int totalSize) {
        Index index = new Index();

        int start = pageOn > 1 ? (pageOn - 1) * pageSize : 0;
        int end = start + pageSize;
        end = end > totalSize ? totalSize : end;

        index.setStart(start);
        index.setEnd(end);

        return index;
    }

    /**
     * 列表的后置处理，可以定义一些通用操作。
     * 如：金币封装
     * 因为金币数据是不同人都不同情况，但是列表基本数据是需要缓存的，所以这种操作放到后置里完成
     * @param p 请求参数
     * @param ts 列表数据
     * @return 处理后的列表
     */
    protected List<T> afterGetCommodityList(P p, List<T> ts) {

        if (log.isDebugEnabled()) {
            log.debug("列表后置处理，请求参数: {}", JSON.toJSONString(p));
        }

        // TODO 如果不是vip还需要移除vip 0元购
        return ts.stream()

                .peek(p1 -> {
                    // 设置金币信息
                    if (p.isQueryGoldInfo()) {
                        setGoldInfo(p1, p.getUserId());
                    }

                    // 查询标签信息
                    if (p.isQueryLabel()) {
                        setLabelUrl(p1);
                    }
                }).collect(Collectors.toList());

    }

    /**
     * 将包含商品id列表的数据，封装成对应的商品列表
     * @param p param
     * @return 商品列表
     */
    abstract protected List<T> doGetCommodityList(P p);

    /**
     * 商品列表描述信息
     * @return 商品列表描述信息
     */
    abstract protected String description();


}
