package com.bxm.fossicker.commodity.service.commodity.list.impl;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bxm.fossicker.commodity.common.utils.OkHttpUtils;
import com.bxm.fossicker.commodity.common.utils.SignMD5Util;
import com.bxm.fossicker.commodity.model.dto.CommodityListItemResult;
import com.bxm.fossicker.commodity.model.dto.DtkCategorySearchResult;
import com.bxm.fossicker.commodity.model.param.CategoryCommodityListParam;
import com.bxm.fossicker.commodity.model.param.CommodityDetailQueryParam;
import com.bxm.fossicker.commodity.model.vo.CategoryCommodityListItem;
import com.bxm.fossicker.commodity.service.commodity.list.AbstractCommodityListService;
import com.google.common.collect.Lists;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.TreeMap;
import java.util.stream.Collectors;

/**
 * 大淘客分类查询列表
 * @Author: Gonzo
 * @date  2020-03-03 15:46
 * @since 1.6.0
 */
@Service
public class CategoryCommodityListService extends AbstractCommodityListService<CategoryCommodityListItem,
        CategoryCommodityListParam> {

    @Override
    protected List<CategoryCommodityListItem> doGetCommodityList(CategoryCommodityListParam param) {


        // 查询接口
        List<DtkCategorySearchResult> categorySearchResults = dtkCategorySearchFromApi(param);
        // 转换对象
        return categorySearchResults.stream()
                // 库内获取数据
                .map(p -> {
                    CommodityDetailQueryParam queryParam = initCommodityDetailQueryParam(param, p.getGoodsId());

                    // 需要只查询本地信息
                    queryParam.setQueryLocalOnly(true);

                    CommodityListItemResult result = commodityListItemBaseInfo(queryParam);

                    // 校验商品信息是否获取成功
                    if (!checkCommodityListBaseItemSuccess(result)) {
                        return null;
                    }

                    CategoryCommodityListItem listItem = new CategoryCommodityListItem();
                    BeanUtils.copyProperties(result.getListItem(), listItem);

                    if (Objects.nonNull(p.getQuanMLink())) {
                        //获取定金
                        listItem.setHandsel(p.getQuanMLink());
                    }

                    return listItem;

                })
                // 过滤空的
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
    }

    /**
     * 大淘客分类搜索
     * @param param param
     * @return 大淘客分类搜索
     */
    private List<DtkCategorySearchResult> dtkCategorySearchFromApi(CategoryCommodityListParam param) {
        Map<String, Object> paramMapper = searchParam(param);
        String jsonStr = "";
        try {
            jsonStr = OkHttpUtils.get(commodityInfoProperties.getDtkCategorySearchUrl(), paramMapper);
            // 数据为空时请求三次
            if (StringUtils.isBlank(jsonStr)) {
                for (int i = 1; i <= 3; i++) {
                    jsonStr = OkHttpUtils.get(commodityInfoProperties.getDtkCategorySearchUrl(), paramMapper);
                    if (StringUtils.isNotBlank(jsonStr)) {
                        break;
                    }
                }
            }
        } catch(Exception e) {
            log.error("查询大淘客接口失败", e);
            return Lists.newArrayList();
        }

        if (StringUtils.isBlank(jsonStr)) {
            return Lists.newArrayList();
        }

        List<DtkCategorySearchResult> results = Lists.newArrayList();
        JSONObject dtkJson = JSONObject.parseObject(jsonStr);
        if (Integer.parseInt(dtkJson.getString("code")) == 0 && "成功".equals(dtkJson.getString("msg"))) {
            JSONObject dataJson = JSONObject.parseObject(dtkJson.getString("data"));
            JSONArray dtkArray = JSONArray.parseArray(dataJson.getString("list"));
            for (int i = 0; i < dtkArray.size(); i++) {
                JSONObject job = dtkArray.getJSONObject(i);
                DtkCategorySearchResult result = new DtkCategorySearchResult();
                result.setGoodsId(job.getLong("goodsId"));
                result.setQuanMLink(job.getDouble("quanMLink"));
                results.add(result);
            }
        }

        return results;
    }

    //组装搜索条件
    private Map<String, Object> searchParam(CategoryCommodityListParam param) {
        Map<String, Object> paraMap = new TreeMap<>();
        paraMap.put("version", commodityInfoProperties.getDtkCommodityVersion());
        paraMap.put("appKey", commodityInfoProperties.getDtkCommodityAppKey());
        if (StringUtils.isNotBlank(param.getSortType())) {
            paraMap.put("sort", param.getSortType());
        }
        if (StringUtils.isNotBlank(param.getCids())) {
            //一级类目id
            paraMap.put("cids", param.getCids());
        }
        // 二级类目id，仅允许传一个二级id，当一级类目id和二级类目id同时传入时，会自动忽略二级类目id
        if (StringUtils.isNotBlank(param.getSubcid())) {
            paraMap.put("subcid", param.getSubcid());
        }
        if (StringUtils.isNotBlank(param.getPriceLowerLimit())) {
            //价格下限
            paraMap.put("priceLowerLimit", param.getPriceLowerLimit());
        }
        if (StringUtils.isNotBlank(param.getPriceUpperLimit())) {
            //价格下限
            paraMap.put("priceUpperLimit", param.getPriceUpperLimit());
        }
        if (!Objects.isNull(param.getPageOn())) {
            paraMap.put("pageId", param.getPageOn());
        }
        if (!Objects.isNull(param.getPageSize())) {
            paraMap.put("pageSize", param.getPageSize());
        }
        paraMap.put("sign", SignMD5Util.getSignStr(paraMap, commodityInfoProperties.getDtkCommodityAppSecret()));
        return paraMap;
    }

    @Override
    protected String description() {
        return "分类列表";
    }

    @Override
    public Class<CategoryCommodityListParam> support() {
        return CategoryCommodityListParam.class;
    }
}
