package com.bxm.fossicker.commodity.service.commodity.list.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.bxm.fossicker.commodity.common.constants.CommodityRedisKeyConstants;
import com.bxm.fossicker.commodity.domain.CommodityInfoMapper;
import com.bxm.fossicker.commodity.model.constant.CommodityDetailType;
import com.bxm.fossicker.commodity.model.constant.CommodityPoolId;
import com.bxm.fossicker.commodity.model.param.CommodityDetailQueryParam;
import com.bxm.fossicker.commodity.model.param.PoolCommodityListParam;
import com.bxm.fossicker.commodity.model.vo.PoolCommodityListItem;
import com.bxm.fossicker.commodity.service.commodity.list.AbstractCommodityListService;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.google.common.collect.Lists;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 商品池列表
 * @Author: Gonzo
 * @date  2020-03-03 18:36
 * @since 1.6.0
 */
@Service
public class PoolCommodityListService extends AbstractCommodityListService<PoolCommodityListItem,
        PoolCommodityListParam> {

    @Autowired
    private RedisStringAdapter redisStringAdapter;

    @Autowired
    private CommodityInfoMapper commodityInfoMapper;

    @Override
    protected List<PoolCommodityListItem> doGetCommodityList(PoolCommodityListParam param) {

        KeyGenerator key = CommodityRedisKeyConstants.COMMODITY_GOODS_LIST_EXTEND.copy()
                .appendKey(param.getPoolId())
                .appendKey(param.getVip() ? "isVip" : "noVip");

        // 缓存开启
        if (commodityListProperties.getCacheSwitch()) {

            if (log.isDebugEnabled()) {
                log.debug("列表缓存开启，优先查询缓存");
            }

            String topPoolGoodsList = redisStringAdapter.getString(key);
            if (StringUtils.isNotBlank(topPoolGoodsList)) {
                return JSONArray.parseArray(topPoolGoodsList, PoolCommodityListItem.class);
            }
        }

        // 缓存不存在 查询数据库
        List<String> goodsIds = commodityInfoMapper.getPoolInfo(param.getPoolId());
        if (CollectionUtils.isEmpty(goodsIds)) {
            return Lists.newArrayList();
        }

        // 封装数据
        List<PoolCommodityListItem> poolCommodityListItems = goodsIds.stream()
                // 转换成商品对象
                .map(p -> {
                    CommodityDetailQueryParam queryParam = initCommodityDetailQueryParam(param, Long.parseLong(p));

                    // 不只查询本地信息
                    queryParam.setQueryLocalOnly(false);
                    // 是否查询vip 0元购商品信息
                    // 暂时只有在查询vip 0 元购商品池的数据时，才允许vip0元购商品展现
                    // 而这个类是专门提供给商品池获取列表的，所以这里通过判断商品池id 是否是vip 0元购的商品池id
                    // 来决定是否过滤vip 0元购的商品信息
                    queryParam.setDetailType(Objects.equals(param.getPoolId(), CommodityPoolId.VIP_ZERO)
                            ? CommodityDetailType.VIP_ZERO
                            : CommodityDetailType.DEFAULT);

                    return commodityListItemBaseInfo(queryParam);
                })
                // 过滤掉查询失败的、失效的，并下架
                .filter(p -> {
                    if (Objects.isNull(p) || Objects.isNull(p.getListItem())) {
                        return false;
                    }

                    if (!checkCommodityListBaseItemSuccess(p)) {
                        // 如果商品未在库内找到或者佣金校验失败，则从库中移除
                        log.info("商品id: {} 商品池id: {}下架商品", p.getListItem().getGoodsId(), param.getPoolId());
                        commodityPoolService.setPoolCommodityInvalidStatus(param.getPoolId(), p.getListItem().getGoodsId());
                        return false;
                    }

                    return true;
                })
                .map(p -> {
                    PoolCommodityListItem listItem = new PoolCommodityListItem();
                    BeanUtils.copyProperties(p.getListItem(), listItem);
                    listItem.setPoolId(param.getPoolId());
                    return listItem;
                })
                .collect(Collectors.toList());


        if (!CollectionUtils.isEmpty(poolCommodityListItems)) {
            // 存入缓存
            redisStringAdapter.set(key, JSON.toJSONString(poolCommodityListItems),
                    commodityListProperties.getPoolCommodityListCacheTimeSecond());
        }

        return poolCommodityListItems;
    }



    @Override
    protected String description() {
        return "商品池列表";
    }

    @Override
    public Class<PoolCommodityListParam> support() {
        return PoolCommodityListParam.class;
    }
}
