package com.bxm.fossicker.commodity.service.commodity.list.impl;

import com.alibaba.fastjson.JSONArray;
import com.bxm.fossicker.commodity.common.constants.CommodityRedisKeyConstants;
import com.bxm.fossicker.commodity.model.dto.CommodityListItemResult;
import com.bxm.fossicker.commodity.model.param.CommodityDetailQueryParam;
import com.bxm.fossicker.commodity.model.param.RecommendCommodityParam;
import com.bxm.fossicker.commodity.model.vo.BaseCommodityListItem;
import com.bxm.fossicker.commodity.service.commodity.list.AbstractCommodityListService;
import com.bxm.fossicker.commodity.service.externel.TbCommodityService;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.google.common.collect.Lists;
import com.taobao.api.domain.NTbkItem;
import com.taobao.api.request.TbkItemRecommendGetRequest;
import com.taobao.api.response.TbkItemRecommendGetResponse;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 推荐列表
 * @Author: Gonzo
 * @date  2020-03-03 18:35
 * @since 1.6.0
 */
@Service
public class RecommendCommodityListService extends AbstractCommodityListService<BaseCommodityListItem,
        RecommendCommodityParam> {

    @Autowired
    private RedisStringAdapter redisStringAdapter;

    @Autowired
    private TbCommodityService tbCommodityService;

    @Override
    protected List<BaseCommodityListItem> doGetCommodityList(RecommendCommodityParam param) {
        KeyGenerator recommendRedisKey = CommodityRedisKeyConstants.COMMODITY_RECOMMEND.copy()
                .appendKey(param.getCommodityId());

        // 缓存开启
        if (commodityListProperties.getCacheSwitch()) {
            String recommendStr = redisStringAdapter.getString(recommendRedisKey);
            if (StringUtils.isNotBlank(recommendStr)) {
                List<BaseCommodityListItem> redisInfoList = JSONArray.parseArray(recommendStr, BaseCommodityListItem.class);
                if (!CollectionUtils.isEmpty(redisInfoList)) {
                    return redisInfoList;
                }
            }
        }

        // 查找接口
        List<NTbkItem> tbkItems = tbInfoList(param.getCommodityId());
        if (CollectionUtils.isEmpty(tbkItems)) {
            return Lists.newArrayList();
        }

        return tbkItems.stream()
                .map(p -> {
                    CommodityDetailQueryParam queryParam = initCommodityDetailQueryParam(param, p.getNumIid());
                    // 不只查询本地信息
                    queryParam.setQueryLocalOnly(false);
                    return commodityListItemBaseInfo(queryParam);
                })
                // 过滤掉查询失败的、失效的
                .filter(this::checkCommodityListBaseItemSuccess)
                .map(CommodityListItemResult::getListItem)
                .collect(Collectors.toList());
    }



    /**
     * 查询接口
     * @param commodityId 推荐的商品id
     * @return 推荐商品列表
     */
    private List<NTbkItem> tbInfoList(Long commodityId) {
        // 组装参数
        TbkItemRecommendGetRequest req = new TbkItemRecommendGetRequest();
        req.setFields("num_iid,title,pict_url,small_images,reserve_price,zk_final_price,user_type,provcity,item_url");
        req.setNumIid(commodityId);
        req.setCount(10L);
        req.setPlatform(2L);
        //请求淘宝客服务
        TbkItemRecommendGetResponse rsp = tbCommodityService.getRecommendCommodity(req);

        if (Objects.isNull(rsp) || CollectionUtils.isEmpty(rsp.getResults())) {
            log.warn("从淘宝接口获取到的推荐商品为空,商品id:{}", commodityId);
            return null;
        }

        return rsp.getResults();
    }

    @Override
    protected String description() {
        return "推荐列表";
    }

    @Override
    public Class<RecommendCommodityParam> support() {
        return RecommendCommodityParam.class;
    }
}
