package com.bxm.fossicker.commodity.service.commodity.list.impl;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bxm.fossicker.commodity.common.utils.OkHttpUtils;
import com.bxm.fossicker.commodity.config.CommodityInfoProperties;
import com.bxm.fossicker.commodity.model.dto.CommodityListItemResult;
import com.bxm.fossicker.commodity.model.dto.HdkWellChoseCommodityResult;
import com.bxm.fossicker.commodity.model.dto.Index;
import com.bxm.fossicker.commodity.model.param.CommodityDetailQueryParam;
import com.bxm.fossicker.commodity.model.param.WellChoseCommodityListParam;
import com.bxm.fossicker.commodity.model.vo.CategoryCommodityListItem;
import com.bxm.fossicker.commodity.service.commodity.list.AbstractCommodityListService;
import com.google.common.collect.Lists;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.TreeMap;
import java.util.stream.Collectors;

/**
 * 精选列表
 * @Author: Gonzo
 * @date  2020-03-03 14:00
 * @since 1.6.0
 */
@Service
public class WellChoseCommodityListService extends AbstractCommodityListService<CategoryCommodityListItem,
        WellChoseCommodityListParam> {

    @Autowired
    private CommodityInfoProperties commodityInfoProperties;

    @Override
    protected List<CategoryCommodityListItem> doGetCommodityList(WellChoseCommodityListParam param) {
        // 查询精选接口
        List<HdkWellChoseCommodityResult> searchResults = getHdkWellChoseCommodityFromApi(param);

        // 转换对象
        return searchResults.stream()
                // 库内获取数据
                .map(p -> {
                    CommodityDetailQueryParam queryParam = initCommodityDetailQueryParam(param, p.getItemId());
                    // 只查询库内商品
                    queryParam.setQueryLocalOnly(true);
                    // 校验优惠券金额
                    queryParam.setValidateCouponPriceInfo(true);

                    CommodityListItemResult result = commodityListItemBaseInfo(queryParam);

                    // 校验商品信息是否获取成功
                    if (!checkCommodityListBaseItemSuccess(result)) {
                        return null;
                    }

                    CategoryCommodityListItem listItem = new CategoryCommodityListItem();
                    BeanUtils.copyProperties(result.getListItem(), listItem);

                    if (Objects.nonNull(p.getDeposit())) {
                        //获取定金
                        listItem.setHandsel(p.getDeposit());
                    }

                    return listItem;
                })
                // 过滤空的
                .filter(Objects::nonNull)
                .collect(Collectors.toList());

    }

    /**
     * 分页查询好单库精选接口
     * @param param param
     * @return 精选商品id信息
     */
    private List<HdkWellChoseCommodityResult> getHdkWellChoseCommodityFromApi(WellChoseCommodityListParam param) {

        //调用好单库接口装载数据
        Map<String, Object> hdkMap = new TreeMap<>();
        hdkMap.put("apikey", commodityInfoProperties.getHdkCommodityAppKey());
        //sale_type 使用1 实时销量榜
        hdkMap.put("sale_type", "1");
        //item_type 使用1 只获取营销返利商品
        hdkMap.put("item_type", "1");
        //页码
        hdkMap.put("min_id", "1");
        //每页返回条数
        hdkMap.put("back", "100");
        String hdkStr = "";
        try {
            hdkStr = OkHttpUtils.get(commodityInfoProperties.getHdkRankListUrl(), hdkMap);
            //数据为空时，请求三次
            if (StringUtils.isBlank(hdkStr) || hdkStr.length() < 5) {
                for (int i = 1; i <= 3; i++) {
                    hdkStr = OkHttpUtils.get(commodityInfoProperties.getHdkRankListUrl(), hdkMap);
                    if (hdkStr.length() > 5) {
                        break;
                    }
                }
            }
        } catch(Exception e) {
            log.error("请求好单库接口失败", e);
        }

        if (StringUtils.isBlank(hdkStr)) {
            return Lists.newArrayList();
        }

        JSONObject hdkJson = JSONObject.parseObject(hdkStr);
        List<HdkWellChoseCommodityResult> results = Lists.newArrayList();
        // 成功
        if (Objects.nonNull(hdkJson) && Objects.nonNull(hdkJson.getInteger("code") )
                && hdkJson.getInteger("code") == 1) {

            JSONArray hdkArray = JSONArray.parseArray(hdkJson.getString("data"));
            for (int i = 0; i < hdkArray.size(); i++) {
                // 遍历 jsonarray 数组，把每一个对象转成 json 对象
                JSONObject job = hdkArray.getJSONObject(i);

                HdkWellChoseCommodityResult result = new HdkWellChoseCommodityResult();
                result.setItemId(job.getLong("itemid"));
                result.setDeposit(job.getDouble("deposit"));

                results.add(result);
            }
        }

        // 分页截取
        Index index = getIndex(param.getPageOn(), param.getPageSize(), results.size());
        results = results.subList(index.getStart(), index.getEnd());

        return results;
    }

    @Override
    protected String description() {
        return "精选列表";
    }

    @Override
    public Class<WellChoseCommodityListParam> support() {
        return WellChoseCommodityListParam.class;
    }
}
