package com.bxm.fossicker.commodity.service.externel.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bxm.fossicker.commodity.common.constants.CommdityMsgCode;
import com.bxm.fossicker.commodity.common.constants.SearchType;
import com.bxm.fossicker.commodity.common.enums.CommoditySortTypeEnum;
import com.bxm.fossicker.commodity.common.utils.OkHttpUtils;
import com.bxm.fossicker.commodity.common.utils.SignMD5Util;
import com.bxm.fossicker.commodity.config.CommodityInfoProperties;
import com.bxm.fossicker.commodity.model.dto.WyCommissionDataDTO;
import com.bxm.fossicker.commodity.model.dto.WyQueryResultDTO;
import com.bxm.fossicker.commodity.model.dto.WySearchCommodityInfoDTO;
import com.bxm.fossicker.commodity.model.dto.WySearchResultDTO;
import com.bxm.fossicker.commodity.model.param.CommoditySearchParam;
import com.bxm.fossicker.commodity.model.param.GuessCommodityQueryParam;
import com.bxm.fossicker.commodity.service.externel.QueryWithRetryService;
import com.bxm.fossicker.commodity.service.externel.WyCommodityService;
import com.bxm.newidea.component.apollo.i18n.MessageHelper;
import com.bxm.newidea.component.tools.StringUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.IOException;
import java.util.*;

/**
 * 维易接口实现类
 *
 * @author Dong_hy
 */
@Service
@Slf4j
public class WyCommodityServiceImpl implements WyCommodityService {

    private final QueryWithRetryService queryWithRetryService;

    private final CommodityInfoProperties commodityInfoProperties;

    /**
     * 维易返回状态码-成功
     */
    private static String SUCCESS = "0";

    /**
     * 维易返回状态码-检索信息不存在对应的商品
     */
    private static String SEARCH_WITHOUT_RES = "15";

    @Autowired
    public WyCommodityServiceImpl(QueryWithRetryService queryWithRetryService,
                                  CommodityInfoProperties commodityInfoProperties) {
        this.queryWithRetryService = queryWithRetryService;
        this.commodityInfoProperties = commodityInfoProperties;
    }

    @Override
    public WySearchResultDTO searchCommodity(CommoditySearchParam commonditySearchParam) {
        String queryUrl = commodityInfoProperties.getWySearchUrl();
        Map<String, Object> searchParam = assembleSearchParam(commonditySearchParam);

        String searchResultStr;
        try {
            searchResultStr = OkHttpUtils.get(queryUrl, searchParam);
        } catch (IOException e) {
            log.error(e.getMessage(), e);
            return null;
        }
        if (StringUtils.isBlank(searchResultStr)) {
            return null;
        }
        WyQueryResultDTO wyQueryResultDTO = JSON.parseObject(searchResultStr, WyQueryResultDTO.class);

        //list和data都为空 才会是请求数据为空.searchType为1时,数据在list中,searchType为2时在data中
        if (Objects.isNull(wyQueryResultDTO)) {
            log.info("维易查询接口不存在数据，参数：[{}]，结果:[{}]", commonditySearchParam, searchResultStr);
            //请求失败
            return null;
        }

        //查询成功状态码
        if (SUCCESS.equals(wyQueryResultDTO.getError())) {
            WySearchResultDTO result = WySearchResultDTO.builder()
                    .searchType(wyQueryResultDTO.getSearchType())
                    .commodityList(getResultList(wyQueryResultDTO))
                    .build();
            if (StringUtils.isBlank(result.getSearchType())) {
                result.setSearchType(SearchType.CONVERT_LINK);
            }
            return result;
        } else if (SEARCH_WITHOUT_RES.equals(wyQueryResultDTO.getError())) {
            log.info("搜索的商品在维易平台不存在淘客商品，搜索参数：{}", commonditySearchParam);
            String msg = MessageHelper.getMessageWithDefault(CommdityMsgCode.WEIYI_SEARCH_WITHOUT_RES,
                    "该宝贝已下架或非淘客宝贝");
            return WySearchResultDTO.builder()
                    .msg(msg)
                    .searchType(SearchType.SEARCH_WITHOUT_RES)
                    .build();
        }

        log.warn("维易搜索接口异常，搜索参数：{},搜索结果：{}", commonditySearchParam, searchResultStr);
        return null;
    }

    @Override
    public List<WySearchCommodityInfoDTO> guessLike(GuessCommodityQueryParam guessCommodityQueryParam) {
        String queryUrl = commodityInfoProperties.getWyGuessUrl();
        Map<String, Object> guessParam = assembleGuessParam(guessCommodityQueryParam);


        String searchResultStr;
        try {
            searchResultStr = queryWithRetryService.doGetWithRetry(queryUrl, guessParam);
        } catch (Exception e) {
            log.error("请求异常,请求路径为:{}, 请求参数为:{}, 异常为:{}", queryUrl, JSON.toJSONString(guessParam), e);
            return null;
        }

        WyQueryResultDTO wyGuessResultDTO =
                JSON.parseObject(searchResultStr, WyQueryResultDTO.class);
        if (Objects.isNull(wyGuessResultDTO) || !SUCCESS.equals(wyGuessResultDTO.getError())
                || Objects.isNull(wyGuessResultDTO.getData())) {
            //请求失败
            return null;
        }
        return JSONArray.parseArray(wyGuessResultDTO.getData().toString(), WySearchCommodityInfoDTO.class);
    }

    @Override
    public WyCommissionDataDTO getCommissionInfo(Long goodsId) {
        String queryUrl = commodityInfoProperties.getWyCommissionUrl();
        Map<String, Object> paramMap = assembleCommissionParam(goodsId);

        String searchResultStr;
        try {
            searchResultStr = queryWithRetryService.doGetWithRetry(queryUrl, paramMap);
        } catch (Exception e) {
            log.error("请求异常,请求路径为:{}, 请求参数为:{}, 异常为:{}", queryUrl, JSON.toJSONString(paramMap), e);
            return null;
        }

        WyQueryResultDTO wyCommissionResultDTO =
                JSON.parseObject(searchResultStr, WyQueryResultDTO.class);

        if (Objects.isNull(wyCommissionResultDTO) || Objects.isNull(wyCommissionResultDTO.getData())) {
            return null;
        }

        if (log.isDebugEnabled()) {
            log.debug("wy获取高佣信息：[{}]", JSONObject.toJSON(wyCommissionResultDTO));
        }

        return JSON.parseObject(wyCommissionResultDTO.getData().toString(), WyCommissionDataDTO.class);
    }

    private Map<String, Object> assembleSearchParam(CommoditySearchParam commoditySearchParam) {
        Map<String, Object> paramMap = new HashMap<>(8);
        paramMap.put("para", commoditySearchParam.getCommodityName());
        paramMap.put("vekey", "V00002250Y21499601");
        paramMap.put("pagesize", commoditySearchParam.getPageSize());
        paramMap.put("page", commoditySearchParam.getPageNum());
        paramMap.put("coupon", commoditySearchParam.getCouponFlag());
        if (StringUtils.isNotBlank(commoditySearchParam.getSortType())) {
            //根据code转换分类类型
            paramMap.put("sort", CommoditySortTypeEnum.getTypeByCode(commoditySearchParam.getSortType()));
        }
        if (null != commoditySearchParam.getPriceLowerLimit()) {
            paramMap.put("start_price", commoditySearchParam.getPriceLowerLimit());
        }

        if (null != commoditySearchParam.getPriceUpperLimit()) {
            paramMap.put("end_price", commoditySearchParam.getPriceUpperLimit());
        }


        return paramMap;

    }

    private Map<String, Object> assembleGuessParam(GuessCommodityQueryParam guessCommodityQueryParam) {
        Map<String, Object> paramMap = new HashMap<>(8);


        paramMap.put("vekey", "V00002250Y21499601");
        paramMap.put("device_type", guessCommodityQueryParam.getDeviceType());
        paramMap.put("device_value", StringUtils.isNotBlank(guessCommodityQueryParam.getDeviceNo())
                ? SignMD5Util.MD5(guessCommodityQueryParam.getDeviceNo())
                : "");
        paramMap.put("device_encrypt", "MD5");
        paramMap.put("page_no", guessCommodityQueryParam.getPageNum());
        paramMap.put("page_size", guessCommodityQueryParam.getPageSize());


        return paramMap;

    }

    private Map<String, Object> assembleCommissionParam(Long goodsId) {
        Map<String, Object> paramMap = new HashMap<>(4);

        paramMap.put("para", goodsId);
        paramMap.put("vekey", "V00002250Y21499601");


        return paramMap;

    }


    private List<WySearchCommodityInfoDTO> getResultList(WyQueryResultDTO wyQueryResultDTO) {
        if (SearchType.KEY_WORDS.equals(wyQueryResultDTO.getSearchType())) {
            return wyQueryResultDTO.getResultList();
        }
        if (SearchType.CONVERT_LINK.equals(wyQueryResultDTO.getSearchType())) {
            List<WySearchCommodityInfoDTO> resultList = new ArrayList<>();
            resultList.add(JSON.parseObject(wyQueryResultDTO.getData().toString(), WySearchCommodityInfoDTO.class));
            return resultList;
        }
        return null;
    }
}
