package com.bxm.fossicker.commodity.service.impl;

import com.bxm.fossicker.commodity.common.dto.PlatformCommissionRateInfo;
import com.bxm.fossicker.commodity.common.utils.CommodityCalUtils;
import com.bxm.fossicker.commodity.model.dto.CommodityCouponInfoFromSuper;
import com.bxm.fossicker.commodity.model.dto.CommodityInfoDTO;
import com.bxm.fossicker.commodity.model.entry.CommodityInfo;
import com.bxm.fossicker.commodity.model.vo.CommodityCouponInfo;
import com.bxm.fossicker.commodity.service.CommodityCouponInfoService;
import com.bxm.fossicker.commodity.service.commodity.info.CommodityInfoSourceProxyService;
import com.bxm.fossicker.commodity.service.CommodityPriceInfoService;
import com.bxm.fossicker.constant.CommonCommodityRedisKey;
import com.bxm.fossicker.integration.dto.CommodityCouponInfoDTO;
import com.bxm.fossicker.integration.param.GetCouponInfoParam;
import com.bxm.fossicker.integration.service.CommoditySyncIntegrationService;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Objects;

@Service
@Slf4j
public class CommodityCouponInfoServiceImpl implements CommodityCouponInfoService {

    @Autowired
    private CommoditySyncIntegrationService commoditySyncIntegrationService;

    @Autowired
    private RedisHashMapAdapter redisHashMapAdapter;

    @Autowired
    private CommodityInfoSourceProxyService commodityInfoSourceProxyService;

    @Autowired
    private CommodityPriceInfoService commodityPriceInfoService;

    @Override
    public CommodityCouponInfo getCommodityCouponInfo(GetCouponInfoParam param) {

        CommodityCouponInfo info = new CommodityCouponInfo();
        info.setGoodsId(param.getGoodsId());
        // 设置来源，因为有不同的三方，决定了从哪个三方获取
        setSourceFromIfNecessary(param);

        CommodityCouponInfoDTO commodityCouponInfoDTO = commoditySyncIntegrationService.loadCouponInfoFromCache(param);

        if (Objects.nonNull(commodityCouponInfoDTO)) {
            // 是否转换协议
            info.setCouponUrl(param.isReplaceHttp2TbOpen() ?
                    assembleCouponUrlByType(commodityCouponInfoDTO.getCouponUrl())
                    : commodityCouponInfoDTO.getCouponUrl() );
        }

        return info;
    }

    private void setSourceFromIfNecessary(GetCouponInfoParam param) {

        if (Objects.isNull(param.getSourceFrom())) {

            // 从redis获取
            KeyGenerator keyGenerator = CommonCommodityRedisKey.LOCAL_COMMODITY_INFO.copy()
                    .appendKey(Math.abs(param.getGoodsId().hashCode()) % 10);

            CommodityInfo commodityInfo = redisHashMapAdapter.get(keyGenerator, param.getGoodsId(), CommodityInfo.class);

            if (Objects.nonNull(commodityInfo)) {
                param.setSourceFrom(commodityInfo.getSoruce());
            }
        }
    }


    @Override
    public String assembleCouponUrlByType(String couponUrl) {

        if (StringUtils.isBlank(couponUrl)) {
            return couponUrl;
        }

        return couponUrl.replace("https", "tbopen");
    }

    @Override
    public CommodityCouponInfoFromSuper getCouponInfo(Long goodsId, Long userId) {

        CommodityInfoDTO commodityInfoDTO = commodityInfoSourceProxyService.getCommodityDetail(goodsId);

        if (null == commodityInfoDTO) {
            log.error("获取商品为空,商品id:{}", goodsId);
            return null;
        }

        CommodityCouponInfoFromSuper resultInfo = new CommodityCouponInfoFromSuper();

        // 分享赚和佣金优惠券信息
        PlatformCommissionRateInfo cInfo = commodityPriceInfoService.getPlatformCommissionRateInfo(goodsId);

        if (null != cInfo) {
            Double commissionPrice = CommodityCalUtils.calCommissionForPlatformCommission(cInfo,
                    commodityInfoDTO.getPlatformCommissionPrice());

            resultInfo.setCommissionPrice(commissionPrice);
            resultInfo.setCouponPrice(commodityInfoDTO.getCouponPrice());
            resultInfo.setCouponUrl(commodityInfoDTO.getCouponUrl());
        }

        return resultInfo;
    }
}
