package com.bxm.fossicker.commodity.service.impl;

import com.bxm.fossicker.activity.facade.TaskFacadeService;
import com.bxm.fossicker.activity.facade.enums.ActivityFacadeEnum;
import com.bxm.fossicker.commodity.common.constants.CommodityRedisKeyConstants;
import com.bxm.fossicker.commodity.common.enums.CommodityDrawGoldStatusEnum;
import com.bxm.fossicker.commodity.config.CommodityInfoProperties;
import com.bxm.fossicker.commodity.facade.CommodityGoldFacadeService;
import com.bxm.fossicker.commodity.model.dto.CommodityGoldInfoDTO;
import com.bxm.fossicker.commodity.model.param.BrowseCommodityParam;
import com.bxm.fossicker.commodity.service.BrowseFlowService;
import com.bxm.fossicker.commodity.service.CommodityGoldService;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;

/**
 * 商品金币服务
 *
 * @author donghy
 */
@Service
public class CommodityGoldServiceImpl implements CommodityGoldService, CommodityGoldFacadeService {

    /**
     * REDIS string服务
     */
    @Autowired
    private RedisStringAdapter redisStringAdapter;

    /**
     * 商品信息配置服务
     */
    @Autowired
    private CommodityInfoProperties commodityInfoProperties;


    @Autowired
    private BrowseFlowService browseFlowService;

    @Autowired
    private TaskFacadeService taskFacadeService;

    @Override
    public CommodityGoldInfoDTO getGoldInfoByCommodityId(Long commodityId, Long userId) {
        //userId为空的话,默认显示金币为可领
        if (null == userId) {
            return CommodityGoldInfoDTO.builder()
                    //设置是否可领取金币
                    .drawFlag(1)
                    //设置可领取的金币数额
                    .browseGold(getGoldByCommodity())
                    .commodityId(commodityId).build();
        }


        return CommodityGoldInfoDTO.builder()
                //设置是否可领取金币
                .drawFlag(isAvailableBrowseGold(commodityId, userId))
                //设置可领取的金币数额
                .browseGold(getGoldByCommodity())
                .commodityId(commodityId)
                .build();
    }

    @Override
    public Integer getBrowseGoldNum() {
        return Integer.valueOf(commodityInfoProperties.getBrowseCommonGold());
    }

    @Override
    public void addLimitTimesByDoubleGold(Long userId) {
        //翻倍奖励自增步数2
        KeyGenerator limitKey = CommodityRedisKeyConstants.COMMODITY_BROWSE_GOLD_LIMIT.copy().appendKey(userId.toString()).appendKey(DateUtils.getCurrentDate());
        ;
        redisStringAdapter.incrementWithDefault(limitKey, 1L, 2, 172800L);
    }

    @Override
    public Boolean getRewardByBrowse(BrowseCommodityParam browseCommodityParam) {
        Long userId = browseCommodityParam.getUserId();
        Long goodsId = browseCommodityParam.getGoodsId();
        //hotfix-是否可以领金币校验以及1天100次校验
        if (!isAvailableGold(browseCommodityParam.getGoodsId(), browseCommodityParam.getUserId())) {
            return false;
        }


        //添加浏览记录
        browseFlowService.addBrowseCommodityRecord(browseCommodityParam);

        //存浏览记录至redis
        saveBrowseCommodityRecordToRedis(userId.toString(), browseCommodityParam.getGoodsId().toString());


        //hotfix 领取即增加
        KeyGenerator limitKey = CommodityRedisKeyConstants.COMMODITY_BROWSE_GOLD_LIMIT.copy().appendKey(userId.toString()).appendKey(DateUtils.getCurrentDate());
        ;
        redisStringAdapter.increment(limitKey, 172800L);

        //调用完成任务
        return taskFacadeService.complete(userId, ActivityFacadeEnum.FINISH_VIEW);
    }

    @Override
    public Boolean isAvailableGold(Long goodsId, Long userId) {
        return CommodityDrawGoldStatusEnum.AVAILABLE.getStatus().equals(isAvailableBrowseGold(goodsId, userId));
    }


    /**
     * 获取是否可领取金币
     *
     * @param commodityId 商品id
     * @param userId      用户id
     * @return 可领取返回1, 不可领取返回0
     */
    private Integer isAvailableBrowseGold(Long commodityId, Long userId) {
        if (null == commodityId) {
            return CommodityDrawGoldStatusEnum.UNAVAILABLE.getStatus();
        }

        KeyGenerator browseCommodityKey = CommodityRedisKeyConstants.COMMODITY_BROWSE_RECORD.copy()
                .appendKey(userId.toString()).appendKey(commodityId.toString());


        String browseRecord = redisStringAdapter.getString(browseCommodityKey);

        KeyGenerator limitKey = CommodityRedisKeyConstants.COMMODITY_BROWSE_GOLD_LIMIT.copy().appendKey(userId.toString()).appendKey(DateUtils.getCurrentDate());
        Integer maxDrawGoldTimes = commodityInfoProperties.getMaxDrawGoldTimes();
        if (maxDrawGoldTimes < redisStringAdapter.getInt(limitKey)) {
            return CommodityDrawGoldStatusEnum.UNAVAILABLE.getStatus();
        }


        //有领取记录,则返回不可领
        if (StringUtils.isNotBlank(browseRecord)) {
            return CommodityDrawGoldStatusEnum.UNAVAILABLE.getStatus();
        }

        return CommodityDrawGoldStatusEnum.AVAILABLE.getStatus();

    }

    /**
     * 获取可领取的商品浏览金币数
     *
     * @return 商品浏览金币数
     */
    private Integer getGoldByCommodity() {
        return commodityInfoProperties.getBrowseCommonGold();
    }

    /**
     * 存浏览记录至redis
     *
     * @param userId  用户id
     * @param goodsId 商品id
     */
    private void saveBrowseCommodityRecordToRedis(String userId, String goodsId) {

        //设置浏览记录到redis
        KeyGenerator browseCommodityKey = CommodityRedisKeyConstants.COMMODITY_BROWSE_RECORD.copy().appendKey(userId).appendKey(goodsId);

        redisStringAdapter.set(browseCommodityKey, "record", calCountDownTimeByToday());

    }

    /**
     * 计算与当天时间的剩余数
     *
     * @return
     */
    private Long calCountDownTimeByToday() {
        LocalTime localTime = LocalTime.of(23, 59, 59);

        LocalDateTime todayEndTime = LocalDateTime.of(LocalDate.now(), localTime);

        return Duration.between(LocalDateTime.now(), todayEndTime).toMillis() / 1000;
    }


}
