package com.bxm.fossicker.commodity.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.fossicker.activity.facade.TaskFacadeService;
import com.bxm.fossicker.activity.facade.enums.ActivityFacadeEnum;
import com.bxm.fossicker.commodity.common.constants.CommodityRedisKeyConstants;
import com.bxm.fossicker.commodity.common.constants.SearchType;
import com.bxm.fossicker.commodity.common.dto.CommissionAndCoupon;
import com.bxm.fossicker.commodity.common.dto.PlatformCommissionRateInfo;
import com.bxm.fossicker.commodity.common.enums.LocalCommodityStatusEnum;
import com.bxm.fossicker.commodity.common.utils.CommodityCalUtils;
import com.bxm.fossicker.commodity.common.utils.OkHttpUtils;
import com.bxm.fossicker.commodity.config.AdvertTicketProperties;
import com.bxm.fossicker.commodity.config.CommodityInfoProperties;
import com.bxm.fossicker.commodity.facade.AdvertPlaceService;
import com.bxm.fossicker.commodity.model.constant.CommodityDetailResultCode;
import com.bxm.fossicker.commodity.model.constant.CommodityDetailType;
import com.bxm.fossicker.commodity.model.constant.CommodityPoolId;
import com.bxm.fossicker.commodity.model.dto.*;
import com.bxm.fossicker.commodity.model.param.*;
import com.bxm.fossicker.commodity.model.vo.BaseCommodityListItem;
import com.bxm.fossicker.commodity.service.*;
import com.bxm.fossicker.commodity.service.commodity.info.CommodityInfoSourceProxyService;
import com.bxm.fossicker.commodity.service.externel.DtkCommodityService;
import com.bxm.fossicker.commodity.service.externel.HdkCommodityService;
import com.bxm.fossicker.commodity.service.externel.TbCommodityService;
import com.bxm.fossicker.commodity.service.strategy.CommodityLabelService;
import com.bxm.fossicker.constant.LocalCommoditySourceFrom;
import com.bxm.fossicker.integration.service.CrawlerIntegrationService;
import com.bxm.fossicker.user.facade.vip.VipFacadeService;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.io.IOException;
import java.math.BigDecimal;
import java.util.*;

/**
 * 商品信息服务实现类
 *
 * @author donghy
 */
@Service
@Slf4j
public class CommodityInfoServiceImpl implements CommodityInfoService, AdvertPlaceService {

    @Autowired
    private CommodityCouponInfoService commodityCouponInfoService;

    @Autowired
    private RedisStringAdapter redisStringAdapter;

    @Autowired
    private RedisHashMapAdapter redisHashMapAdapter;

    @Autowired
    private CrawlerIntegrationService crawlerIntegrationService;

    @Autowired
    private CommodityCollectService commodityCollectService;

    @Autowired
    private BrowseFlowService browseFlowService;

    @Autowired
    private CommodityPriceInfoService commodityPriceInfoService;

    @Autowired
    private CommodityGoldService commodityGoldService;

    @Autowired
    private TaskFacadeService taskFacadeService;

    @Autowired
    private TbCommodityService tbCommodityService;

    @Autowired
    private SecKillCommodityService secKillCommodityService;

    @Autowired
    private DtkCommodityService dtkCommodityService;

    @Autowired
    private BlackCommodityService blackCommodityService;

    @Autowired
    private CommodityInfoSourceProxyService commodityInfoSourceProxyService;

    @Autowired
    private CommodityInfoProperties commodityInfoProperties;

    @Autowired
    private AdvertTicketProperties advertTicketProperties;

    @Autowired
    private HistoryBrowsingService historyBrowsingService;

    @Autowired
    CommodityInfoSourceProxyService commodityInfoSourceService;

    @Autowired
    private HdkCommodityService hdkCommodityService;

    @Autowired
    private VipFacadeService vipFacadeService;

    @Autowired
    private CommodityLabelService commodityLabelService;

    @Autowired
    private CommodityPoolService commodityPoolService;


    @Override
    public CommodityDetailDTO getCommodityDetail(CommodityDetailParam param) {
        CommodityDetailQueryParam param1 = new CommodityDetailQueryParam();
        // 复制基础参数
        BeanUtils.copyProperties(param, param1);

        param1.setGoodsId(param.getGoodsId());
        param1.setUserId(param.getUserId());
        param1.setH5Flag(param.isH5Flag());
        // 查询详情的类型
        param1.setDetailType(param.getDetailType());
        // 详情需要校验价格信息 如果金额不对则返回valid = 0
        param1.setValidateCommissionPriceInfo(true);

        param1.setSaveHistory(true);
        param1.setSetLable(true);
        param1.setQueryBrowerAward(true);
        param1.setQueryCollect(true);
        param1.setQueryNewbie(true);
        param1.setQueryRemind(true);
        param1.setQueryDetail(true);
        param1.setQueryShare(true);
        param1.setQueryCouponInfo(true);


        return getCommodityDetail(param1);
    }

    @Override
    public CommodityDetailDTO getCommodityDetail(CommodityDetailQueryParam param) {

        // 调用查询方法 查询商品信息
        CommodityDetailResult result = getCommodityDetailWithResultCode(param);

        // 如果查询成功则直接返回信息
        if (Objects.equals(result.getResultCode(), CommodityDetailResultCode.SUCCESS)) {
            return result.getDetail();
        }

        if (!param.isDoNotPrintWarnLog()) {
            log.warn("获取商品详情错误: {}, 查询参数: {}", result.getResultCode().getDesc(), JSON.toJSONString(param));
        }

        // 不成功 但是有数据
        if (param.isIfNotValidThenReturnBaseInfo() && Objects.nonNull(result.getDetail())) {
            return result.getDetail();
        }

        // 原本的查询不到是会返回null的
        if (Objects.equals(result.getResultCode(), CommodityDetailResultCode.LOCAL_DOWN)
                || Objects.equals(result.getResultCode(), CommodityDetailResultCode.NOT_IN_LOCAL)
                || Objects.equals(result.getResultCode(), CommodityDetailResultCode.NOT_FOUND)) {
            return null;
        }

        return CommodityDetailDTO.builder().validStatus(0).build();
    }

    /**
     * 获取商品基本信息，根据维易信息转换
     * @param param 查询信息
     * @return 根据维易信息转换的商品对象
     */
    private CommodityDetailResult doGetCommodityDetailFromGuessLikeWyInfo(CommodityDetailQueryParam param) {
        WySearchCommodityInfoDTO wyInfo = param.getWySearchCommodityInfoParam().getWyInfo();

        // 原价
        Double reservePrice = Double.valueOf(wyInfo.getZkFinalPrice());

        // 基本信息
        CommodityDetailDTO detailDTO = CommodityDetailDTO.builder()
                .commodityPrice(reservePrice)
                .reservePrice(reservePrice)
                .commodityTitle(wyInfo.getShortTitle())
                .goodsId(wyInfo.getItemId())
                .mainPic(assembleHttps(wyInfo.getPictUrl()))
                .salesNum(wyInfo.getVolume().intValue())
                .shopName(wyInfo.getShopTitle())
                .shopType(Objects.isNull(wyInfo.getUserType()) ? 0 : wyInfo.getUserType())
                // 数据来源 三方
                .source(LocalCommoditySourceFrom.THIREDPARTY)
                .validStatus(1)
                .build();


        // 查询高佣接口，获取优惠券和佣金还有平台的佣金信息
        CommissionAndCoupon commissionAndCoupon =
                commodityPriceInfoService.getCommodityCommissionAndCoupon(wyInfo.getItemId());

        // 商品优惠券、佣金获取失败
        if (Objects.isNull(commissionAndCoupon)) {
            return buildResultCode(CommodityDetailResultCode.PRICE_INFO_NOT_FOUND);
        }

        // 佣金
        Double platformCommissionPrice = CommodityCalUtils.calPlatformCommission(commissionAndCoupon, reservePrice);
        // 设置平台佣金
        detailDTO.setPlatformCommissionPrice(platformCommissionPrice);


        return CommodityDetailResult.builder()
                .resultCode(CommodityDetailResultCode.SUCCESS)
                .detail(detailDTO)
                .build();
    }

    private CommodityDetailResult doGetCommodityDetailFromSearchWyInfo(CommodityDetailQueryParam param) {
        WySearchCommodityInfoDTO wySearchCommodityInfoDTO = param.getWySearchCommodityInfoParam().getWyInfo();

        //优惠券信息、商品原价以及第三方佣金比例直接从第三方获取--减少请求第三方提高速度
        BigDecimal rate = new BigDecimal("10000");
        // search_type = 2时，这时返回值中的commission_rate字段不需要除以100，是正常佣金百分值。如50.0表示50%
        if (Objects.equals(param.getWySearchCommodityInfoParam().getSearchType(), SearchType.CONVERT_LINK)) {
            log.info("定向搜索类型，使用正常比例计算");
            rate = new BigDecimal("100");
        }

        // 商品原价
        Double reservePrice = Double.valueOf(wySearchCommodityInfoDTO.getZkFinalPrice());
        // 优惠券价格
        Double couponPrice = CommodityCalUtils.getCouponPriceByRegex(wySearchCommodityInfoDTO.getCouponInfo());
        // 券后价
        Double discountPrice = CommodityCalUtils.calDiscountPrice(reservePrice, couponPrice);
        // 平台佣金
        Double platformCommission = CommodityCalUtils
                .calPlatformCommission(new BigDecimal(param.getWySearchCommodityInfoParam().getWyInfo().getCommissionRate())
                        .divide(rate, 2, BigDecimal.ROUND_DOWN),
                        BigDecimal.valueOf(couponPrice),
                        BigDecimal.valueOf(reservePrice));

        // 构建商品信息
        CommodityDetailDTO detailDTO = CommodityDetailDTO.builder()
                .goodsId(param.getGoodsId())
                .mainPic(wySearchCommodityInfoDTO.getPictUrl())
                .commodityTitle(Optional.ofNullable(wySearchCommodityInfoDTO.getShortTitle()).orElse(wySearchCommodityInfoDTO.getTitle()))
                .reservePrice(reservePrice)
                .shopType(wySearchCommodityInfoDTO.getUserType())
                .shopName(wySearchCommodityInfoDTO.getNick())
                // 平台佣金值
                .platformCommissionPrice(platformCommission)
                // 券后价
                .discountPrice(discountPrice)
                // 优惠券价格--根据优惠券信息正则截取得到
                .couponPrice(couponPrice)
                // 来源
                .source(LocalCommoditySourceFrom.THIREDPARTY)
                .salesNum(wySearchCommodityInfoDTO.getVolume().intValue())
                .validStatus(1)
                .build();

        return CommodityDetailResult.builder()
                .detail(detailDTO)
                .resultCode(CommodityDetailResultCode.SUCCESS)
                .build();
    }

    /**
     * 查询商品基本信息
     * 根据请求参数，从库中 or 三方 or 带进来的对象信息进行转换得到商品基本信息
     * @param param param
     * @return 商品信息
     */
    private CommodityDetailResult getCommodityBaseDetail(CommodityDetailQueryParam param) {

        // 是否需要从维易对象中获取基本数据
        if (Objects.nonNull(param.getWySearchCommodityInfoParam())) {

            if (Objects.nonNull(param.getWySearchCommodityInfoParam().getConvertType())) {
                if (param.getWySearchCommodityInfoParam().getConvertType().intValue() == 1) {
                    // 搜索
                    return doGetCommodityDetailFromSearchWyInfo(param);
                } else if(param.getWySearchCommodityInfoParam().getConvertType().intValue() ==  2) {
                    // 猜你喜欢
                    return doGetCommodityDetailFromGuessLikeWyInfo(param);
                }
            }

            return buildResultCode(CommodityDetailResultCode.WY_INFO_ERROR);
        }

        // 否则查询库内 or 爬虫

        // 版本 < 1.5.0（老版本的优惠券信息是在商品详情的） 或者是站外h5都要查询优惠券
        boolean queryCoupon = param.isH5Flag()
                ||  (StringUtils.isNotBlank(param.getCurVer())
                // 版本 < 1.5.0
                && com.bxm.newidea.component.tools.StringUtils.compareVersion(param.getCurVer(), "1.5.0") < 0);

        // 设置优惠券是否查询
        param.setQueryCouponInfo(param.isQueryCouponInfo() && queryCoupon);

        // 查询商品信息 库内 or 三方
        CommodityInfoDTO commodityInfoDTO = commodityInfoSourceService.getCommodityDetail(GetCommodityDetailParam.builder()
                .commodityId(param.getGoodsId())
                // 是否查询优惠券信息
                .queryCouponInfo(param.isQueryCouponInfo())
                // 是否库内下架则返回null 如果只查询库内商品则这这个查询参数需要为true
                .ifLocalDownThenReturnNull(param.isQueryLocalOnly())
                .build());

        // 如果只查询本地
        if (param.isQueryLocalOnly()) {
            // 本地库不存在则返回
            if (Objects.isNull(commodityInfoDTO)) {
                return buildResultCode(CommodityDetailResultCode.NOT_IN_LOCAL);
            }

            // 已下架则返回
            if (LocalCommodityStatusEnum.Valid.getStatus() != commodityInfoDTO.getStatus()) {
                return buildResultCode(CommodityDetailResultCode.LOCAL_DOWN);
            }

            // 否则就是查询本地库存在且上架的状态
        } else {
            // 不限本地库查询

            // 本地和三方都不存在则返回
            if (Objects.isNull(commodityInfoDTO)) {
                return buildResultCode(CommodityDetailResultCode.NOT_FOUND);
            }

            // 如果查询到了，且是本地商品信息
            if (Objects.equals(commodityInfoDTO.getSoruce(), LocalCommoditySourceFrom.HDK)
                    || Objects.equals(commodityInfoDTO.getSoruce(), LocalCommoditySourceFrom.DTK)) {

                // 库内商品 判断是否失效
                if (Objects.equals(commodityInfoDTO.getStatus(), LocalCommodityStatusEnum.InValid.getStatus())) {

                    if (log.isDebugEnabled()) {
                        log.debug("商品: {} 在库内为失效状态，查询三方数据", param.getGoodsId());
                    }
                    commodityInfoDTO = commodityInfoSourceProxyService.getThirdpartyCommodity(GetCommodityDetailParam.builder()
                            .commodityId(param.getGoodsId())
                            // 不查优惠券信息
                            .queryCouponInfo(false)
                            .build());
                }
            }
        }

        CommodityDetailDTO detailInfo;
        if (Objects.isNull(commodityInfoDTO) || Objects.isNull(detailInfo = convert(commodityInfoDTO))) {
            return buildResultCode(CommodityDetailResultCode.NOT_FOUND);
        }

        return CommodityDetailResult.builder()
                .resultCode(CommodityDetailResultCode.SUCCESS)
                .detail(detailInfo)
                .build();
    }

    /**
     * 获取商品的价格信息
     * @param commissionInfo 佣金比例信息
     * @param platformCommissionPrice 平台佣金金额
     * @param reservePrice 原价
     * @param couponPrice 优惠券价格
     * @param isVip 是否vip
     * @return 商品的价格信息
     */
    private CommodityPrice getCommodityPrice(PlatformCommissionRateInfo commissionInfo,
                                             Double platformCommissionPrice, Double reservePrice, Double couponPrice,
                                             boolean isVip) {

        // 普通佣金
        Double commissionPrice = CommodityCalUtils.calCommissionForPlatformCommissionWithVip(commissionInfo,
                platformCommissionPrice, false);

        // 分享赚的钱 原本是非vip的普通佣金金额 因为下面计算会判断是否是新人商品从而修改佣金值，所以分享赚的钱这里需要记录起来
        Double shareCommissionPrice = commissionPrice;

        // vip佣金
        Double vipCommissionPrice = CommodityCalUtils.calCommissionForPlatformCommissionWithVip(commissionInfo,
                platformCommissionPrice, true);

        // 商品最终购买价格
        Double commodityPrice = CommodityCalUtils.calPriceWithSubtractingCouponAndCommission(reservePrice,
                couponPrice,
                // vip和非vip的返后价不一样
                isVip ? vipCommissionPrice : commissionPrice);

        // 购买省价格
        Double favourablePrice = CommodityCalUtils.calFavourablePrice(isVip
                        ? vipCommissionPrice : commissionPrice,
                couponPrice);

        // 多返的佣金,即vip佣金减去普通佣金
        Double extraCommissionPrice = CommodityCalUtils.calExtraCommissionPrice(commissionPrice, vipCommissionPrice);


        return CommodityPrice.builder()
                // 普通佣金
                .commissionPrice(commissionPrice)
                // 商品最终购买价格
                .commodityPrice(commodityPrice)
                // 多返的佣金
                .extraCommissionPrice(extraCommissionPrice)
                // 购买省价格
                .favourablePrice(favourablePrice)
                // 分享赚的钱
                .shareCommissionPrice(shareCommissionPrice)
                // vip佣金
                .vipCommissionPrice(vipCommissionPrice)
                .build();
    }

    /**
     * 新人专享商品处理
     * 如果是新人专享商品且是新人，则价格特殊处理
     * @param detailInfo 商品信息
     * @param newbieRightFlag 是否新人专享商品
     * @param newbieRightUser 是否新人
     * @return 新人专享商品价格是否校验通过
     */
    private boolean queryNewbie(CommodityDetailDTO detailInfo, boolean newbieRightFlag, boolean newbieRightUser) {

        // 如果是新人商品 且是新人，则价格特殊处理
        if (newbieRightFlag && newbieRightUser) {
            if (commodityPriceInfoService.isRuleNewbiePrice(detailInfo.getDiscountPrice())) {
                // 将商品佣金自动变为券后价
                detailInfo.setCommissionPrice(detailInfo.getDiscountPrice());
                detailInfo.setVipCommissionPrice(detailInfo.getDiscountPrice());
                // 购买价格为0
                detailInfo.setCommodityPrice(0D);

            } else {

                //商品下架且剔除该商品
                commodityPoolService.setNewbieInvalidStatus(detailInfo.getGoodsId());
                return false;
            }
        }

        return true;
    }

    /**
     * vip 0元购商品处理
     * 如果是vip 0元购专享商品，且查询vip0元购信息同时还是vip，在价格符合规则的情况下，进行价格特殊处理
     * @param detailInfo 商品信息
     * @param queryVipZeroInfo 是否查询vip0元商品信息
     * @param vipUser 是否vip
     * @param isPublicRequest 是否是公开请求 如果是公开请求，则可能没有用户信息，则就不判断是否vip了
     * @return vip 0元购商品是否校验通过
     */
    private boolean queryVipZero(CommodityDetailDTO detailInfo, boolean queryVipZeroInfo, boolean vipUser,
                                 boolean isPublicRequest) {
        // 如果不是vip商品
        if (!commodityPriceInfoService.isVipZeroCommodity(detailInfo.getGoodsId())) {
            return true;
        }

        // 是vip 商品

        // 需要查询vio0元购信息，且是vip用户 or 公开请求
        if (queryVipZeroInfo && (isPublicRequest || vipUser)) {
            // 校验佣金是否超过vip0元购金额限制
            if (commodityPriceInfoService.isRuleVipZeroPrice(detailInfo.getPlatformCommissionPrice(),
                    detailInfo.getDiscountPrice())) {
                // 将商品佣金自动变为券后价
                detailInfo.setCommissionPrice(detailInfo.getDiscountPrice());
                detailInfo.setVipCommissionPrice(detailInfo.getDiscountPrice());
                // 购买价格为0
                detailInfo.setCommodityPrice(0D);
                // vip0元购专享商品
                detailInfo.setVipZero(Boolean.TRUE);
                return true;
            } else {
                // 商品下架且剔除该商品
                log.info("vip 0元购商品: {} 金额校验失败，从商品池中移除", detailInfo.getGoodsId());
                commodityPoolService.setPoolCommodityInvalidStatus(CommodityPoolId.VIP_ZERO, detailInfo.getGoodsId());
            }
        }

        // 否则就不通过
        return false;
    }

    /**
     * 设置商品的各种价格信息
     * @param param param
     * @param detailInfo 商品信息
     * @param platformCommissionPrice 平台佣金值
     * @return 是否成功设置
     */
    private CommodityDetailResultCode setCommodityPriceInfo(CommodityDetailQueryParam param, CommodityDetailDTO detailInfo,
                                                        Double platformCommissionPrice) {

        // 获取佣金比例 并设置购买可得的佣金
        PlatformCommissionRateInfo commissionInfo = commodityPriceInfoService.getPlatformCommissionRateInfo(param.getGoodsId());

        // 佣金信息获取失败
        if (Objects.isNull(commissionInfo)) {
            return CommodityDetailResultCode.PRICE_ERROR;
        }

        // vip服务获取是否vip
        if (Objects.isNull(param.getVipUser())) {
            param.setVipUser(vipFacadeService.isVip(param.getUserId()));
        }

        // 佣金值 （三方平台给的佣金值）
        if (Objects.isNull(platformCommissionPrice)) {
            platformCommissionPrice = 0D;
        }

        // 获取商品价格信息
        CommodityPrice commodityPriceInfo = getCommodityPrice(commissionInfo, platformCommissionPrice,
                detailInfo.getReservePrice(), detailInfo.getCouponPrice(), param.getVipUser());

        // 普通佣金值
        detailInfo.setNormalCommissionPrice(commodityPriceInfo.getCommissionPrice());
        // 最终的vip佣金值
        detailInfo.setCommissionPrice(param.getVipUser()
                ? commodityPriceInfo.getVipCommissionPrice() : commodityPriceInfo.getCommissionPrice());
        // 是否特殊返佣
        detailInfo.setSpecialCommissionFlag(commissionInfo.isSpecialCommissionFlag());
        // vip佣金值
        detailInfo.setVipCommissionPrice(commodityPriceInfo.getVipCommissionPrice());
        // 购买省
        detailInfo.setFavourablePrice(commodityPriceInfo.getFavourablePrice());
        // 多返的佣金
        detailInfo.setExtraCommissionPrice(commodityPriceInfo.getExtraCommissionPrice());
        // 分享赚的钱 默认就是非vip的商品普通佣金金额
        detailInfo.setShareMoney(commodityPriceInfo.getShareCommissionPrice());
        // 商品最终购买价格
        detailInfo.setCommodityPrice(commodityPriceInfo.getCommodityPrice());

        // 新人专享商品的特殊处理
        if (param.isQueryNewbie()) {
            // 设置是否新人
            if (Objects.isNull(param.getNewbieRightUser())) {
                param.setNewbieRightUser(commodityPriceInfoService.isNewbieRightForUser(param.getUserId()));
            }

            // 是否新人商品
            boolean newbieRightCommodity = commodityPriceInfoService.isNewbieRightCommodity(detailInfo.getGoodsId());
            // 如果是新人商品且是新人 则展示新人专享标签
            detailInfo.setNewbieRight(newbieRightCommodity && param.getNewbieRightUser());

            // 新人专享商品相关处理
            if (!queryNewbie(detailInfo, newbieRightCommodity, param.getNewbieRightUser())) {
                return CommodityDetailResultCode.NOVICE_PRICE_NOT_VALIDATE;
            }
        } else {
            detailInfo.setNewbieRight(false);
        }

        // 校验vip 0元购信息
        if (!queryVipZero(detailInfo, Objects.equals(param.getDetailType(), CommodityDetailType.VIP_ZERO),
                param.getVipUser(), param.isPublicRequest())) {
            return CommodityDetailResultCode.VIP_ZERO_CHECK_FAILED;
        }

        return CommodityDetailResultCode.SUCCESS;
    }

    /**
     * 根据查询参数，获取商品信息
     * 如果获取失败或者校验失败会返回错误码
     * 根据错误码针对不同的业务进行处理
     * @param param 查询参数
     * @return 商品信息 以及错误码
     */
    @Override
    public CommodityDetailResult getCommodityDetailWithResultCode(CommodityDetailQueryParam param) {

        // TODO 可以将下面的处理流程封装起来

        if (log.isDebugEnabled()) {
            log.debug("查询商品信息: query param: {}", param);
        }

        // 黑名单校验
        if (blackCommodityService.isBlackCommodity(param.getGoodsId())) {
            return buildResultCode(CommodityDetailResultCode.IN_BLACK);
        }

        // 获取商品基本信息
        CommodityDetailResult result = getCommodityBaseDetail(param);

        // 如果获取失败 则返回
        if (!Objects.equals(result.getResultCode(), CommodityDetailResultCode.SUCCESS)
                || Objects.isNull(result.getDetail())) {
            return result;
        }

        // 商品基本信息
        CommodityDetailDTO detailInfo = result.getDetail();

        // 设置商品的各种价格信息
        CommodityDetailResultCode resultCode = setCommodityPriceInfo(param, detailInfo,
                detailInfo.getPlatformCommissionPrice());

        // 如果获取失败 则返回
        if (!Objects.equals(resultCode, CommodityDetailResultCode.SUCCESS)) {
            return buildResultCode(resultCode, detailInfo);
        }

        // 非H5需要转换协议
        if (!param.isH5Flag()) {
            detailInfo.setCouponUrl(commodityCouponInfoService.assembleCouponUrlByType(detailInfo.getCouponUrl()));
        }

        // 获取卖家推荐默认头像
        if (param.isQueryDetail()) {
            detailInfo.setHeadImg(getDefaultHeadImg());
        }

        // 浏览奖励金币
        if (param.isQueryBrowerAward()) {
            CommodityGoldInfoDTO commodityGoldInfoDTO = commodityGoldService.getGoldInfoByCommodityId(param.getGoodsId(),
                    param.getUserId());
            if (null != commodityGoldInfoDTO) {
                detailInfo.setGoldStatus(commodityGoldInfoDTO.getDrawFlag());
                detailInfo.setGoldNum(commodityGoldInfoDTO.getBrowseGold());
            }
        }

        // 是否收藏的信息
        if (param.isQueryCollect()) {
            Integer collectStatus = commodityCollectService.getCollectStatus(param.getGoodsId(), param.getUserId());
            detailInfo.setCollectStatus(Objects.isNull(collectStatus) ? 0 : collectStatus);
        }

        // 是否提醒的信息
        if (param.isQueryRemind()) {
            detailInfo.setRemindStatus(secKillCommodityService.getRemindCommodityStatus(param.getGoodsId(), param.getUserId()));
        }

        // 设置标签信息
        if (param.isSetLable()) {
            if (null != detailInfo.getNewbieRight() && detailInfo.getNewbieRight()) {
                detailInfo.setLabelUrl(new ArrayList<>());
            } else {
                detailInfo.setLabelUrl(commodityLabelService.labelTypeFromRedis(detailInfo.getGoodsId()));
            }
        } else {
            detailInfo.setLabelUrl(Lists.newArrayList());
        }

        // 添加商品浏览记录
        if (param.isSaveHistory()) {
            historyBrowsingService.insertFootprint(param.getUserId(), param.getGoodsId());
        }

        // 校验金额信息
        if (param.isValidateCommissionPriceInfo()) {
            if (!validateCommissionAndCommodityPrice(detailInfo.getCommissionPrice(), detailInfo.getCommodityPrice())) {
                return buildResultCode(CommodityDetailResultCode.PRICE_NOTE_VALIDATE, detailInfo);
            }
        }

        // 校验优惠券金额信息
        if (param.isValidateCouponPriceInfo()) {
            if (!validateCoupon(detailInfo.getCouponPrice())) {
                return buildResultCode(CommodityDetailResultCode.COUPON_PRICE_NOTE_VALIDATE, detailInfo);
            }
        }

        // 查询成功
        return CommodityDetailResult.builder()
                .detail(detailInfo)
                .resultCode(CommodityDetailResultCode.SUCCESS)
                .build();
    }

    /**
     * 构建错误信息返回
     * 并带有商品基本信息
     * @param resultCode 错误信息
     * @return 错误信息对象
     */
    private CommodityDetailResult buildResultCode(CommodityDetailResultCode resultCode) {
        return CommodityDetailResult.builder()
                .resultCode(resultCode)
                .build();
    }

    /**
     * 构建错误信息返回
     * 并带有商品基本信息
     * @param resultCode 错误信息
     * @return 错误信息对象
     */
    private CommodityDetailResult buildResultCode(CommodityDetailResultCode resultCode,
                                                  CommodityDetailDTO detail) {
        if (Objects.nonNull(detail)
                && !Objects.equals(resultCode, CommodityDetailResultCode.SUCCESS)) {
            detail.setValidStatus(0);
        }

        return CommodityDetailResult.builder()
                .resultCode(resultCode)
                .detail(detail)
                .build();
    }

    @Override
    public CommodityListItemResult getCommodityListItemBaseInfo(CommodityDetailQueryParam queryParam) {
        // 查询商品信息
        CommodityDetailResult result = getCommodityDetailWithResultCode(queryParam);

        // 成功直接返回
        if (Objects.equals(result.getResultCode(), CommodityDetailResultCode.SUCCESS)
                && Objects.nonNull(result.getDetail())) {

            return CommodityListItemResult.builder()
                    .listItem(covert(result.getDetail()))
                    .resultCode(CommodityDetailResultCode.SUCCESS)
                    .build();
        }

        if (!queryParam.isDoNotPrintWarnLog()) {
            log.warn("获取商品详情错误: {}, 查询参数: {}", result.getResultCode().getDesc(), JSON.toJSONString(queryParam));
        }

        // 不成功 但是有数据
        if (queryParam.isIfNotValidThenReturnBaseInfo() && Objects.nonNull(result.getDetail())) {
            return CommodityListItemResult.builder()
                    .listItem(covert(result.getDetail()))
                    .resultCode(result.getResultCode())
                    .build();
        }

        // 带错误码的返回
        return CommodityListItemResult.builder()
                .resultCode(result.getResultCode())
                .build();
    }

    /**
     * 转换对象
     * @param detail detail
     * @return 新对象
     */
    private BaseCommodityListItem covert(CommodityDetailDTO detail) {
        // 转换对象封装数据
        BaseCommodityListItem commodity = new BaseCommodityListItem();

        // 商品id
        commodity.setGoodsId(detail.getGoodsId());
        // 商品标题
        commodity.setCommodityTitle(detail.getCommodityTitle());
        // 列表图片
        commodity.setImgUrl(detail.getMainPic());
        // 商品最终购买价格
        commodity.setCommodityPrice(detail.getCommodityPrice());
        // 店铺类型
        commodity.setShopType(detail.getShopType());
        // 店铺名称
        commodity.setShopName(detail.getShopName());
        // 红包金额
        commodity.setCouponPrice(detail.getCouponPrice());
        // 最终的佣金值 可能是普通佣金，也可能是vip佣金 也可能是0元购的佣金等
        commodity.setCommissionPrice(detail.getCommissionPrice());
        // 新人专享标签
        commodity.setNewbieRight(detail.getNewbieRight());
        // 月销量
        commodity.setMonthSales(Objects.isNull(detail.getSalesNum()) ? 0L : detail.getSalesNum().longValue());
        // 原价
        commodity.setReservePrice(detail.getReservePrice());
        // vip 佣金 如果商品是0元购，则当前佣金值就为0元购的佣金金额
        commodity.setVipCommissionPrice(detail.getVipCommissionPrice());
        // 券后价
        commodity.setDiscountPrice(detail.getDiscountPrice());

        // true:是特殊返佣, 不显示vip返佣金额  false:普通返佣
        commodity.setSpecialCommissionFlag(detail.isSpecialCommissionFlag());

        // 来源
        commodity.setSource(detail.getSource());
        return commodity;
    }

    /**
     * 设置htts
     * @param picUrl url
     * @return 正确url
     */
    private String assembleHttps(String picUrl) {
        if (StringUtils.isBlank(picUrl)) {
            return null;
        }

        StringBuilder stringBuilder = new StringBuilder("https:");
        return stringBuilder.append(picUrl).toString();
    }

    /**
     * 校验价格信息 原价为空 佣金是否为0
     * @param commissionPrice 佣金金额
     * @param commodityPrice 最终购买价格
     * @return 校验成功否
     */
    private boolean validateCommissionAndCommodityPrice(Double commissionPrice, Double commodityPrice) {

        if (Objects.isNull(commissionPrice)
                || Objects.isNull(commodityPrice)) {
            return false;
        }

        // 佣金是否为0
        return !Objects.equals(commissionPrice,  (double) 0);
    }

    /**
     * 校验价格信息 优惠券为0，原价为空
     * @param couponPrice reservePrice
     * @return 校验成功否
     */
    private boolean validateCoupon( Double couponPrice) {

        if (Objects.isNull(couponPrice)) {
            return false;
        }
        // 优惠券是否为0
        return !Objects.equals(couponPrice, (double) 0);
    }

    @Override
    public Boolean getRewardByBrowse(BrowseCommodityParam browseCommodityParam) {
        return commodityGoldService.getRewardByBrowse(browseCommodityParam);
    }

    @Override
    public Boolean getRewardByBrowseForH5(H5BrowseCommodityParam browseCommodityParam) {
        //由前端根据微信uid获取到平台的用户id
        Long browseUserId = browseCommodityParam.getBrowseUserId();
        Long shareUserId = browseCommodityParam.getShareUserId();
        Long goodsId = browseCommodityParam.getGoodsId();

        if (null == browseUserId) {
            log.info("非平台老用户,不给予增加金币");
            return true;
        }

        if (!commodityGoldService.isAvailableGold(goodsId, browseUserId)) {
            log.info("该用户今日不可领浏览金币,用户id:{}", browseUserId);
            return true;
        }

        //以下为未浏览过该商品的情况
        browseFlowService.addBrowseShareCommodityRecord(browseUserId, shareUserId, goodsId);

        //设置浏览记录到redis
        saveBrowseCommodityRecordToRedis(browseUserId.toString(), goodsId.toString());

        //浏览者获取金币奖励
        taskFacadeService.complete(browseUserId, ActivityFacadeEnum.FINISH_VIEW);

        //浏览者和分享者不是同一个人,才能获取分享奖励
        if (!browseUserId.equals(shareUserId)) {
            //分享者获取金币奖励-也扣除分享者的可浏览金币次数
            KeyGenerator limitKey = CommodityRedisKeyConstants.COMMODITY_BROWSE_GOLD_LIMIT.copy().appendKey(shareUserId.toString()).appendKey(DateUtils.getCurrentDate());
            ;
            redisStringAdapter.increment(limitKey, 172800L);

            return taskFacadeService.complete(browseCommodityParam.getShareUserId(), ActivityFacadeEnum.SHARE_VIEW);
        }

        return true;
    }



    @Override
    public List<CommodityDetailInfoDTO> getCommodityDetailPic(Long commodityId) {
        return commodityInfoSourceProxyService.getCommodityDetailInfo(commodityId);
    }

    @Override
    public String createTaoWord(String url, String imgUrl, String commodityTitle) {
        return tbCommodityService.createTaoWord(url, imgUrl, commodityTitle);
    }

    @Override
    public H5CommodityDetailDTO getH5CommodityDetailDTO(Long commodityId, Long shareUserId, String relationId) {
        CommodityDetailParam param = new CommodityDetailParam();
        param.setGoodsId(commodityId);
        param.setUserId(shareUserId);
        param.setH5Flag(true);

        CommodityDetailDTO commodityDetailDTO = getCommodityDetail(param);
        if (Objects.isNull(commodityDetailDTO)) {
            log.warn("站外获取商品详情为空, 商品id为:{}", commodityId);
            return null;
        }

        if (StringUtils.isBlank(commodityDetailDTO.getCouponUrl())) {
            log.warn("站外获取商品详情的优惠券为空, commodityId: {}", commodityId);
            return null;
        }

        H5CommodityDetailDTO resultDTO = new H5CommodityDetailDTO();
        BeanUtils.copyProperties(commodityDetailDTO, resultDTO);

        //若是黑名单的商品，则直接返回状态为0的结果集
        if (0 == resultDTO.getValidStatus()) {
            return resultDTO;
        }
        //淘口令生成所需的图片
        String imgUrl = CollectionUtils.isEmpty(commodityDetailDTO.getImgUrlList()) ? null : commodityDetailDTO.getImgUrlList().get(0);
        String taoWord = createTaoWord(assembleCouponUrl(commodityDetailDTO.getCouponUrl(), relationId),
                imgUrl, commodityDetailDTO.getCommodityTitle());

        resultDTO.setTaoWord(taoWord);

        return resultDTO;
    }

    @Override
    public H5CommodityDetailDTO getH5CommodityDetailDTOForAdvert(Long commodityId, String bxmId) {
        CommodityDetailDTO commodityDetailDTO = getCommodityDetailForAdvert(commodityId);
        if (null == commodityDetailDTO) {
            log.error("获取商品详情为空,商品id为:{}", commodityId);
            return null;
        }
        H5CommodityDetailDTO resultDTO = new H5CommodityDetailDTO();
        BeanUtils.copyProperties(commodityDetailDTO, resultDTO);

        String url = setAdvertPid(commodityDetailDTO, bxmId);

        //淘口令生成所需的图片
        String imgUrl = CollectionUtils.isEmpty(commodityDetailDTO.getImgUrlList()) ? null : commodityDetailDTO.getImgUrlList().get(0);
        String taoWord = createTaoWord(url == null ? commodityDetailDTO.getCouponUrl() : url,
                imgUrl, commodityDetailDTO.getCommodityTitle());

        resultDTO.setTaoWord(taoWord);

        return resultDTO;
    }


    /**
     * 设置广告部投放的专属pid
     *
     * @param commodityDetailDTO
     * @Param bxmId
     */
    private String setAdvertPid(CommodityDetailDTO commodityDetailDTO, String bxmId) {
        AdvertPlaceDTO advertPlaceDTO = getBxmAdvertPlaceInfo(bxmId);
        if (Objects.isNull(advertPlaceDTO)) {
            HdkCommissionDataDTO hdkCommissionDataDTO = hdkCommodityService.getCommissionAndCoupon
                    (commodityDetailDTO.getGoodsId(), "mm_456760114_597650089_109744800352");
            if (!Objects.isNull(hdkCommissionDataDTO)) {
                return hdkCommissionDataDTO.getCoupon_click_url();
            } else {
                return null;
            }
        }
        KeyGenerator advertPidKey = CommodityRedisKeyConstants.COMMODITY_ADVERT_PID_MAP.copy();
        String pid = redisHashMapAdapter.get(advertPidKey, advertPlaceDTO.getPlaceId(), String.class);
        if (StringUtils.isEmpty(pid)) {
            log.error("未配置变现猫广告位或配置有误，请根据appKey-business:pid配置。appKey:{}, business:{}",
                    advertPlaceDTO.getAppKey(), advertPlaceDTO.getBusiness());
            HdkCommissionDataDTO hdkCommissionDataDTO = hdkCommodityService.getCommissionAndCoupon
                    (commodityDetailDTO.getGoodsId(), "mm_456760114_597650089_109744800352");
            if (!Objects.isNull(hdkCommissionDataDTO)) {
                return hdkCommissionDataDTO.getCoupon_click_url();
            } else {
                return null;
            }
        }
        //将pid对应的bxmId（内部包含广告位信息）缓存起来，方便广告部统计广告位效果
        KeyGenerator bxmIdKey = CommodityRedisKeyConstants.COMMODITY_ADVERT_PID_BXMID_MAP.copy();
        redisHashMapAdapter.put(bxmIdKey, pid, bxmId);
        try {
            HdkCommissionDataDTO hdkCommissionDataDTO = hdkCommodityService.getCommissionAndCoupon
                    (commodityDetailDTO.getGoodsId(), pid);
            if (!Objects.isNull(hdkCommissionDataDTO)) {
                return hdkCommissionDataDTO.getCoupon_click_url();
            }
        } catch (Exception e) {
            log.error("重置优惠卷链接失败，couponUrl:{}", commodityDetailDTO.getCouponUrl());
            e.printStackTrace();
        }
        return null;
    }

    /**
     * 从变现猫获取appKey
     *
     * @param bxmId 变现猫id
     * @return 变现猫返回数据都封装类
     */
    private AdvertPlaceDTO getBxmAdvertPlaceInfo(String bxmId) {
        String url = advertTicketProperties.getParseBxmIdUrl();
        Map<String, Object> params = new HashMap<>();
        params.put("bxm_id", bxmId);
        AdvertPlaceDTO advertPlaceDTO = new AdvertPlaceDTO();
        try {
            String result = OkHttpUtils.get(url, params);
            JSONObject resultJson = JSONObject.parseObject(result);
            if (Objects.isNull(resultJson.getBoolean("success")) || !resultJson.getBoolean("success")) {
                log.error("解析bxmId失败，url:{}，bxmId:{}, 返回参数:{}", url, bxmId, result);
            }
            advertPlaceDTO = JSON.toJavaObject(resultJson.getJSONObject("data"), AdvertPlaceDTO.class);
        } catch (IOException e) {
            log.error("解析bxmId失败,url:{}，bxmId:{}", url, bxmId);
            e.printStackTrace();
        }
        return advertPlaceDTO;
    }

    @Override
    public void checkAndsendOrderToBxmAdvert(Long orderId, String pid) {
        KeyGenerator pidKey = CommodityRedisKeyConstants.COMMODITY_ADVERT_PID_BXMID_MAP.copy();
        String bxmId = redisHashMapAdapter.get(pidKey, pid, String.class);

        KeyGenerator sendOrderKey = CommodityRedisKeyConstants.COMMODITY_SEND_ADVERT.copy().appendKey(String.valueOf
                (orderId));
        if (!StringUtils.isEmpty(bxmId) && !redisStringAdapter.hasKey(sendOrderKey)) {
            String url = advertTicketProperties.getSendOrderUrl();
            Map<String, Object> params = new HashMap<>();
            params.put("bxmId", bxmId);
            params.put("type", 5);
            try {
                String result = OkHttpUtils.get(url, params);
                JSONObject resultJson = JSONObject.parseObject(result);
                if (Objects.isNull(resultJson.getBoolean("successed")) || !resultJson.getBoolean("successed")) {
                    log.error("推送给广告部到效果监控订单失败：订单id:{}, 返回参数:{}", orderId, result);
                    return;
                }
                redisStringAdapter.set(sendOrderKey, "true", 3600L * 24 * 30);
            } catch (IOException e) {
                log.error("推送给广告部到效果监控订单失败：订单id:{}", orderId);
                e.printStackTrace();
            }
        }
    }

    /**
     * 获取基本信息
     *
     * @return 基本信息
     */
    private CommodityDetailDTO convert(CommodityInfoDTO commodityInfoDTO) {


        return CommodityDetailDTO.builder()
                // 商品价格-到手价,商品原价-优惠券价格-佣金
                .commodityPrice(commodityInfoDTO.getCommodityPrice())
                // 商品描述得分
                .commodityScore(commodityInfoDTO.getCommodityScore())
                // 商品标题
                .commodityTitle(commodityInfoDTO.getCommodityTitle())
                // 优惠券价格
                .couponPrice(commodityInfoDTO.getCouponPrice())
                // 优惠券链接
                .couponUrl(commodityInfoDTO.getCouponUrl())
                // 券后价,商品原价-优惠券价格
                .discountPrice(commodityInfoDTO.getDiscountPrice())
                // 商品id
                .goodsId(commodityInfoDTO.getGoodsId())
                // 商品图片list
                .imgUrlList(commodityInfoDTO.getImgUrlList())
                // 物流得分
                .logisticsScore(commodityInfoDTO.getLogisticsScore())
                // 商品主图
                .mainPic(commodityInfoDTO.getMainPic())
                // 推荐内容
                .recommendContent(commodityInfoDTO.getRecommendContent())
                // 商品原价
                .reservePrice(commodityInfoDTO.getReservePrice())
                // 销量
                .salesNum(commodityInfoDTO.getSalesNum())
                // 卖家服务得分
                .sellerScore(commodityInfoDTO.getSellerScore())
                // 店铺标志
                .shopIcon(commodityInfoDTO.getShopIcon())
                // 店铺名称
                .shopName(commodityInfoDTO.getShopName())
                // 店铺类型，1-天猫，0-淘宝
                .shopType(Objects.isNull(commodityInfoDTO.getShopType()) ? 0 : commodityInfoDTO.getShopType().intValue())
                // 店铺链接url
                .shopUrl(commodityInfoDTO.getShopUrl())
                // 图文详情
                .commodityDetailInfos(commodityInfoDTO.getDetailInfo())
                // 平台佣金
                .platformCommissionPrice(commodityInfoDTO.getPlatformCommissionPrice())
                .validStatus(1)
                // 数据来源
                .source(commodityInfoDTO.getSoruce())
                .build();
    }

    /**
     * 保存浏览信息
     * @param userId 用户id
     * @param goodsId 商品id
     */
    private void saveBrowseCommodityRecordToRedis(String userId, String goodsId) {

        //设置浏览记录到redis
        KeyGenerator browseCommodityKey = CommodityRedisKeyConstants.COMMODITY_BROWSE_RECORD.copy().appendKey(userId).appendKey(goodsId);

        redisStringAdapter.set(browseCommodityKey, "record", DateUtils.getCurSeconds());
    }

    /**
     * 组装优惠券url relationId
     *
     * @param url        优惠券url
     * @param relationId 渠道关系id
     * @return 拼接relationId的优惠券url
     */
    private String assembleCouponUrl(String url, String relationId) {
        if (StringUtils.isNotBlank(url)) {
            StringBuilder sb = new StringBuilder(url);

            return sb.append("&relationId=").append(relationId).toString();
        } else {
            log.warn("组装优惠券url失败, relationId: {}", relationId);
        }

        return null;
    }

    /**
     * 获取指定卖家默认头像-在配置中随机获取
     *
     * @return 默认头像url
     */
    private String getDefaultHeadImg() {
        int min = 0;
        int max = 12;
        int randomNum = new Random().nextInt(max - min) + min;

        List<DefaultHeadImgDTO> headImgList = JSON.parseArray(commodityInfoProperties.getHeadImgList(), DefaultHeadImgDTO.class);

        return headImgList.get(randomNum).getHeadImg();
    }

    /**
     * 获取广告部投放落地页商品详情
     *
     * @param commodityId 商品id
     * @return 商品详情
     */
    private CommodityDetailDTO getCommodityDetailForAdvert(Long commodityId) {
        CommodityDetailDTO detailInfo;

        CommodityInfoDTO commodityInfoDTO = commodityInfoSourceProxyService.getCommodityDetail(commodityId);

        if (Objects.isNull(commodityInfoDTO) || Objects.isNull(detailInfo = convert(commodityInfoDTO))) {
            log.error("获取商品信息失败,无此商品信息");
            return null;
        }

        detailInfo.setCouponUrl(commodityCouponInfoService.assembleCouponUrlByType(detailInfo.getCouponUrl()));

        //获取卖家推荐默认头像
        detailInfo.setHeadImg(getDefaultHeadImg());

        return detailInfo;
    }

}
