package com.bxm.fossicker.commodity.service.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.fossicker.commodity.common.constants.CommodityRedisKeyConstants;
import com.bxm.fossicker.commodity.common.dto.CommissionAndCoupon;
import com.bxm.fossicker.commodity.common.dto.CommodityCommissionAndCoupon;
import com.bxm.fossicker.commodity.common.dto.PlatformCommissionRateInfo;
import com.bxm.fossicker.commodity.common.utils.CommodityNumberUtils;
import com.bxm.fossicker.commodity.config.CommissionProperties;
import com.bxm.fossicker.commodity.config.CommodityInfoProperties;
import com.bxm.fossicker.commodity.domain.CommodityInfoMapper;
import com.bxm.fossicker.commodity.facade.CommodityCommissionFacadeService;
import com.bxm.fossicker.commodity.facade.constant.CommissionType;
import com.bxm.fossicker.commodity.facade.dto.CommissionEleFacadeDTO;
import com.bxm.fossicker.commodity.facade.dto.CommissionPlatformFacadeDTO;
import com.bxm.fossicker.commodity.facade.param.GetEleCommissionParam;
import com.bxm.fossicker.commodity.facade.param.GetPlatformCommissionParam;
import com.bxm.fossicker.commodity.model.constant.CommodityDetailType;
import com.bxm.fossicker.commodity.model.dto.CommodityDetailDTO;
import com.bxm.fossicker.commodity.model.dto.CommodityInfoDTO;
import com.bxm.fossicker.commodity.model.param.CommodityDetailQueryParam;
import com.bxm.fossicker.commodity.model.param.GetCommodityDetailParam;
import com.bxm.fossicker.commodity.model.vo.CustomRebateBean;
import com.bxm.fossicker.commodity.service.CommodityInfoService;
import com.bxm.fossicker.commodity.service.CommodityPriceInfoService;
import com.bxm.fossicker.commodity.service.commodity.info.CommodityInfoSourceProxyService;
import com.bxm.fossicker.commodity.service.strategy.PriceByThirdStrategyService;
import com.bxm.fossicker.constant.CommonCommodityRedisKey;
import com.bxm.fossicker.user.facade.UserInfoFacadeService;
import com.bxm.fossicker.user.facade.dto.SuperiorDto;
import com.bxm.fossicker.user.facade.vip.VipFacadeService;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Objects;

/**
 * 获取商品佣金及优惠券信息服务
 *
 * @author donghy
 */
@Service
@Slf4j
public class CommodityPriceInfoServiceImpl implements CommodityPriceInfoService, CommodityCommissionFacadeService {

    @Autowired
    @Qualifier("priceStrategyByTbService")
    private PriceByThirdStrategyService reservePriceByTbService;

    @Autowired
    @Qualifier("commissionStrategyByWyService")
    private PriceByThirdStrategyService commissionStrategyByWyService;

    @Autowired
    private RedisHashMapAdapter redisHashMapAdapter;

    @Resource
    private CommissionProperties commissionProperties;


    @Autowired
    private CommodityInfoProperties commodityInfoProperties;

    @Autowired
    private CommodityInfoSourceProxyService commodityInfoSourceProxyService;

    @Autowired
    private UserInfoFacadeService userInfoFacadeService;

    @Autowired
    private RedisSetAdapter redisSetAdapter;

    @Autowired
    private VipFacadeService vipFacadeService;

    @Autowired
    private CommodityInfoMapper commodityInfoMapper;

    @Autowired
    private CommodityInfoService commodityInfoService;

    @Override
    public CommodityCommissionAndCoupon getCommodityCommissionAndCoupon(Long goodsId, Long userId) {
        return getCommodityCommissionAndCoupon(goodsId, userId, false);
    }

    @Override
    public CommodityCommissionAndCoupon getCommodityCommissionAndCoupon(Long goodsId, Long userId, Boolean newbieNotShowFlag) {

        //第三方佣金服务工厂创建
        PriceByThirdStrategyService priceByThirdStrategyService = commissionServiceFactory();
        if (null == priceByThirdStrategyService) {
            log.error("创建第三方服务失败,请检查代码");
            return null;
        }
        // 获取三方佣金比例、优惠券信息
        CommissionAndCoupon commissionAndCoupon = priceByThirdStrategyService.getCommissionAndCoupon(goodsId);
        if (Objects.isNull(commissionAndCoupon)) {
            return null;
        }
        // 设置三方佣金比例、优惠券信息
        CommodityCommissionAndCoupon commissionInfoDTO = new CommodityCommissionAndCoupon();
        BeanUtils.copyProperties(commissionAndCoupon, commissionInfoDTO);

        // 获取自身平台比例佣金
        PlatformCommissionRateInfo platformCommissionRateInfo = getPlatformCommissionRateInfo(goodsId);
        if (Objects.nonNull(platformCommissionRateInfo)) {
            // 设置自身平台比例佣金
            BeanUtils.copyProperties(platformCommissionRateInfo, commissionInfoDTO);
        }

        return commissionInfoDTO;
    }

    @Override
    public CommissionAndCoupon getCommodityCommissionAndCoupon(Long goodsId) {

        //第三方佣金服务工厂创建
        PriceByThirdStrategyService priceByThirdStrategyService = commissionServiceFactory();
        if (null == priceByThirdStrategyService) {
            log.error("创建第三方服务失败,请检查代码");
            return null;
        }

        // 获取三方数据并返回
        return priceByThirdStrategyService.getCommissionAndCoupon(goodsId);
    }

    @Override
    public CommissionEleFacadeDTO getEleCommission(GetEleCommissionParam param) {
        CommissionEleFacadeDTO facadeDTO = new CommissionEleFacadeDTO();
        if (Objects.isNull(param) || Objects.isNull(param.getPayPrice())) {
            log.warn("计算饿了么订单返佣失败，请求参数: {}", JSON.toJSONString(param));
            return facadeDTO;
        }

        BigDecimal payPrice = param.getPayPrice();
        // vip 自购
        facadeDTO.setVipPurchasePrice(payPrice.multiply(commissionProperties.getEle()
                .getVipPurchaseRateForPayPrice()));
        // 自购
        facadeDTO.setPurchasePrice(payPrice.multiply(commissionProperties.getEle()
                .getCommonCommissionPurchaseRateForPayPrice()));
        // 上级
        facadeDTO.setParentPrice(payPrice.multiply(commissionProperties.getEle()
                .getCommonCommissionRateParentForPayPrice()));
        // 上上级
        facadeDTO.setGrandParentPrice(payPrice.multiply(commissionProperties.getEle()
                .getCommonCommissionRateGrandParentForPayPrice()));
        // 饿了么
        facadeDTO.setCommissionType(CommissionType.ELE);

        return facadeDTO;
    }

    @Override
    public CommissionPlatformFacadeDTO getPlatformCommission(GetPlatformCommissionParam param) {

        //根据会员id查询在自身平台的id以及上层关系。
        SuperiorDto superiorDtoBySpecialId = userInfoFacadeService.getSuperiorBySpecialId(param.getSpecialId());
        //根据relationId获取平台id以及上层关系
        SuperiorDto superiorDtoByRelationId = userInfoFacadeService.getSuperiorByRelationId(param.getRelationId());
        // 是否自购单
        boolean isPurchaseBySelf = isPurchaseBySelf(superiorDtoBySpecialId, superiorDtoByRelationId);
        boolean isVip = vipFacadeService.isVip(superiorDtoByRelationId.getUserId());
        BigDecimal unitPrice = param.getPayPrice().divide(BigDecimal.valueOf(Objects.isNull(param.getGoodsNum())
                        ? 1L : param.getGoodsNum()),
                2, RoundingMode.DOWN);

        log.info("获取返佣信息，isPurchaseBySelf: {} isVip: {}, 订单支付金额: {}, 商品数量: {}, 计算商品单价: {} ",
                isPurchaseBySelf, isVip, param.getPayPrice(), param.getGoodsNum(), unitPrice);

        CommissionPlatformFacadeDTO platformFacadeDTO;
        // 如果是自购单
        if (isPurchaseBySelf) {
            // 特殊返佣
            platformFacadeDTO = getCustomCommission(param.getGoodsId(), param.getCommissionPrice());

            if (Objects.nonNull(platformFacadeDTO)) {
                log.info("根据请求参数: {} 获取到特殊返佣信息: {}", JSON.toJSONString(param),
                        JSON.toJSONString(platformFacadeDTO));
                return platformFacadeDTO;
            }

            // 新人返佣
            platformFacadeDTO = getNoviceCommission(param.getGoodsId(), unitPrice,
                    superiorDtoByRelationId.getUserId());

            if (Objects.nonNull(platformFacadeDTO)) {
                log.info("根据请求参数: {} 获取到新人返佣信息: {}", JSON.toJSONString(param),
                        JSON.toJSONString(platformFacadeDTO));
                return platformFacadeDTO;
            }

            // vip 0元购返佣
            platformFacadeDTO = getVipZeroCommission(param.getGoodsId(), unitPrice, isVip, superiorDtoByRelationId.getUserId());

            if (Objects.nonNull(platformFacadeDTO)) {
                log.info("根据请求参数: {} 获取到vip 0元购返佣信息: {}", JSON.toJSONString(param),
                        JSON.toJSONString(platformFacadeDTO));
                return platformFacadeDTO;
            }
        }

        // VIP返佣比例
        String vipPurchaseCommission = vipFacadeService.vipPurchase();
        String purchaseCommission = commodityInfoProperties.getCommonCommissionPurchase();
        String parentCommission = commodityInfoProperties.getCommonCommissionParent();
        String grandparentCommission = commodityInfoProperties.getCommonCommissionGrandParent();

        // 配置项中取普通商品佣金
        platformFacadeDTO = CommissionPlatformFacadeDTO.builder()
                .goodsId(param.getGoodsId())
                .vipPurchasePrice(calCommonCommissionPrice(vipPurchaseCommission, param.getCommissionPrice()))
                .purchasePrice(calCommonCommissionPrice(purchaseCommission, param.getCommissionPrice()))
                .commissionType(CommissionType.DEFAULT)
                .build();


        // 1.6.0 之后，父级返佣是根据购买者获得的佣金来计算的
        if (isVip) {
            // vip 用户上级的返佣
            platformFacadeDTO.setParentPrice(calCommonCommissionPrice(parentCommission,
                    platformFacadeDTO.getVipPurchasePrice()));
            // vip 用户上上级的返佣
            platformFacadeDTO.setGrandParentPrice(calCommonCommissionPrice(grandparentCommission,
                    platformFacadeDTO.getVipPurchasePrice()));
        } else {
            // 非vip用户上级的返佣
            platformFacadeDTO.setParentPrice(calCommonCommissionPrice(parentCommission,
                    platformFacadeDTO.getPurchasePrice()));
            // 非vip用户上上级的返佣
            platformFacadeDTO.setGrandParentPrice(calCommonCommissionPrice(grandparentCommission,
                    platformFacadeDTO.getPurchasePrice()));
        }

        log.info("根据请求参数: {} 获取到普通返佣信息: {}", JSON.toJSONString(param),
                JSON.toJSONString(platformFacadeDTO));

        return platformFacadeDTO;
    }

    @Override
    public CommissionPlatformFacadeDTO getPlatformCommission(Long goodsId, BigDecimal commissionPrice, String specialId,
                                                             String relationId, BigDecimal payPrice, Long goodsNum) {

        return getPlatformCommission(GetPlatformCommissionParam.builder()
                .goodsId(goodsId)
                .relationId(relationId)
                .specialId(specialId)
                .commissionPrice(commissionPrice)
                .payPrice(payPrice)
                .goodsNum(goodsNum)
                .build());
    }

    /**
     * 获取特殊佣金 如果符合条件的话
     *
     * @param goodsId
     * @param commissionPrice
     * @return 特殊佣金 or null
     */
    private CommissionPlatformFacadeDTO getCustomCommission(Long goodsId, BigDecimal commissionPrice) {
        // 先查是否是特殊商品佣金
        KeyGenerator platformCommissionKey = CommodityRedisKeyConstants.COMMODITY_PLATFORM_COMMISSION;
        CustomRebateBean customRebateBean =
                redisHashMapAdapter.get(platformCommissionKey, goodsId.toString(), CustomRebateBean.class);


        if (Objects.nonNull(customRebateBean)) {
            CommissionPlatformFacadeDTO commissionPlatformDTO = CommissionPlatformFacadeDTO.builder()
                    .goodsId(goodsId)
                    .build();
            commissionPlatformDTO.setPurchasePrice(calCommissionPrice(customRebateBean.getPurchasePrice(),
                    customRebateBean.getPurchaseRate(), commissionPrice));

            // 父级返佣是购买者得的佣金的比例
            commissionPlatformDTO.setParentPrice(calCommissionPrice(customRebateBean.getParentPrice(),
                    customRebateBean.getParentRate(), commissionPlatformDTO.getPurchasePrice()));

            commissionPlatformDTO.setGrandParentPrice(calCommissionPrice(customRebateBean.getGrandParentPrice(),
                    customRebateBean.getGrandParentRate(), commissionPrice));

            // 特殊返佣
            commissionPlatformDTO.setCommissionType(CommissionType.CUSTOM);
            return commissionPlatformDTO;
        }

        return null;
    }

    /**
     * 获取新人佣金 如果符合条件的话
     *
     * @param goodsId        商品id
     * @param unitPrice      单价
     * @param purchaseUserId 自购用户id
     * @return 新人佣金 or null
     */
    private CommissionPlatformFacadeDTO getNoviceCommission(Long goodsId, BigDecimal unitPrice, Long purchaseUserId) {

        if (isNewbieRight(purchaseUserId, goodsId)) {
            CommissionPlatformFacadeDTO platformFacadeDTO = new CommissionPlatformFacadeDTO();

            platformFacadeDTO.setGoodsId(goodsId);
            // vip购买反佣金 - 商品的购买价格
            platformFacadeDTO.setVipPurchasePrice(unitPrice);
            // 普通用户购买反佣金 - 商品的购买价格
            platformFacadeDTO.setPurchasePrice(unitPrice);
            // 父级 不返佣
            platformFacadeDTO.setParentPrice(BigDecimal.ZERO);
            // 顶级 不返佣
            platformFacadeDTO.setGrandParentPrice(BigDecimal.ZERO);
            // 新人返佣
            platformFacadeDTO.setCommissionType(CommissionType.NOVICE);
            return platformFacadeDTO;
        }

        return null;
    }

    /**
     * 获取vip 0元购佣金 如果符合条件的话
     *
     * @param goodsId   商品id
     * @param unitPrice 商品最终购买单价
     * @param isVip     是否vip
     * @return vip 0元购佣金
     */
    private CommissionPlatformFacadeDTO getVipZeroCommission(Long goodsId, BigDecimal unitPrice, boolean isVip,
                                                             Long userId) {

        // 是否vip0元购返佣商品且是vip
        if (isVip && isVipZeroCommodity(goodsId)) {

            CommodityDetailDTO commodityDetailDTO = commodityInfoService.getCommodityDetail(CommodityDetailQueryParam.builder()
                    .goodsId(goodsId)
                    .vipUser(true)
                    .userId(userId)
                    // 查询vip 0元购金额
                    .detailType(CommodityDetailType.VIP_ZERO)
                    .build());


            // 如果查询不到商品 则使用默认的返佣信息
            if (Objects.isNull(commodityDetailDTO)) {
                log.warn("查询vip 0元购商品: {} 失败，无法计算返佣信息", goodsId);
                return null;
            }

            // 最大的返佣金额为商品的到手价
            Double maxCommodityPrice = commodityDetailDTO.getDiscountPrice();
            if (Objects.isNull(maxCommodityPrice)) {
                log.warn("查询vip 0元购商品: {} 购买价格失败，无法计算返佣信息，查询到的商品信息: {}", goodsId,
                        JSON.toJSONString(commodityDetailDTO));
                return null;
            }

            BigDecimal maxCommodityPriceD = BigDecimal.valueOf(maxCommodityPrice);
            BigDecimal rebate = maxCommodityPriceD.compareTo(unitPrice) < 0 ? maxCommodityPriceD : unitPrice;

            log.info("计算vip 0元购商品最终返佣金额: {} 订单的购买单价: {}, 商品的券后价: {} ", rebate,
                    unitPrice, maxCommodityPriceD);

            CommissionPlatformFacadeDTO platformFacadeDTO = new CommissionPlatformFacadeDTO();

            platformFacadeDTO.setGoodsId(goodsId);
            // vip购买反佣金 - 商品的购买价格
            platformFacadeDTO.setVipPurchasePrice(rebate);
            // 普通用户购买反佣金 - 商品的购买价格
            platformFacadeDTO.setPurchasePrice(rebate);

            // 父级返佣 是购买者的获得金额的百分比
            platformFacadeDTO.setParentPrice(calCommonCommissionPrice(commodityInfoProperties.getCommonCommissionParent(),
                    platformFacadeDTO.getVipPurchasePrice()));

            // 上上级
            platformFacadeDTO.setGrandParentPrice(calCommonCommissionPrice(commodityInfoProperties.getCommonCommissionGrandParent(),
                    platformFacadeDTO.getVipPurchasePrice()));

            // vip 0元购返佣
            platformFacadeDTO.setCommissionType(CommissionType.VIP_ZERO);
            return platformFacadeDTO;
        }

        return null;
    }

    @Override
    public Double getReservePrice(Long goodsId) {
        //第三方佣金服务工厂创建
        PriceByThirdStrategyService priceByThirdStrategyService = reservePriceServiceFactory();
        if (null == priceByThirdStrategyService) {
            log.error("创建第三方服务失败,请检查代码");
            return null;
        }

        return priceByThirdStrategyService.getPriceInfo(goodsId);
    }

    @Override
    public Boolean isNewbieRight(Long userId, Long goodsId) {
        // 先判断是否新人专享商品
        if (!isNewbieRightCommodity(goodsId)) {
            return false;
        }

        //判断该用户是否领取过新人首单奖励
        return isNewbieRightForUser(userId);
    }

    @Override
    public CommissionAndCoupon getCommissionAndCouponByThird(Long goodsId) {

        PriceByThirdStrategyService priceByThirdStrategyService = commissionServiceFactory();
        if (null == priceByThirdStrategyService) {
            log.error("创建第三方服务失败,请检查代码");
            return null;
        }
        return priceByThirdStrategyService.getCommissionAndCoupon(goodsId);
    }

    @Override
    public boolean isNewbieRightForUser(Long userId) {
        if (Objects.isNull(userId)) {
            return false;
        }
        return !redisSetAdapter.exists(CommonCommodityRedisKey.COMMODITY_NEWBIE_USER_SET, userId);
    }

    @Override
    public boolean isNewbieRightCommodity(Long goodsId) {
        // 是否是新人专享商品，加入的时候是字符串，所以要转字符串进行比较
        KeyGenerator newbieCommoditySetKey = CommodityRedisKeyConstants.COMMODITY_NEWBIE_SET;
        return redisSetAdapter.exists(newbieCommoditySetKey, String.valueOf(goodsId));
    }

    @Override
    public boolean isVipZeroCommodity(Long goodsId) {
        // 是否是新人专享商品，加入的时候是字符串，所以要转字符串进行比较
        KeyGenerator newbieCommoditySetKey = CommodityRedisKeyConstants.COMMODITY_VIP_ZERO_SET;
        return redisSetAdapter.exists(newbieCommoditySetKey, String.valueOf(goodsId));
    }


    @Override
    public PlatformCommissionRateInfo getPlatformCommissionRateInfo(Long goodsId) {
        PlatformCommissionRateInfo platformCommissionRateInfo = new PlatformCommissionRateInfo();

        // vip服务提供获取vip专享返佣的接口
        platformCommissionRateInfo.setVipCommissionRate(Double.valueOf(vipFacadeService.vipPurchase()));
        platformCommissionRateInfo.setGoodsId(goodsId);

        //从redis中查询平台特殊商品佣金
        KeyGenerator platformCommissionKey = CommodityRedisKeyConstants.COMMODITY_PLATFORM_COMMISSION;
        CustomRebateBean customRebateBean =
                redisHashMapAdapter.get(platformCommissionKey, goodsId.toString(), CustomRebateBean.class);

        //若redis中有值-则取redis中的值,redis中有值则必定是特殊返佣配置
        if (Objects.nonNull(customRebateBean)) {
            // 设置特殊佣金标志位true
            platformCommissionRateInfo.setSpecialCommissionFlag(true);

            if (!CommodityNumberUtils.isNullOrZero(customRebateBean.getPurchasePrice())) {
                platformCommissionRateInfo.setCommissionRegularPrice(customRebateBean.getPurchasePrice().doubleValue());
            }

            // redis中取特殊商品佣金
            if (!CommodityNumberUtils.isNullOrZero(customRebateBean.getPurchaseRate())) {
                platformCommissionRateInfo.setCommissionRate(customRebateBean.getPurchaseRate()
                        .divide(new BigDecimal("100"), 2, BigDecimal.ROUND_DOWN)
                        .doubleValue());
            }

            return platformCommissionRateInfo;
        }

        // redis没值则从去默认的佣金比例--默认的佣金比例则特殊佣金的标志位为false
        platformCommissionRateInfo.setSpecialCommissionFlag(false);
        platformCommissionRateInfo.setCommissionRate(Double.valueOf(commodityInfoProperties.getCommonCommissionPurchase()));

        return platformCommissionRateInfo;
    }

    @Override
    public String getCommonPurchaseRate() {
        return commodityInfoProperties.getCommonCommissionPurchase();
    }

    @Override
    public boolean isRuleNewbiePrice(Double commissionPrice) {
        return commodityInfoProperties.getNewbieMaxPrice() >= commissionPrice;
    }

    @Override
    public boolean isRuleVipZeroPrice(Double platformCommissionPrice, Double discountPrice) {
        if (Objects.isNull(platformCommissionPrice) || Objects.isNull(discountPrice)) {
            log.warn("佣金: {} 券后价: {}参数错误，无法计算vip价格是否超过阈值", platformCommissionPrice, discountPrice);
            return false;
        }

        /**
         * 补贴金额阈值：
         *
         * 补贴金额=商品券后价*（1-返佣比例）+商品券后价*10%（其中，返佣比例为高佣转链返回的返佣比例）
         *
         * 补贴金额最高6元，大于6元则下架（其中，6元为可配置）
         */

        BigDecimal discountPriceD = BigDecimal.valueOf(discountPrice);
        BigDecimal platformCommissionPriceD = BigDecimal.valueOf(platformCommissionPrice);

        // 平台补贴给购买者的金额 = 券后价 - 商品的平台佣金
        BigDecimal subsidyForPurchase = discountPriceD.subtract(platformCommissionPriceD);
        // 补贴给购买者上级的金额 = 券后价（vip0元购的用户可得佣金金额） * 上级返佣比例
        BigDecimal subsidyForPurchaseParent = discountPriceD
                .multiply(new BigDecimal(commodityInfoProperties.getCommonCommissionParent()))
                .setScale(2, BigDecimal.ROUND_DOWN);
        // 总补贴
        BigDecimal totalSubsidy = subsidyForPurchaseParent.add(subsidyForPurchase);

        // 总补贴不能大于规定的vip最大金额
        return totalSubsidy.compareTo(BigDecimal.valueOf(commodityInfoProperties.getVipZeroMaxPrice())) <= 0;
    }

    /**
     * 工厂方法创建
     *
     * @return
     */
    private PriceByThirdStrategyService commissionServiceFactory() {
        return commissionStrategyByWyService;
    }

    /**
     * 工厂方法创建
     *
     * @return
     */
    private PriceByThirdStrategyService reservePriceServiceFactory() {
        return reservePriceByTbService;
    }

    /**
     * 若平台配置了佣金固定值,则直接取固定值,不然就用购买者的佣金乘以上级佣金比例
     *
     * @param platformCommissionPrice 平台侧佣金固定值
     * @param platformCommissionRate  平台侧佣金比例 百分数,需要除以100
     * @param commissionPrice         购买者得到的佣金
     * @return 特定的佣金值-（截取小数位后两位的值）
     */
    private BigDecimal calCommissionPrice(BigDecimal platformCommissionPrice, BigDecimal platformCommissionRate,
                                          BigDecimal commissionPrice) {
        if (null == commissionPrice) {
            log.warn("三方提供的佣金值为空");
            return new BigDecimal("0.00");
        }

        BigDecimal finalCommissionPrice;
        if (null != platformCommissionPrice && BigDecimal.ZERO.compareTo(platformCommissionPrice) != 0) {
            finalCommissionPrice = platformCommissionPrice.setScale(2, BigDecimal.ROUND_DOWN);
        } else {
            finalCommissionPrice = commissionPrice.multiply(platformCommissionRate.divide(new BigDecimal("100"),
                    2, BigDecimal.ROUND_DOWN));
        }

        return finalCommissionPrice;
    }

    /**
     * 根据三方计算通用商品佣金值
     *
     * @param platformCommissionRate 平台佣金比例
     * @param commissionPrice        三方提供的佣金值
     * @return 佣金值
     */
    private BigDecimal calCommonCommissionPrice(String platformCommissionRate, BigDecimal commissionPrice) {
        if (null == commissionPrice) {
            log.error("三方提供的佣金值为空");
            return new BigDecimal("0.00");
        }

        if (StringUtils.isBlank(platformCommissionRate)) {
            return new BigDecimal("0.00");
        }

        return commissionPrice.multiply(new BigDecimal(platformCommissionRate)).setScale(2, BigDecimal.ROUND_DOWN);
    }

    /**
     * 判断是否是自购单-两个信息得到的用户id是否是同一个用户
     *
     * @param superiorDtoBySpecialId  根据specialId查询到的用户信息
     * @param superiorDtoByRelationId 根据relationId查询到的用户信息
     * @return true-自购单，false-非自购单
     */
    private boolean isPurchaseBySelf(SuperiorDto superiorDtoBySpecialId, SuperiorDto superiorDtoByRelationId) {

        //会员id,渠道id都不为空且两个对应的userId相等 则为自购订单
        return null != superiorDtoBySpecialId && null != superiorDtoByRelationId
                && null != superiorDtoBySpecialId.getUserId()
                && superiorDtoBySpecialId.getUserId().equals(superiorDtoByRelationId.getUserId());
    }

    @Override
    public Double getCouponPrice(Long goodsId) {
        if (Objects.isNull(goodsId)) {
            log.warn("获取商品优惠券，商品ID错误，ID为[{}]", goodsId);
            return 0D;
        }

        // 查询商品信息
        GetCommodityDetailParam param = new GetCommodityDetailParam();
        param.setCommodityId(goodsId);
        param.setQueryCouponInfo(false);
        CommodityInfoDTO commodityDetail = commodityInfoSourceProxyService.getCommodityDetail(param);

        if (Objects.isNull(commodityDetail)) {
            log.warn("获取优惠券金额为空, 商品id为[{}]", goodsId);
            return 0D;
        }

        return Objects.isNull(commodityDetail.getCouponPrice()) ? 0D : commodityDetail.getCouponPrice();

    }
}
