package com.bxm.fossicker.commodity.service.impl;

import com.bxm.fossicker.commodity.convert.CommodityDetailToSearchConvert;
import com.bxm.fossicker.commodity.domain.HistoryBrowsingMapper;
import com.bxm.fossicker.commodity.model.dto.CommodityDetailDTO;
import com.bxm.fossicker.commodity.model.dto.CommoditySearchDTO;
import com.bxm.fossicker.commodity.model.dto.HistoryBrowsingDTO;
import com.bxm.fossicker.commodity.model.param.CommodityDetailQueryParam;
import com.bxm.fossicker.commodity.model.param.HistoryBrowsingParam;
import com.bxm.fossicker.commodity.model.vo.HistoryBrowsingVO;
import com.bxm.fossicker.commodity.service.CommodityInfoService;
import com.bxm.fossicker.commodity.service.HistoryBrowsingService;
import com.bxm.fossicker.user.facade.vip.VipFacadeService;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.github.pagehelper.Page;
import com.github.pagehelper.PageHelper;
import com.google.common.collect.Lists;
import lombok.extern.log4j.Log4j2;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

import static com.bxm.fossicker.commodity.common.constants.CommodityRedisKeyConstants.BROWSE_HISTORY_SET;

/**
 * @author: yangxin
 * @date: 2019/12/20  15:32
 */
@Service
@Log4j2
public class HistoryBrowsingServiceImpl implements HistoryBrowsingService {

    @Autowired
    private HistoryBrowsingMapper historyBrowsingMapper;

    @Autowired
    private SequenceCreater sequenceCreater;

    @Autowired
    private VipFacadeService vipFacadeService;

    @Autowired
    private RedisSetAdapter redisSetAdapter;

    @Autowired
    private CommodityInfoService commodityInfoService;


    @Override
    public List<HistoryBrowsingDTO> getFootprint(HistoryBrowsingParam param) {
        if (Objects.isNull(param.getUserId())) {
            log.warn("获取历史浏览记录，用户ID[{}]为空", param.getUserId());
            return Lists.newArrayList();
        }

        if (Objects.isNull(param.getPageSize())) {
            param.setPageSize(10);
        }

        if (Objects.isNull(param.getPageOn())) {
            param.setPageOn(1);
        }

        List<HistoryBrowsingDTO> dtoList;

        while (true) {
            Page<HistoryBrowsingVO> histories = PageHelper.startPage(param.getPageOn(), param.getPageSize())
                    .doSelectPage(() -> historyBrowsingMapper.getFootprint(param.getUserId()));

            if (CollectionUtils.isEmpty(histories)) {
                return Lists.newArrayList();
            }

            boolean vipFlag = vipFacadeService.isVip(param.getUserId());

            dtoList = histories.stream()
                    // 收集成<Date, GoodsIds> map
                    .collect(Collectors.groupingBy(p -> DateUtils.getClearDate(p.getCreateTime()),
                            Collectors.mapping(HistoryBrowsingVO::getGoodsId, Collectors.toList())))
                    .entrySet().stream()
                    // 倒叙排序
                    .sorted((a, b) -> a.getKey().compareTo(b.getKey()) * -1)
                    // 转换对象
                    .map(goodsIdList -> {
                        List<CommoditySearchDTO> historyCommodities = goodsIdList.getValue().stream()
                                // 转换商品对象
                                .map(goodsId -> build(goodsId, param.getUserId(), vipFlag))
                                // 过滤为null的
                                .filter(Objects::nonNull)
                                .collect(Collectors.toList());

                        String day = DateUtils.DATE_FORMAT_MONTH_DAY_FORMAT_THREAD_LOCAL.get().format(goodsIdList.getKey());
                        day = day.replace("月", ".");
                        day = day.replace("日", "");

                        // 生成对象
                        HistoryBrowsingDTO browsingDTO = new HistoryBrowsingDTO();
                        browsingDTO.setDate(day);
                        browsingDTO.setCommoditySearchDTO(historyCommodities);

                        return browsingDTO;
                    })
                    // 过滤掉空的那天
                    .filter(p -> !CollectionUtils.isEmpty(p.getCommoditySearchDTO()))
                    .collect(Collectors.toList());

            // 如果集合过滤完，还有数据 则返回
            if (!CollectionUtils.isEmpty(dtoList)) {
                break;
            }

            // 否则请求下一页
            param.setPageOn(param.getPageOn() + 1);
        }

        return dtoList;
    }

    /**
     * 存储用户每日浏览历史的缓存
     *
     * @param userId 用户ID
     * @return 缓存键值
     */
    private KeyGenerator buildKey(Long userId) {
        return BROWSE_HISTORY_SET.copy().appendKey(userId).appendKey(DateUtils.formatDate(new Date()));
    }

    @Override
    public Boolean insertFootprint(Long userId, Long goodsId) {
        if (null == userId || null == goodsId) {
            log.warn("添加商品浏览记录，userID或goodsID错误。userID为[{}]，goodsID为[{}]", userId, goodsId);
            return false;
        }
        KeyGenerator key = buildKey(userId);

        boolean existsHistory = redisSetAdapter.exists(key, goodsId);
        //存在历史记录则更新
        int recordCount;
        if (existsHistory) {
            Date now = new Date();
            now = DateUtils.clearTimePart(now);
            recordCount = historyBrowsingMapper.updateFootprintByUserId(userId, goodsId, now);
        } else {
            recordCount = insertHistory(userId, goodsId);
            redisSetAdapter.add(key, goodsId);
            redisSetAdapter.expire(key, DateUtils.getCurSeconds());
        }

        return recordCount > 0;
    }

    /**
     * 添加历史记录
     */
    private int insertHistory(Long userId, Long goodsId) {
        HistoryBrowsingVO addHistoryBrowsingVO = new HistoryBrowsingVO();
        addHistoryBrowsingVO.setId(sequenceCreater.nextLongId());
        addHistoryBrowsingVO.setUserId(userId);
        addHistoryBrowsingVO.setGoodsId(goodsId);
        addHistoryBrowsingVO.setCreateTime(new Date());
        return historyBrowsingMapper.insertFootprint(addHistoryBrowsingVO);
    }

    /**
     * 查询商品详情
     *
     * @param goodsId 商品ID
     * @param userId  用户ID
     * @param vipFlag 是否VIP用户
     * @return 商品查询对象信息
     */
    private CommoditySearchDTO build(Long goodsId, Long userId, boolean vipFlag) {
        CommodityDetailQueryParam param = CommodityDetailQueryParam.builder()
                .userId(userId)
                .vipUser(vipFlag)
                .goodsId(goodsId)
                .queryBrowerAward(true)
                .setLable(true)
                .build();

        CommodityDetailDTO commodityDetail = commodityInfoService.getCommodityDetail(param);
        return CommodityDetailToSearchConvert.convert(commodityDetail);
    }


    public List<HistoryBrowsingDTO> build(HashMap<String, List<Long>> sameDayHistory, Long userId) {
        List<HistoryBrowsingDTO> histories = new ArrayList<>();

        for (Map.Entry<String, List<Long>> entry : sameDayHistory.entrySet()) {
            List<CommoditySearchDTO> commoditySearchDTOList = Lists.newArrayList();
            try {
                HistoryBrowsingDTO historyBrowsingDTO = new HistoryBrowsingDTO();
                historyBrowsingDTO.setDate(entry.getKey());
                historyBrowsingDTO.setCommoditySearchDTO(commoditySearchDTOList);
                histories.add(historyBrowsingDTO);
            } catch (Exception e) {
                log.warn("商品详情历史浏览记录，时间格式化错误");
            }

        }
        return histories;
    }

}
