package com.bxm.fossicker.commodity.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.bxm.fossicker.commodity.common.constants.CommodityRedisKeyConstants;
import com.bxm.fossicker.commodity.common.dto.PlatformCommissionRateInfo;
import com.bxm.fossicker.commodity.common.enums.SecKillTimeEnum;
import com.bxm.fossicker.commodity.common.utils.CommodityCalUtils;
import com.bxm.fossicker.commodity.domain.CommodityInfoMapper;
import com.bxm.fossicker.commodity.model.dto.CommodityGoldInfoDTO;
import com.bxm.fossicker.commodity.model.dto.SecCommodityPage;
import com.bxm.fossicker.commodity.model.dto.SecKillCommodityDTO;
import com.bxm.fossicker.commodity.model.entry.CommodityInfo;
import com.bxm.fossicker.commodity.model.param.QuerySecKillCommodityParam;
import com.bxm.fossicker.commodity.service.BlackCommodityService;
import com.bxm.fossicker.commodity.service.CommodityGoldService;
import com.bxm.fossicker.commodity.service.CommodityPriceInfoService;
import com.bxm.fossicker.commodity.service.LocalTqgCommodityInfoService;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.github.pagehelper.Page;
import com.github.pagehelper.PageHelper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;


@Service
@Slf4j
public class LocalTqgCommodityServiceImpl implements LocalTqgCommodityInfoService {

    @Autowired
    private CommodityInfoMapper commodityInfoMapper;

    @Autowired
    private CommodityPriceInfoService commodityPriceInfoService;

    @Autowired
    private RedisStringAdapter redisStringAdapter;

    @Autowired
    private CommodityGoldService commodityGoldService;

    @Autowired
    private BlackCommodityService blackCommodityService;

    /**
     * 限时秒杀-聚划算首页展示过期时间-1小时
     */
    private static final Long JU_COMMODITY_EXPIRES_SEC = 3600L;


    @Override
    public SecCommodityPage<SecKillCommodityDTO> getTqgCommodityList(QuerySecKillCommodityParam secKillCommodityParam, SecKillTimeEnum secKillTimeEnum) {

        SecCommodityPage<SecKillCommodityDTO> resultPage = new SecCommodityPage<>();
        resultPage.setPageNum(secKillCommodityParam.getPageNum());
        resultPage.setPageSize(secKillCommodityParam.getPageSize());
        //将秒杀时间范围枚举的结束时间hour作为key,防止秒杀跨时间之后的信息未更新
        KeyGenerator secKillKey = CommodityRedisKeyConstants.COMMODITY_SEC_KILL_TAO.copy()
                .appendKey(secKillTimeEnum.getStartTimeHour()).appendKey(secKillCommodityParam.getPageNum());

        Long userId = secKillCommodityParam.getUserId();
        //从redis中获取,获取到直接返回缓存信息
        String commodityListFromRedisStr = redisStringAdapter.getString(secKillKey);
        if (StringUtils.isNotBlank(commodityListFromRedisStr)) {
            log.info("从缓存中获取淘抢购信息,redisKey:{}", secKillKey);

            List<SecKillCommodityDTO> redisInfoList = JSONArray.parseArray(commodityListFromRedisStr, SecKillCommodityDTO.class);
            redisInfoList.forEach(secKillCommodityDTO -> getOtherInfo(secKillCommodityDTO, userId));
            //TODO 可以使用布隆过滤器处理
            redisInfoList.removeIf(secKillCommodityDTO -> blackCommodityService.isBlackCommodity(secKillCommodityDTO.getGoodsId()));
            resultPage.setList(redisInfoList);
            return resultPage;
        }
        Page<CommodityInfo> page = PageHelper.startPage(secKillCommodityParam)
                .doSelectPage(() -> this.commodityInfoMapper.getTqgCommodityListByNow(secKillCommodityParam));

        Integer secKillType = secKillCommodityParam.getType();
        //数据转换

        List<CommodityInfo> queryList = page.getResult();
        if (null == queryList) {
            return resultPage;

        }

        //不为空
        List<SecKillCommodityDTO> secKillCommodityDTOList = queryList.stream().map(commodityInfo -> convert(commodityInfo, userId, secKillType)).collect(Collectors.toList());
        secKillCommodityDTOList.removeIf(Objects::isNull);

        //设置入redis
        redisStringAdapter.set(secKillKey, JSON.toJSONString(secKillCommodityDTOList), JU_COMMODITY_EXPIRES_SEC);
        resultPage.setList(secKillCommodityDTOList);

        return resultPage;
    }

    private SecKillCommodityDTO convert(CommodityInfo commodityInfo, Long userId, Integer secKillType) {

        SecKillCommodityDTO secKillCommodityDTO = SecKillCommodityDTO.builder()
                .commodityTitle(commodityInfo.getCommodityTitle())
                .goodsId(Long.valueOf(commodityInfo.getGoodId()))
                .imgUrl(commodityInfo.getMainPic())
                .totalAmount(commodityInfo.getTotalAmount())
                .soldNum(commodityInfo.getSoldNum())
                .startTime(DateUtils.formatDateTime(commodityInfo.getStartTime()))
                .endTime(DateUtils.formatDateTime(commodityInfo.getEndTime()))
                //已售数量小于总数量,则商品售卖状态为1-可售,否则为0-不可售(已抢光)
                .soldStatus(commodityInfo.getSoldNum() < commodityInfo.getTotalAmount() ? 1 : 0)
                .reservePrice(commodityInfo.getReservePrice().doubleValue())
                .couponPrice(commodityInfo.getCouponPrice().doubleValue())
                .build();

        //设置销售进度条
        secKillCommodityDTO.setSoldPercent(getSoldPercent(commodityInfo.getSoldNum(), commodityInfo.getTotalAmount()));

        //未开抢状态
        if (3 == secKillType) {
            secKillCommodityDTO.setSoldStatus(1);
            secKillCommodityDTO.setSoldNum(0L);
            secKillCommodityDTO.setSoldPercent(0D);
        }

        PlatformCommissionRateInfo platformCommission = commodityPriceInfoService.getPlatformCommissionRateInfo(secKillCommodityDTO.getGoodsId());
        if (null == platformCommission) {
            return null;
        }

        Double couponPrice = commodityInfo.getCouponPrice().doubleValue();
        Double reservePrice = commodityInfo.getReservePrice().doubleValue();
        Double commissionPrice = CommodityCalUtils.calCommissionForPlatformCommission(platformCommission, commodityInfo.getCommissionPrice().doubleValue());
        Double commodityPrice = CommodityCalUtils.calPriceWithSubtractingCouponAndCommission(reservePrice, couponPrice, commissionPrice);

        secKillCommodityDTO.setCouponPrice(couponPrice);
        secKillCommodityDTO.setCommissionPrice(commissionPrice);
        secKillCommodityDTO.setCommodityPrice(commodityPrice);

        //获取动态信息
        getOtherInfo(secKillCommodityDTO, userId);


        return secKillCommodityDTO;

    }


    /**
     * 获取销售进度条
     * @param soldNum 出售数量
     * @param totalNum 总数量
     * @return
     */
    private Double getSoldPercent(Long soldNum, Long totalNum) {
        if (null == soldNum || null == totalNum) {
            return 0D;
        }

        BigDecimal soldDecimal = new BigDecimal(soldNum.toString());
        BigDecimal totalDecimal = new BigDecimal(totalNum.toString());

        BigDecimal result = soldDecimal.divide(totalDecimal, 3, BigDecimal.ROUND_HALF_UP);
        //大于0小于0.01则展示1%
        if (-1 == result.compareTo(new BigDecimal("0.01")) && 0 != result.compareTo(new BigDecimal("0"))) {
            return 1D;
        }

        if (0 == result.compareTo(new BigDecimal(1)) || 1 == result.compareTo(new BigDecimal(1))) {
            return 100D;
        }

        return result.multiply(new BigDecimal("100")).doubleValue();

    }


    /**
     * 获取其他动态信息
     * @param secKillCommodityDTO 秒杀商品信息DTO
     * @param userId 用户id
     * @return
     */
    public SecKillCommodityDTO getOtherInfo(SecKillCommodityDTO secKillCommodityDTO, Long userId) {
        CommodityGoldInfoDTO commodityGoldInfoDTO = commodityGoldService.getGoldInfoByCommodityId(secKillCommodityDTO.getGoodsId(), userId);
        if (null != commodityGoldInfoDTO) {
            //设置金币数和金币可领状态
            secKillCommodityDTO.setGoldNum(commodityGoldInfoDTO.getBrowseGold());
            secKillCommodityDTO.setGoldStatus(commodityGoldInfoDTO.getDrawFlag());
        }

        //设置提醒信息
        KeyGenerator remindCommodityKey = CommodityRedisKeyConstants.COMMODITY_REMIND.copy()
                .appendKey(userId)
                .appendKey(secKillCommodityDTO.getGoodsId());

        //新人专享角标
        secKillCommodityDTO.setNewbieRight(commodityPriceInfoService.isNewbieRight(userId, secKillCommodityDTO.getGoodsId()));

        //若redis中有商品提醒记录,则提醒状态为1-已设置提醒,否则为0-未设置提醒
        secKillCommodityDTO.setRemindStatus(StringUtils.isNotBlank(redisStringAdapter.getString(remindCommodityKey)) ? 1 : 0);



        return secKillCommodityDTO;
    }



}
