package com.bxm.fossicker.commodity.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.bxm.fossicker.commodity.common.constants.CommodityRedisKeyConstants;
import com.bxm.fossicker.commodity.common.dto.CommodityCommissionAndCoupon;
import com.bxm.fossicker.commodity.common.dto.PlatformCommissionRateInfo;
import com.bxm.fossicker.commodity.common.enums.SecKillTimeEnum;
import com.bxm.fossicker.commodity.common.utils.CommodityCalUtils;
import com.bxm.fossicker.commodity.common.utils.SignMD5Util;
import com.bxm.fossicker.commodity.config.CommodityInfoProperties;
import com.bxm.fossicker.commodity.model.dto.*;
import com.bxm.fossicker.commodity.model.entry.CommodityInfo;
import com.bxm.fossicker.commodity.model.param.InexpensiveCommodityParam;
import com.bxm.fossicker.commodity.model.param.QuerySecKillCommodityParam;
import com.bxm.fossicker.commodity.model.param.RemindCommodityParam;
import com.bxm.fossicker.commodity.service.*;
import com.bxm.fossicker.commodity.service.commodity.info.CommodityInfoSourceProxyService;
import com.bxm.fossicker.commodity.service.externel.TbCommodityService;
import com.bxm.fossicker.commodity.service.strategy.CommodityLabelService;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.taobao.api.request.TbkJuTqgGetRequest;
import com.taobao.api.response.TbkJuTqgGetResponse;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.*;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 限时秒杀商品服务类
 * @author donghy
 */
@Service
@Slf4j
public class SecKillCommodityServiceImpl implements SecKillCommodityService {

    /**
     * redis String服务
     */
    @Autowired
    private RedisStringAdapter redisStringAdapter;

    /**
     * 商品金币服务
     */
    @Autowired
    private CommodityGoldService commodityGoldService;

    /**
     * 商品佣金服务服务
     */
    @Autowired
    private CommodityPriceInfoService commodityPriceInfoService;

    /**
     * 淘宝客服务
     */
    @Autowired
    private TbCommodityService tbCommodityService;

    /**
     * 黑名单服务
     */
    @Autowired
    private BlackCommodityService blackCommodityService;


    /**
     * 商品信息相关配置服务
     */
    @Autowired
    private CommodityInfoProperties commodityInfoProperties;

    @Autowired
    private LocalCommodityInfoService localCommodityInfoService;

    @Autowired
    private CommodityInfoSourceProxyService commodityInfoSourceProxyService;



    @Autowired
    private LocalTqgCommodityInfoService localTqgCommodityInfoService;

    @Autowired
    private CommodityLabelService commodityLabelService;

    /**
     * 限时秒杀-淘抢购首页展示过期时间-20分钟
     */
    private static final Long SEC_KILL_COMMODITY_EXPIRES_SEC = 1200L;

    /**
     * 限时秒杀-聚划算首页展示过期时间-1小时
     */
    private static final Long JU_COMMODITY_EXPIRES_SEC = 3600L;

    private static final BigDecimal ONE_HUNDRED = BigDecimal.valueOf(100);


    @Override
    public SecKillCommodityIndexListDTO getSecKillCommodityForIndex(Long userId) {

        //---根据当前时间判断所在哪个秒杀时间范围内
        LocalTime nowTime = LocalTime.now();
        SecKillTimeEnum secKillTimeEnum = SecKillTimeEnum.isWhichInTime(nowTime);

        //格式化枚举类中的时间
        LocalTime startTime = LocalTime.of(secKillTimeEnum.getStartTimeHour(), secKillTimeEnum.getStartTimeMin(),
                secKillTimeEnum.getStartTimeSec());
        LocalTime endTime = LocalTime.of(secKillTimeEnum.getEndTimeHour(), secKillTimeEnum.getEndTimeMin(),
                secKillTimeEnum.getEndTimeSec());

        //格式化时间格式,若是22天档的,那么结束时间为第二天的10点
        LocalDateTime endTimeDateTime;
        if (secKillTimeEnum == SecKillTimeEnum.SEC_KILL_22) {
            endTimeDateTime = LocalDate.now().plusDays(1).atTime(endTime);
        } else {
            endTimeDateTime = LocalDate.now().atTime(endTime);
        }

        //格式化时间成date
        ZoneId zone = ZoneId.systemDefault();
        Date startTimeDate = Date.from(LocalDate.now().atTime(startTime).atZone(zone).toInstant());
        Date endTimeDate = Date.from(endTimeDateTime.atZone(zone).toInstant());


        QuerySecKillCommodityParam secKillCommodityParam = new QuerySecKillCommodityParam();
        secKillCommodityParam.setStartTime(DateUtils.formatDateTime(startTimeDate));
        secKillCommodityParam.setEndTime(DateUtils.formatDateTime(endTimeDate));
        secKillCommodityParam.setType(1);
        secKillCommodityParam.setUserId(userId);

        SecCommodityPage<SecKillCommodityDTO> secCommodityPage = getSecKillCommodityList(secKillCommodityParam);

        if (Objects.isNull(secCommodityPage) && CollectionUtils.isEmpty(secCommodityPage.getList())) {
            // 淘抢购

            List<SecCommodityTimeDTO> commodityTimeDTOS = getSecTimeList();

            for (SecCommodityTimeDTO commodityTimeDTO : commodityTimeDTOS) {
                secKillCommodityParam = new QuerySecKillCommodityParam();
                secKillCommodityParam.setStartTime(commodityTimeDTO.getStartTime());
                secKillCommodityParam.setEndTime(commodityTimeDTO.getEndTime());
                secKillCommodityParam.setType(commodityTimeDTO.getType());
                secKillCommodityParam.setUserId(userId);

                secCommodityPage = getSecKillCommodityList(secKillCommodityParam);

                if (!CollectionUtils.isEmpty(secCommodityPage.getList())) {
                    break;
                }
            }
        }

        SecKillCommodityIndexListDTO secKillCommodityIndexListDTO = new SecKillCommodityIndexListDTO();

        if (Objects.nonNull(secCommodityPage) && !CollectionUtils.isEmpty(secCommodityPage.getList())) {
            List<SecKillCommodityIndexDTO> taoResultList = secCommodityPage.getList()
                    .stream()
                    .limit(3)
                    .map(this::convert)
                    .collect(Collectors.toList());

            //获取商品浏览金币-金币数额以及是否展示
            if (!CollectionUtils.isEmpty(taoResultList)) {
                taoResultList.forEach(secKillCommodityIndexDTO -> getGoldInfo(secKillCommodityIndexDTO, userId));
            }
            secKillCommodityIndexListDTO.setTaoCommodityList(taoResultList);
        }


        InexpensiveCommodityParam inexpensiveCommodityParam = new InexpensiveCommodityParam();
        inexpensiveCommodityParam.setPageId("1");
        inexpensiveCommodityParam.setPageSize(200);
        inexpensiveCommodityParam.setSortType(0);
        inexpensiveCommodityParam.setUserId(userId);
        DtkPageInfo<InexpensiveCommodityDTO> dtoDtkPageInfo = getJuCommodity(inexpensiveCommodityParam);

        List<SecKillCommodityIndexDTO> juSecKillCommodityList = dtoDtkPageInfo.getList()
                .stream()
                .limit(3)
                .map(this::convert)
                .collect(Collectors.toList());

        //聚划算
        if (!CollectionUtils.isEmpty(juSecKillCommodityList)) {
            juSecKillCommodityList.forEach(secKillCommodityIndexDTO -> getGoldInfo(secKillCommodityIndexDTO, userId));
        }
        secKillCommodityIndexListDTO.setJuCommodityList(juSecKillCommodityList);

        //计算倒计时时间并设置入返回值
        secKillCommodityIndexListDTO.setCountDownTime(calCountDownTime(secKillTimeEnum));

        return secKillCommodityIndexListDTO;
    }

    @Override
    public List<SecCommodityTimeDTO> getSecTimeList() {
        LocalTime nowTime = LocalTime.now();
        SecKillTimeEnum nowTimeEnum = SecKillTimeEnum.isWhichInTime(nowTime);

        return sortListByEnum(nowTimeEnum);
    }

    @Override
    public SecCommodityPage<SecKillCommodityDTO> getSecKillCommodityList(QuerySecKillCommodityParam secKillCommodityParam) {

        //标识当前时间在哪个时间段
        DateTimeFormatter df = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
        LocalDateTime startTime = LocalDateTime.parse(secKillCommodityParam.getStartTime(), df);
        SecKillTimeEnum secKillTimeEnum = SecKillTimeEnum.isWhichInTime(startTime.toLocalTime());


        SecCommodityPage<SecKillCommodityDTO> resultPage = localTqgCommodityInfoService.getTqgCommodityList(secKillCommodityParam, secKillTimeEnum);
        //增加标签数据
        resultPage.getList().forEach(p->p.setLabelType(commodityLabelService.goodsLabel(p.getNewbieRight(),0,p.getGoodsId())));
        resultPage.setCountDownTime(calCountDownTimeByType(secKillCommodityParam));


        return resultPage;
    }


    @Override
    public DtkPageInfo<InexpensiveCommodityDTO> getJuCommodity(InexpensiveCommodityParam inexpensiveCommodityParam) {
        DtkPageInfo<InexpensiveCommodityDTO> resultPage = queryJuCommodityByDtk(inexpensiveCommodityParam);
        resultPage.getList().forEach(p->p.setLabelType(commodityLabelService.goodsLabel(p.getNewbieRight(),0,p.getGoodsId())));
        return resultPage;
    }

    @Override
    public Boolean remindCommodity(RemindCommodityParam remindCommodityParam) {
        KeyGenerator remindCommodityKey = CommodityRedisKeyConstants.COMMODITY_REMIND.copy()
                .appendKey(remindCommodityParam.getUserId())
                .appendKey(remindCommodityParam.getGoodsId());

        redisStringAdapter.set(remindCommodityKey, remindCommodityParam.getStartTime(), 86400L);
        return true;
    }

    @Override
    public int getRemindCommodityStatus(Long goodsId, Long userId) {
        KeyGenerator remindCommodityKey = CommodityRedisKeyConstants.COMMODITY_REMIND.copy()
                .appendKey(userId)
                .appendKey(goodsId);

        String remindCommodityRecord = redisStringAdapter.getString(remindCommodityKey);

        return StringUtils.isBlank(remindCommodityRecord) ? 0 : 1;
    }

    /**
     * 获取秒杀产品首页展示-淘抢购
     *
     * @param secKillTimeEnum 秒杀时间枚举类
     * @return
     */
    private List<SecKillCommodityIndexDTO> queryIndexSecKillCommodityByTao(SecKillTimeEnum secKillTimeEnum, Long userId) {

        //将秒杀时间范围枚举的结束时间hour作为key,防止秒杀跨时间之后的信息未更新
        KeyGenerator secKillKey = CommodityRedisKeyConstants.COMMODITY_SEC_KILL_INDEX_TAO.copy().appendKey(secKillTimeEnum.getStartTimeHour());
        //从redis中获取,获取到直接返回缓存信息
        String commodityListFromRedisStr = redisStringAdapter.getString(secKillKey);
        if (StringUtils.isNotBlank(commodityListFromRedisStr)) {
            log.info("从缓存中获取到的首页淘抢购信息,redisKey:{}", secKillKey);
            List<SecKillCommodityIndexDTO> commodityList =
                    JSONArray.parseArray(commodityListFromRedisStr, SecKillCommodityIndexDTO.class);
            return commodityList.stream().limit(3).collect(Collectors.toList());
        }

        //组装参数
        TbkJuTqgGetRequest req = assembleTqgParamForIndex(secKillTimeEnum);
        TbkJuTqgGetResponse rsp = tbCommodityService.getTqgCommodity(req);

        List<SecKillCommodityIndexDTO> resultList = new ArrayList<>();
        if (null == rsp || !rsp.isSuccess() || CollectionUtils.isEmpty(rsp.getResults())) {
            log.error("从淘宝接口获取到的商品列表信息为空");
            return resultList;
        }

        //参数转换
        resultList = convertSecKillIndexCommodityInfoList(rsp.getResults(), userId);

        if (!CollectionUtils.isEmpty(resultList)) {
            redisStringAdapter.set(secKillKey, JSON.toJSONString(resultList), SEC_KILL_COMMODITY_EXPIRES_SEC);

        }

        return resultList;
    }

    /**
     * 获取聚划算商品详情列表
     * @param inexpensiveCommodityParam
     * @return
     */
    private DtkPageInfo<InexpensiveCommodityDTO> queryJuCommodityByDtk(InexpensiveCommodityParam inexpensiveCommodityParam) {

        Long userId = inexpensiveCommodityParam.getUserId();
        DtkPageInfo<InexpensiveCommodityDTO> resultPage = new DtkPageInfo();


        List<InexpensiveCommodityDTO> resultList;


        //从redis中获取,获取到直接返回缓存信息
        KeyGenerator juCommodityRedisKey = createJuCommodityRedisKey(inexpensiveCommodityParam);
        String commodityListFromRedisStr = redisStringAdapter.getString(juCommodityRedisKey);

        if (StringUtils.isNotBlank(commodityListFromRedisStr)) {
            log.info("从缓存中获取到聚划算信息,redisKey:{}",juCommodityRedisKey);

            resultList = JSONArray.parseArray(commodityListFromRedisStr, InexpensiveCommodityDTO.class);


        } else {

            List<CommodityInfo> commodityInfos = localCommodityInfoService.queryJuhuasuanCommodity();

            //参数转换
            resultList = commodityInfos.stream()
                    .map(p -> convert(p, userId))
                    .collect(Collectors.toList());


            //若不为空则设置入redis
            if (!CollectionUtils.isEmpty(resultList)) {
                redisStringAdapter.set(juCommodityRedisKey, JSON.toJSONString(resultList), JU_COMMODITY_EXPIRES_SEC);

                // 数据预热
                commodityInfoSourceProxyService.preheatCommodityInfo(resultList.stream()
                        .map(InexpensiveCommodityDTO::getGoodsId)
                        .collect(Collectors.toList()));

            }

            resultPage.setList(resultList);
            resultPage.setPageId("");
            resultPage.setPageSize(200);
        }

        //剔除黑名单商品
        resultList.removeIf(inexpensiveCommodityDTO -> blackCommodityService.isBlackCommodity(inexpensiveCommodityDTO.getGoodsId()));

        //金币信息
        resultList.forEach(inexpensiveCommodityDTO -> getGoldInfo(inexpensiveCommodityDTO, userId));


        resultPage.setList(resultList);
        resultPage.setPageId("");
        resultPage.setPageSize(200);

        return resultPage;
    }

    private InexpensiveCommodityDTO convert(CommodityInfo info, Long userId) {


        long goodsId = Long.valueOf(info.getGoodId());
        InexpensiveCommodityDTO inexpensiveCommodityDTO = InexpensiveCommodityDTO.builder()
                .commodityTitle(info.getCommodityTitle())
                .goodsId(Long.valueOf(info.getGoodId()))
                .imgUrl(info.getMainPic())
                .shopType(Objects.isNull(info.getShopType()) ? 0 : info.getShopType().intValue())
                .soldNum(Objects.toString(info.getSoldNum()))
                .build();


        //获取销售进度条
        BigDecimal soldPercent = BigDecimal.ZERO;
        if (Objects.nonNull(info.getTotalAmount()) && info.getTotalAmount() != 0) {
            soldPercent = BigDecimal.valueOf(info.getSoldNum())
                    .divide(BigDecimal.valueOf(info.getTotalAmount()), 2, RoundingMode.HALF_DOWN)
                    .multiply(ONE_HUNDRED)
                    .setScale(0, RoundingMode.HALF_DOWN);
        }

        // 如果销量很小，给个1
        Double soldPercentd = 1.0;
        if (soldPercent.intValue() == 0 && info.getSoldNum() > 0) {
            soldPercentd = soldPercentd.doubleValue();
        }

        if (soldPercent.intValue() > 100) {
            soldPercentd = 100D;
        }

        inexpensiveCommodityDTO.setSoldPercent(soldPercentd);

        //获取商品原价
        inexpensiveCommodityDTO.setReservePrice(Objects.isNull(info.getReservePrice()) ? 0.00 : info.getReservePrice().doubleValue());

        //佣金以及优惠券信息

        //佣金以及优惠券信息
        PlatformCommissionRateInfo commissionInfoDTO = commodityPriceInfoService.getPlatformCommissionRateInfo(goodsId);

        inexpensiveCommodityDTO.setCouponPrice(Objects.isNull(info.getCouponPrice()) ? 0.00 : info.getCouponPrice().doubleValue());
        // 用户可得佣金
        inexpensiveCommodityDTO.setCommissionPrice(CommodityCalUtils.calCommissionForPlatformCommission(commissionInfoDTO.getCommissionRegularPrice(),
                info.getCommissionPrice().doubleValue(),
                commissionInfoDTO.getCommissionRate()));

        // 商品最终价格
        inexpensiveCommodityDTO.setCommodityPrice(CommodityCalUtils.calPriceWithSubtractingCouponAndCommission(info.getDiscountPrice().doubleValue(),
                inexpensiveCommodityDTO.getCommissionPrice().doubleValue()));


        //封装金币信息
        getGoldInfo(inexpensiveCommodityDTO, userId);


        return inexpensiveCommodityDTO;
    }


    private List<SecKillCommodityDTO> querySecKillCommodityByTqg(QuerySecKillCommodityParam secKillCommodityParam,
                                                                 SecKillTimeEnum secKillTimeEnum){
        //将秒杀时间范围枚举的结束时间hour作为key,防止秒杀跨时间之后的信息未更新
        KeyGenerator secKillKey = CommodityRedisKeyConstants.COMMODITY_SEC_KILL_TAO.copy()
                .appendKey(secKillTimeEnum.getStartTimeHour()).appendKey(secKillCommodityParam.getPageNum());

        Long userId = secKillCommodityParam.getUserId();
        //从redis中获取,获取到直接返回缓存信息
        String commodityListFromRedisStr = redisStringAdapter.getString(secKillKey);
        if (StringUtils.isNotBlank(commodityListFromRedisStr)) {
            log.info("从缓存中获取淘抢购信息,redisKey:{}", secKillKey);

            List<SecKillCommodityDTO> redisInfoList = JSONArray.parseArray(commodityListFromRedisStr, SecKillCommodityDTO.class);
            redisInfoList.forEach(secKillCommodityDTO -> getOtherInfo(secKillCommodityDTO, userId));
            redisInfoList.removeIf(secKillCommodityDTO -> blackCommodityService.isBlackCommodity(secKillCommodityDTO.getGoodsId()));
            return redisInfoList;
        }

        //组装参数,请求淘宝客接口
        TbkJuTqgGetRequest req = assembleTqgParam(secKillCommodityParam, secKillTimeEnum);
        TbkJuTqgGetResponse rsp = tbCommodityService.getTqgCommodity(req);

        List<SecKillCommodityDTO> resultList = new ArrayList<>();
        if (null == rsp || !rsp.isSuccess() || CollectionUtils.isEmpty(rsp.getResults())) {
            log.error("从淘宝接口获取到的商品列表信息为空");
            return resultList;
        }

        //参数转换
        resultList = convertSecKillCommodityInfoList(rsp.getResults(), secKillCommodityParam);

        if (!CollectionUtils.isEmpty(resultList)) {
            redisStringAdapter.set(secKillKey, JSON.toJSONString(resultList), SEC_KILL_COMMODITY_EXPIRES_SEC);

        }
        return resultList;
    }


    /**
     * 获取动态参数-金币、新人专享角标
     * @param inexpensiveCommodityDTO 聚划算商品信息
     * @param userId 用户id
     * @return 设置金币数和金币可领状态的商品信息
     */
    private InexpensiveCommodityDTO getGoldInfo(InexpensiveCommodityDTO inexpensiveCommodityDTO, Long userId) {

        CommodityGoldInfoDTO commodityGoldInfoDTO = commodityGoldService.getGoldInfoByCommodityId(inexpensiveCommodityDTO.getGoodsId(), userId);
        if (null != commodityGoldInfoDTO) {
            //设置金币数和金币可领状态
            inexpensiveCommodityDTO.setGoldNum(commodityGoldInfoDTO.getBrowseGold());
            inexpensiveCommodityDTO.setGoldStatus(commodityGoldInfoDTO.getDrawFlag());
        }

        //设置新人专享角标
        inexpensiveCommodityDTO.setNewbieRight(commodityPriceInfoService.isNewbieRight(userId, inexpensiveCommodityDTO.getGoodsId()));

        return inexpensiveCommodityDTO;
    }

    /**
     * 获取金币信息-金币数和金币可领状态
     * @param secKillCommodityIndexDTO 商品首页信息
     * @param userId 用户id
     * @return 设置金币数和金币可领状态的商品信息
     */
    private SecKillCommodityIndexDTO getGoldInfo(SecKillCommodityIndexDTO secKillCommodityIndexDTO, Long userId) {

        CommodityGoldInfoDTO commodityGoldInfoDTO = commodityGoldService.getGoldInfoByCommodityId(secKillCommodityIndexDTO.getGoodsId(), userId);
        if (null != commodityGoldInfoDTO) {
            //设置金币数和金币可领状态
            secKillCommodityIndexDTO.setGoldNum(commodityGoldInfoDTO.getBrowseGold());
            secKillCommodityIndexDTO.setGoldStatus(commodityGoldInfoDTO.getDrawFlag());
        }


        return secKillCommodityIndexDTO;
    }

    public SecKillCommodityDTO getOtherInfo(SecKillCommodityDTO secKillCommodityDTO, Long userId) {
        CommodityGoldInfoDTO commodityGoldInfoDTO = commodityGoldService.getGoldInfoByCommodityId(secKillCommodityDTO.getGoodsId(), userId);
        if (null != commodityGoldInfoDTO) {
            //设置金币数和金币可领状态
            secKillCommodityDTO.setGoldNum(commodityGoldInfoDTO.getBrowseGold());
            secKillCommodityDTO.setGoldStatus(commodityGoldInfoDTO.getDrawFlag());
        }

        //设置提醒信息
        KeyGenerator remindCommodityKey = CommodityRedisKeyConstants.COMMODITY_REMIND.copy()
                .appendKey(userId)
                .appendKey(secKillCommodityDTO.getGoodsId());

        //若redis中有商品提醒记录,则提醒状态为1-已设置提醒,否则为0-未设置提醒
        secKillCommodityDTO.setRemindStatus(StringUtils.isNotBlank(redisStringAdapter.getString(remindCommodityKey)) ? 1 : 0);

        //新人专享角标
        secKillCommodityDTO.setNewbieRight(commodityPriceInfoService.isNewbieRight(userId, secKillCommodityDTO.getGoodsId()));



        return secKillCommodityDTO;
    }



    /**
     * 根据时间范围枚举, 组装秒杀开始结束时间
     *
     * @param req             参数请求类-计算出时间值后设置入这个对象
     * @param secKillTimeEnum 限时秒杀时间枚举类
     */
    private void assembleTimeByEnum(TbkJuTqgGetRequest req, SecKillTimeEnum secKillTimeEnum) {


        //格式化枚举类中的时间
        LocalTime startTime = LocalTime.of(secKillTimeEnum.getStartTimeHour(), secKillTimeEnum.getStartTimeMin(),
                secKillTimeEnum.getStartTimeSec());
        LocalTime endTime = LocalTime.of(secKillTimeEnum.getEndTimeHour(), secKillTimeEnum.getEndTimeMin(),
                secKillTimeEnum.getEndTimeSec());

        //格式化时间格式,若是22天档的,那么结束时间为第二天的10点
        LocalDateTime endTimeDateTime;
        if (secKillTimeEnum == SecKillTimeEnum.SEC_KILL_22) {
            endTimeDateTime = LocalDate.now().plusDays(1).atTime(endTime);
        } else {
            endTimeDateTime = LocalDate.now().atTime(endTime);
        }

        //格式化时间成date
        ZoneId zone = ZoneId.systemDefault();
        Date startTimeDate = Date.from(LocalDate.now().atTime(startTime).atZone(zone).toInstant());
        Date endTimeDate = Date.from(endTimeDateTime.atZone(zone).toInstant());

        req.setStartTime(startTimeDate);
        req.setEndTime(endTimeDate);
    }

    /**
     * 组装首页聚划算商品请求参数
     * @return 请求参数
     */
    private Map<String, Object> assembleJuCommodityQueryParamIndex() {

        Map<String, Object> paramMap = new TreeMap<>();


        paramMap.put("version", commodityInfoProperties.getDtkCommodityVersion());
        paramMap.put("appKey", commodityInfoProperties.getDtkCommodityAppKey());
        paramMap.put("pageSize", "6");
        paramMap.put("pageId", "1");
        paramMap.put("juHuaSuan", 1);
        paramMap.put("sign", SignMD5Util.getSignStr(paramMap, commodityInfoProperties.getDtkCommodityAppSecret()));

        return paramMap;

    }

    /**
     * 组装聚划算商品请求参数
     * @param inexpensiveCommodityParam 请求参数
     * @return 请求参数结果集
     */
    private Map<String, Object> assembleJuCommodityQueryParam(InexpensiveCommodityParam inexpensiveCommodityParam) {

        Map<String, Object> paramMap = new TreeMap<>();

        paramMap.put("version", commodityInfoProperties.getDtkCommodityVersion());
        paramMap.put("appKey", commodityInfoProperties.getDtkCommodityAppKey());
        paramMap.put("pageSize", inexpensiveCommodityParam.getPageSize());
        paramMap.put("pageId", inexpensiveCommodityParam.getPageId());
        paramMap.put("juHuaSuan", 1);
        paramMap.put("sort", inexpensiveCommodityParam.getSortType());
        if (null != inexpensiveCommodityParam.getPriceLowerLimit()) {
            paramMap.put("priceLowerLimit", inexpensiveCommodityParam.getPriceLowerLimit());
        }
        if (null != inexpensiveCommodityParam.getPriceUpperLimit()) {
            paramMap.put("priceUpperLimit", inexpensiveCommodityParam.getPriceUpperLimit());
        }

        paramMap.put("sign", SignMD5Util.getSignStr(paramMap, commodityInfoProperties.getDtkCommodityAppSecret()));

        return paramMap;

    }

    /**
     * 转换聚划算商品信息成首页展示信息--仅展示三条信息
     *
     * @param juCommoditySearchDataDTO 聚划算商品data信息DTO
     * @return 限时秒杀-首页展示信息列表
     */
    private List<SecKillCommodityIndexDTO> convertIndexJuCommodityInfoList(JuCommoditySearchDataDTO juCommoditySearchDataDTO,
                                                                           Long userId) {

        List<SecKillCommodityIndexDTO> resultList = new ArrayList<>();

        //三方获取到的数据list 遍历进行值的转换
        List<SecKillByJuInfoDTO> juCommoditySearchDataDTOList = juCommoditySearchDataDTO.getList();
        //截取三条数据

        for (SecKillByJuInfoDTO secKillByJuInfoDTO : juCommoditySearchDataDTOList) {
            SecKillCommodityIndexDTO secKillCommodityIndexDTO = SecKillCommodityIndexDTO.builder()
                    .commodityPrice(secKillByJuInfoDTO.getOriginalPrice())
                    .commodityTitle(secKillByJuInfoDTO.getDtitle())
                    .goodsId(Long.valueOf(secKillByJuInfoDTO.getGoodsId()))
                    .imgUrl(secKillByJuInfoDTO.getMainPic())
                    .build();


            Long goodsId = secKillCommodityIndexDTO.getGoodsId();
            //查询原价
            Double reservePrice = commodityPriceInfoService.getReservePrice(goodsId);
            if (null == reservePrice) {
                log.error("首页聚划算获取原价为空,商品id为:{}", goodsId);
                continue;
            }

            //查询优惠券信息-计算券后价
            CommodityCommissionAndCoupon commissionInfo = commodityPriceInfoService.getCommodityCommissionAndCoupon(goodsId,userId);
            if (null != commissionInfo) {

                //剔除佣金为0的值
                Double commissionPrice = CommodityCalUtils.calCommission(commissionInfo, reservePrice);
                if (isZero(commissionPrice)) {
                    continue;
                }
                secKillCommodityIndexDTO.setCommodityPrice(CommodityCalUtils.calCommodityPriceWithSubtractingCoupon(reservePrice, commissionInfo.getCouponPrice()));
            }

            resultList.add(secKillCommodityIndexDTO);
        }

        //首页仅展示三条
        resultList = resultList.stream().limit(3).collect(Collectors.toList());

        return resultList;

    }


    /**
     * 转换聚划算商品信息
     *
     * @param juCommoditySearchDataDTO 聚划算商品data信息DTO
     * @return 限时秒杀-首页展示信息列表
     */
    private List<InexpensiveCommodityDTO> convertJuCommodityInfoList(JuCommoditySearchDataDTO juCommoditySearchDataDTO,
                                                                     Long userId) {

        List<InexpensiveCommodityDTO> resultList = new ArrayList<>();

        //三方获取到的数据list 遍历进行值的转换
        List<SecKillByJuInfoDTO> juCommoditySearchDataDTOList = juCommoditySearchDataDTO.getList();

        for (SecKillByJuInfoDTO secKillByJuInfoDTO : juCommoditySearchDataDTOList) {
            //黑名单数据剔除
            if (blackCommodityService.isBlackCommodity(Long.valueOf(secKillByJuInfoDTO.getGoodsId()))) {
                continue;
            }
            InexpensiveCommodityDTO inexpensiveCommodityDTO = InexpensiveCommodityDTO.builder()
                    .commodityTitle(secKillByJuInfoDTO.getDtitle())
                    .goodsId(Long.valueOf(secKillByJuInfoDTO.getGoodsId()))
                    .imgUrl(secKillByJuInfoDTO.getMainPic())
                    .shopType(secKillByJuInfoDTO.getShopType())
                    .soldNum(null == secKillByJuInfoDTO.getMonthSales() ? "0" : secKillByJuInfoDTO.getMonthSales().toString())
                    .build();

            //获取销售进度条
            Double soldPercent = (secKillByJuInfoDTO.getDiscounts() * 100D);
            inexpensiveCommodityDTO.setSoldPercent(soldPercent);

            //获取商品原价
            Long goodsId = inexpensiveCommodityDTO.getGoodsId();
            Double reservePrice = commodityPriceInfoService.getReservePrice(goodsId);
            if (null == reservePrice) {
                log.error("聚划算列表获取商品原价为空,商品id为:{}", goodsId);
                continue;
            }
            inexpensiveCommodityDTO.setReservePrice(reservePrice);

            //佣金以及优惠券信息
            CommodityCommissionAndCoupon commissionInfoDTO = commodityPriceInfoService.getCommodityCommissionAndCoupon(goodsId, userId);
            if (null != commissionInfoDTO) {

                Double commissionPrice = CommodityCalUtils.calCommission(commissionInfoDTO, reservePrice);
                Double couponPrice = commissionInfoDTO.getCouponPrice();
                //剔除0元佣金数据
                if (isZero(commissionPrice) || isZero(couponPrice)) {
                    continue;
                }

                inexpensiveCommodityDTO.setCouponPrice(couponPrice);
                inexpensiveCommodityDTO.setCommissionPrice(commissionPrice);
                inexpensiveCommodityDTO.setCommodityPrice(CommodityCalUtils
                        .calPriceWithSubtractingCouponAndCommission(reservePrice, commissionInfoDTO.getCouponPrice(), commissionPrice));
            }

            //封装金币信息
            getGoldInfo(inexpensiveCommodityDTO, userId);

            resultList.add(inexpensiveCommodityDTO);
        }




        return resultList;

    }

    private SecKillCommodityIndexDTO convert(SecKillCommodityDTO secKillCommodityDTO) {
        SecKillCommodityIndexDTO secKillCommodityIndexDTO = SecKillCommodityIndexDTO.builder()
                .reservePrice(secKillCommodityDTO.getReservePrice())
                .commodityPrice(secKillCommodityDTO.getCommodityPrice())
                .commodityTitle(secKillCommodityDTO.getCommodityTitle())
                .goodsId(secKillCommodityDTO.getGoodsId())
                .imgUrl(secKillCommodityDTO.getImgUrl())
                .build();

        return secKillCommodityIndexDTO;
    }

    private SecKillCommodityIndexDTO convert(InexpensiveCommodityDTO inexpensiveCommodityDTO) {

        return SecKillCommodityIndexDTO.builder()
                .commodityPrice(inexpensiveCommodityDTO.getCommodityPrice())
                .commodityTitle(inexpensiveCommodityDTO.getCommodityTitle())
                .goodsId(inexpensiveCommodityDTO.getGoodsId())
                .imgUrl(inexpensiveCommodityDTO.getImgUrl())
                .build();
    }

    /**
     * 转换淘抢购商品信息成首页展示信息
     *
     * @param secKillByTaoInfoList 淘抢购商品信息list
     * @return 限时秒杀首页展示信息list
     */
    private List<SecKillCommodityIndexDTO> convertSecKillIndexCommodityInfoList(List<TbkJuTqgGetResponse.Results> secKillByTaoInfoList,
                                                                                Long userId) {

        //淘抢购信息列表截取
        List<SecKillCommodityIndexDTO> resultList = new ArrayList<>();
        for (TbkJuTqgGetResponse.Results secKillByTaoInfoDTO : secKillByTaoInfoList) {
            SecKillCommodityIndexDTO secKillCommodityIndexDTO = SecKillCommodityIndexDTO.builder()
                    .reservePrice(Double.valueOf(secKillByTaoInfoDTO.getZkFinalPrice()))
                    .commodityPrice(Double.valueOf(secKillByTaoInfoDTO.getZkFinalPrice()))
                    .commodityTitle(secKillByTaoInfoDTO.getTitle())
                    .goodsId(secKillByTaoInfoDTO.getNumIid())
                    .imgUrl(secKillByTaoInfoDTO.getPicUrl()).build();

            Long goodsId = secKillCommodityIndexDTO.getGoodsId();

            Double reservePrice = commodityPriceInfoService.getReservePrice(goodsId);
            if (null == reservePrice) {
                continue;
            }


            //请求高佣接口
            CommodityCommissionAndCoupon commissionInfoDTO =
                    commodityPriceInfoService.getCommodityCommissionAndCoupon(goodsId, userId);
            if (null != commissionInfoDTO) {
                Double commissionPrice = CommodityCalUtils.calCommission(commissionInfoDTO, reservePrice);

                if (isZero(commissionPrice)) {
                    continue;
                }
                secKillCommodityIndexDTO.setCommodityPrice(CommodityCalUtils.calPriceWithSubtractingCouponAndCommission(
                        reservePrice, commissionInfoDTO.getCouponPrice(), commissionPrice));
            }

            resultList.add(secKillCommodityIndexDTO);
        }

        //首页仅展示三条
        resultList = resultList.stream().limit(3).collect(Collectors.toList());


        return resultList;

    }

    /**
     * 转换淘抢购商品信息成详情信息
     *
     * @param secKillByTaoInfoList 淘抢购商品信息list
     * @return 限时秒杀首页展示信息list
     */
    private List<SecKillCommodityDTO> convertSecKillCommodityInfoList(
            List<TbkJuTqgGetResponse.Results> secKillByTaoInfoList, QuerySecKillCommodityParam secKillCommodityParam) {

        Long userId = secKillCommodityParam.getUserId();
        //淘抢购信息列表截取
        List<SecKillCommodityDTO> resultList = new ArrayList<>();
        for (TbkJuTqgGetResponse.Results taoInfo : secKillByTaoInfoList) {

            if (blackCommodityService.isBlackCommodity(taoInfo.getNumIid())) {
                continue;
            }
            SecKillCommodityDTO secKillCommodityDTO = SecKillCommodityDTO.builder()
                    .commodityTitle(taoInfo.getTitle())
                    .goodsId(taoInfo.getNumIid())
                    .imgUrl(taoInfo.getPicUrl())
                    .totalAmount(taoInfo.getTotalAmount())
                    .soldNum(taoInfo.getSoldNum())
                    .startTime(taoInfo.getStartTime())
                    .endTime(taoInfo.getEndTime())
                    //已售数量小于总数量,则商品售卖状态为1-可售,否则为0-不可售(已抢光)
                    .soldStatus(taoInfo.getSoldNum() < taoInfo.getTotalAmount() ? 1 : 0)
                    .build();

            //设置销售进度条
            secKillCommodityDTO.setSoldPercent(getSoldPercent(taoInfo.getSoldNum(), taoInfo.getTotalAmount()));

            //获取商品原价
            Long goodsId = secKillCommodityDTO.getGoodsId();
            Double reservePrice = commodityPriceInfoService.getReservePrice(goodsId);
            if (null == reservePrice) {
                log.error("淘抢购列表获取商品原价为空,商品id为:{}", goodsId);
                continue;
            }
            secKillCommodityDTO.setReservePrice(reservePrice);

            //请求高佣接口
            CommodityCommissionAndCoupon commissionInfoDTO =
                    commodityPriceInfoService.getCommodityCommissionAndCoupon(goodsId, secKillCommodityParam.getUserId());
            if (null != commissionInfoDTO) {
                Double commissionPrice = CommodityCalUtils.calCommission(commissionInfoDTO, reservePrice);
                Double couponPrice = commissionInfoDTO.getCouponPrice();
                //剔除0元佣金、优惠券的数据
                if (isZero(commissionPrice) || isZero(couponPrice)) {
                    continue;
                }


                secKillCommodityDTO.setCommissionPrice(commissionPrice);
                secKillCommodityDTO.setCouponPrice(couponPrice);

                //商品最终价格
                secKillCommodityDTO.setCommodityPrice(CommodityCalUtils
                        .calPriceWithSubtractingCouponAndCommission(reservePrice, commissionInfoDTO.getCouponPrice(), commissionPrice));

            }
            //获取金币信息以及提醒状态--此方法封装商品的活跃信息,在redis缓存获取到后也需要再查询
            getOtherInfo(secKillCommodityDTO, userId);

            resultList.add(secKillCommodityDTO);
        }


        return resultList;

    }

    /**
     * 计算倒计时时间
     *
     * @param secKillTimeEnum
     * @return
     */
    private Long calCountDownTime(SecKillTimeEnum secKillTimeEnum) {
        LocalDateTime nowDateTime = LocalDateTime.now();
        LocalDateTime endTimeDateTime;

        //若是秒杀时间区间是22点时间范围,则结束时间为第二天10点
        if (SecKillTimeEnum.SEC_KILL_22 == secKillTimeEnum) {
            endTimeDateTime = LocalDate.now().plusDays(1).atTime(LocalTime.of(secKillTimeEnum.getEndTimeHour(),
                    secKillTimeEnum.getEndTimeMin(), secKillTimeEnum.getEndTimeSec()));
            //计算相差的毫秒数
            return Duration.between(nowDateTime, endTimeDateTime).toMillis();
        }
        endTimeDateTime = LocalDate.now().atTime(LocalTime.of(secKillTimeEnum.getEndTimeHour(),
                secKillTimeEnum.getEndTimeMin(), secKillTimeEnum.getEndTimeSec()));

        return Duration.between(nowDateTime, endTimeDateTime).toMillis();
    }


    /**
     * 计算倒计时时间
     * @param secKillCommodityParam 淘抢购商品查询参数
     * @return 倒计时时间
     */
    private Long calCountDownTimeByType(QuerySecKillCommodityParam secKillCommodityParam) {
        //格式化时间
        DateTimeFormatter df = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
        LocalDateTime startTime = LocalDateTime.parse(secKillCommodityParam.getStartTime(), df);
        LocalDateTime endTime = LocalDateTime.parse(secKillCommodityParam.getEndTime(), df);
        Long countDownTime = 0L;

        Integer type = secKillCommodityParam.getType();
        LocalDateTime nowTime = LocalDateTime.now();
        if (1 == type) {
            return countDownTime;
        }

        if (2 == type) {
            return Duration.between(nowTime, endTime).toMillis();
        }

        if (3 == type) {
            return Duration.between(nowTime, startTime).toMillis();
        }

        return countDownTime;

    }


    /**
     * 根据当前时间枚举类排序时间列表
     * @param nowTimeEnum 当前时间所在时间枚举类
     * @return 排序后的时间列表
     */
    public List<SecCommodityTimeDTO> sortListByEnum(SecKillTimeEnum nowTimeEnum) {
        List<SecCommodityTimeDTO> secKillTimeList = new ArrayList<>();
        SecKillTimeEnum.buildEnumRelation();

        SecCommodityTimeDTO preByPreTimeDTO = assembleSecKillTimeByPre(nowTimeEnum, nowTimeEnum.getPreEnum().getPreEnum());
        secKillTimeList.add(preByPreTimeDTO);

        SecCommodityTimeDTO preTimeDTO = assembleSecKillTimeByPre(nowTimeEnum, nowTimeEnum.getPreEnum());
        secKillTimeList.add(preTimeDTO);

        SecCommodityTimeDTO nowTimeDTO = assembleSecKillTimeByNow(nowTimeEnum);
        secKillTimeList.add(nowTimeDTO);


        SecCommodityTimeDTO afterTimeDTO = assembleSecKillTimeByAfter(nowTimeEnum, nowTimeEnum.getNextEnum());
        secKillTimeList.add(afterTimeDTO);

        SecCommodityTimeDTO afterByAfterTimeDTO = assembleSecKillTimeByAfter(nowTimeEnum, nowTimeEnum.getNextEnum().getNextEnum());
        secKillTimeList.add(afterByAfterTimeDTO);

        return secKillTimeList;
    }

    private SecCommodityTimeDTO assembleSecKillTimeByPre(SecKillTimeEnum nowTimeEnum, SecKillTimeEnum preTimeEnum) {
        LocalDate nowDay = LocalDate.now();
        LocalDate preDay = LocalDate.now().plusDays(-1L);


        LocalDate startDay;
        LocalDate endTimeDay;
        LocalDateTime startDateTime;
        LocalDateTime endDateTime;


        if (preTimeEnum.getStartTimeHour() < nowTimeEnum.getStartTimeHour()) {
            //组装开始时间
            startDay = nowDay;
            startDateTime = SecKillTimeEnum.getDateStartTimeByEnum(startDay, preTimeEnum);

            //组装结束时间
            if (preTimeEnum == SecKillTimeEnum.SEC_KILL_22) {
                endTimeDay = startDay.plusDays(1L);
            } else {
                endTimeDay = startDay;
            }
            endDateTime = SecKillTimeEnum.getDateEndTimeByEnum(endTimeDay, preTimeEnum);
        } else {

            //组装开始时间
            startDay = preDay;
            startDateTime = SecKillTimeEnum.getDateStartTimeByEnum(startDay, preTimeEnum);

            //组装结束时间
            if (preTimeEnum == SecKillTimeEnum.SEC_KILL_22) {
                endTimeDay = startDay.plusDays(1L);
            } else {
                endTimeDay = startDay;
            }
            endDateTime = SecKillTimeEnum.getDateEndTimeByEnum(endTimeDay, preTimeEnum);

        }

        DateTimeFormatter dataFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
        DateTimeFormatter timeFormatter = DateTimeFormatter.ofPattern("HH:mm");
        SecCommodityTimeDTO secCommodityTimeDTO = SecCommodityTimeDTO.builder()
                .startTime(dataFormatter.format(startDateTime))
                .endTime(dataFormatter.format(endDateTime))
                .showContent("已开抢")
                .type(1)
                .showTime(timeFormatter.format(LocalTime.of(startDateTime.getHour(), startDateTime.getMinute(), startDateTime.getSecond())))
                .build();


        return secCommodityTimeDTO;
    }

    private SecCommodityTimeDTO assembleSecKillTimeByAfter(SecKillTimeEnum nowTimeEnum, SecKillTimeEnum afterTimeEnum) {
        LocalDate nowDay = LocalDate.now();

        LocalDate startDay;
        LocalDate endTimeDay;
        LocalDateTime startDateTime;
        LocalDateTime endDateTime;


        if (afterTimeEnum.getStartTimeHour() < nowTimeEnum.getStartTimeHour()) {
            //组装开始时间
            startDay = nowDay.plusDays(1L);
            startDateTime = SecKillTimeEnum.getDateStartTimeByEnum(startDay, afterTimeEnum);

            //组装结束时间
            if (afterTimeEnum == SecKillTimeEnum.SEC_KILL_22) {
                endTimeDay = startDay.plusDays(1L);
            } else {
                endTimeDay = startDay;
            }
            endDateTime = SecKillTimeEnum.getDateEndTimeByEnum(endTimeDay, afterTimeEnum);
        } else {

            //组装开始时间
            startDay = nowDay;
            startDateTime = SecKillTimeEnum.getDateStartTimeByEnum(startDay, afterTimeEnum);

            //组装结束时间
            if (afterTimeEnum == SecKillTimeEnum.SEC_KILL_22) {
                endTimeDay = startDay.plusDays(1L);
            } else {
                endTimeDay = startDay;
            }
            endDateTime = SecKillTimeEnum.getDateEndTimeByEnum(endTimeDay, afterTimeEnum);

        }

        DateTimeFormatter dataFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
        DateTimeFormatter timeFormatter = DateTimeFormatter.ofPattern("HH:mm");
        SecCommodityTimeDTO secCommodityTimeDTO = SecCommodityTimeDTO.builder()
                .startTime(dataFormatter.format(startDateTime))
                .endTime(dataFormatter.format(endDateTime))
                .showContent("即将开抢")
                .type(3)
                .showTime(timeFormatter.format(LocalTime.of(startDateTime.getHour(), startDateTime.getMinute(), startDateTime.getSecond())))
                .build();


        return secCommodityTimeDTO;
    }

    private SecCommodityTimeDTO assembleSecKillTimeByNow(SecKillTimeEnum nowTimeEnum) {

        LocalDate startDay = LocalDate.now();
        LocalDate endTimeDay;

        //组装开始时间
        LocalDateTime startDateTime = SecKillTimeEnum.getDateStartTimeByEnum(startDay, nowTimeEnum);

        //组装结束时间
        if (nowTimeEnum == SecKillTimeEnum.SEC_KILL_22) {
            endTimeDay = startDay.plusDays(1L);
        } else {
            endTimeDay = startDay;
        }
        LocalDateTime endDateTime = SecKillTimeEnum.getDateEndTimeByEnum(endTimeDay, nowTimeEnum);

        DateTimeFormatter dataFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
        DateTimeFormatter timeFormatter = DateTimeFormatter.ofPattern("HH:mm");
        SecCommodityTimeDTO secCommodityTimeDTO = SecCommodityTimeDTO.builder()
                .startTime(dataFormatter.format(startDateTime))
                .endTime(dataFormatter.format(endDateTime))
                .showContent("抢购进行中")
                .type(2)
                .showTime(timeFormatter.format(LocalTime.of(startDateTime.getHour(), startDateTime.getMinute(), startDateTime.getSecond())))
                .build();


        return secCommodityTimeDTO;
    }

    /**
     * 生成淘抢购商品请求参数
     * @param secKillCommodityParam 请求参数
     * @param secKillTimeEnum 秒杀时间枚举
     * @return 请求参数类
     */
    private TbkJuTqgGetRequest assembleTqgParam(QuerySecKillCommodityParam secKillCommodityParam, SecKillTimeEnum secKillTimeEnum) {

        TbkJuTqgGetRequest req = new TbkJuTqgGetRequest();
        req.setAdzoneId(Long.valueOf(commodityInfoProperties.getTbCommodityZoneId()));
        req.setFields("click_url,pic_url,reserve_price,zk_final_price,total_amount,sold_num,title,category_name,start_time,end_time");
        req.setPageNo(Long.valueOf(secKillCommodityParam.getPageNum()));
        req.setPageSize(Long.valueOf(secKillCommodityParam.getPageSize()));
        assembleTimeByEnum(req, secKillTimeEnum);

        return req;

    }

    /**
     * 组装淘抢购商品前端展示的请求参数--前端展示
     * @param secKillTimeEnum 秒杀时间枚举
     * @return 请求参数类
     */
    private TbkJuTqgGetRequest assembleTqgParamForIndex(SecKillTimeEnum secKillTimeEnum) {


        TbkJuTqgGetRequest req = new TbkJuTqgGetRequest();
        req.setAdzoneId(Long.valueOf(commodityInfoProperties.getTbCommodityZoneId()));
        req.setFields("click_url,pic_url,reserve_price,zk_final_price,total_amount,sold_num,title,category_name,start_time,end_time");
        req.setPageNo(1L);
        req.setPageSize(6L);
        assembleTimeByEnum(req, secKillTimeEnum);

        return req;

    }

    /**
     * 生成聚划算商品redis key--查询条件都拼接到key中,并加以前缀防止参数值相同,造成误差的情况
     * @param inexpensiveCommodityParam 查询参数
     * @return redis key
     */
    private KeyGenerator createJuCommodityRedisKey(InexpensiveCommodityParam inexpensiveCommodityParam) {

        KeyGenerator redisKey = CommodityRedisKeyConstants.COMMODITY_SEC_KILL_JU.copy();
        if (null != inexpensiveCommodityParam.getSortType()) {
            redisKey.appendKey("type_" + inexpensiveCommodityParam.getSortType());
        }

        if (null != inexpensiveCommodityParam.getPriceLowerLimit()) {
            redisKey.appendKey("lower_" + inexpensiveCommodityParam.getPriceLowerLimit());
        }

        if (null != inexpensiveCommodityParam.getPriceUpperLimit()) {
            redisKey.appendKey("upper_" + inexpensiveCommodityParam.getPriceUpperLimit());
        }

        if (StringUtils.isNotBlank(inexpensiveCommodityParam.getPageId())) {
            redisKey.appendKey("page_" + inexpensiveCommodityParam.getPageId());
        }
        return redisKey;
    }

    /**
     * 判断是否为0元佣金、优惠券
     * @param price 佣金面值、优惠券面值
     * @return 是0元则返回true,其他值或者null则返回false
     */
    private boolean isZero(Double price) {
        return new Double("0").equals(price);
    }


    private Double getSoldPercent(Long soldNum, Long totalNum) {
        if (null == soldNum || null == totalNum) {
            return 0D;
        }

        BigDecimal soldDecimal = new BigDecimal(soldNum.toString());
        BigDecimal totalDecimal = new BigDecimal(totalNum.toString());

        BigDecimal result = soldDecimal.divide(totalDecimal, 3, BigDecimal.ROUND_HALF_UP);
        //大于0小于0.01则展示1%
        if (-1 == result.compareTo(new BigDecimal("0.01")) && 0 != result.compareTo(new BigDecimal("0"))) {

            return 1D;
        }

        return result.multiply(new BigDecimal("100")).doubleValue();

    }

    public static void main(String[] args) {
        SecKillCommodityServiceImpl secKillCommodityService = new SecKillCommodityServiceImpl();
        System.out.println(secKillCommodityService.getSoldPercent(0L, 1000L));
    }



}




