package com.bxm.fossicker.commodity.service.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.fossicker.commodity.config.VipZeroCommodityProperties;
import com.bxm.fossicker.commodity.model.constant.CommodityPoolId;
import com.bxm.fossicker.commodity.model.constant.CommodityRedisKey;
import com.bxm.fossicker.commodity.model.param.ChoicenessListParam;
import com.bxm.fossicker.commodity.model.param.PoolCommodityListParam;
import com.bxm.fossicker.commodity.model.param.UserIdParam;
import com.bxm.fossicker.commodity.model.vo.*;
import com.bxm.fossicker.commodity.service.CommodityInfoService;
import com.bxm.fossicker.commodity.service.VipCommodityService;
import com.bxm.fossicker.commodity.service.commodity.list.CommodityListServiceProxy;
import com.bxm.fossicker.user.facade.VirtualUserFacadeService;
import com.bxm.fossicker.user.model.entity.VirtualUserBean;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.RandomUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @Author: Gonzo
 * @date 2020-02-29 21:46
 * @since 1.6.0
 */
@Service
public class VipCommodityServiceImpl implements VipCommodityService {

    @Resource
    private VirtualUserFacadeService virtualUserFacadeService;

    @Resource
    private RedisStringAdapter redisStringAdapter;

    @Resource
    private CommodityInfoService commodityInfoService;

    @Resource
    private VipZeroCommodityProperties vipZeroCommodityProperties;

    @Resource
    private CommodityListServiceProxy commodityListServiceProxy;

    private static final String CONGRATULATIONS = "恭喜";

    private static final String ZERO_PURCHASE = "0元购得";

    @Override
    public List<VipZeroCommodityBriefListVO> zeroBriefList(UserIdParam param, boolean isPublicRequest) {

        List<VipZeroCommodityListVO> vipZeroCommodityListVOS = zeroList(param, isPublicRequest);

        int end = vipZeroCommodityListVOS.size() < vipZeroCommodityProperties.getBriefNum()
                ? vipZeroCommodityListVOS.size() : vipZeroCommodityProperties.getBriefNum();
        if (end > 0) {
            vipZeroCommodityListVOS = vipZeroCommodityListVOS.subList(0, end);

            return vipZeroCommodityListVOS.stream()
                    .filter(Objects::nonNull)
                    .map(this::convert)
                    .collect(Collectors.toList());
        }

        return Lists.newArrayList();
    }

    /**
     * 转换对象
     *
     * @param item item
     * @return VipZeroCommodityBriefListVO
     */
    private VipZeroCommodityBriefListVO convert(VipZeroCommodityListVO item) {

        return VipZeroCommodityBriefListVO.builder()
                .commodityTitle(item.getCommodityTitle())
                .goodsId(item.getGoodsId())
                .reservePrice(item.getReservePrice())
                .commodityPrice(item.getCommodityPrice())
                .imgUrl(item.getImgUrl())
                .build();
    }


    @Override
    public List<VipZeroCommodityListVO> zeroList(UserIdParam param, boolean isPublicRequest) {

        PoolCommodityListParam poolParam = new PoolCommodityListParam();
        poolParam.setPoolId(CommodityPoolId.VIP_ZERO);
        poolParam.setUserId(param.getUserId());
        poolParam.setPublicRequest(isPublicRequest);

        List<PoolCommodityListItem> poolCommodityListItems = commodityListServiceProxy.getCommodityList(poolParam);

        return poolCommodityListItems.stream()
                .filter(Objects::nonNull)
                .map(this::convert)
                .collect(Collectors.toList());
    }

    /**
     * 转换对象
     *
     * @param item item
     * @return VipZeroCommodityBriefListVO
     */
    private VipZeroCommodityListVO convert(PoolCommodityListItem item) {

        return VipZeroCommodityListVO.builder()
                .commodityTitle(item.getCommodityTitle())
                .goodsId(item.getGoodsId())
                .reservePrice(item.getReservePrice())
                .commodityPrice(item.getCommodityPrice())
                .imgUrl(item.getImgUrl())
                .shopName(item.getShopName())
                .shopType(item.getShopType())
                .monthSales(item.getMonthSales())
                .build();
    }

    @Override
    public List<VipZeroPurchaseInfo> vipZeroPurchaseInfos() {

        // 缓存获取
        String infosStr = redisStringAdapter.get(CommodityRedisKey.VIP_ZERO_COMMODITY_PURCHASE_INFOS, String.class);

        List<VipZeroPurchaseInfo> result;
        if (StringUtils.isBlank(infosStr)
                || CollectionUtils.isEmpty(result = JSON.parseArray(infosStr, VipZeroPurchaseInfo.class))) {
            // 不存在则生成购买数据

            // 虚拟用户信息
            List<VirtualUserBean> virtualUsers = virtualUserFacadeService.listVirtualUser(vipZeroCommodityProperties.getPurchaseInfosNum());
            if (!CollectionUtils.isEmpty(virtualUsers)) {
                // vip 0元购商品列表信息
                List<VipZeroCommodityListVO> vipZeroCommodities = zeroList(new UserIdParam(), true);

                result = virtualUsers.stream().map(p -> {
                    VipZeroCommodityListVO vipZeroCommodityBriefListVO = vipZeroCommodities.get(RandomUtils.nextInt(0,
                            vipZeroCommodities.size()));
                    return VipZeroPurchaseInfo.builder()
                            // 恭喜+用户昵称+0元购得+商品名称
                            .content(StringUtils.join(CONGRATULATIONS, p.getNickname(), ZERO_PURCHASE,
                                    vipZeroCommodityBriefListVO.getCommodityTitle()))
                            .headImg(p.getHeadImg())
                            .build();
                }).collect(Collectors.toList());

                // 添加缓存 以及失效时间
                redisStringAdapter.set(CommodityRedisKey.VIP_ZERO_COMMODITY_PURCHASE_INFOS, JSON.toJSONString(result),
                        vipZeroCommodityProperties.getPurchaseCacheExpireSecond());
            } else {
                // 如果获取不到虚拟用户信息，返回空列表
                result = Lists.newArrayList();
            }
        }

        return result;
    }

    @Override
    public List<VipChoicenessCommodityListVO> choicenessList(ChoicenessListParam param) {
        PoolCommodityListParam poolParam = new PoolCommodityListParam();
        poolParam.setPoolId(param.getPoolId());
        poolParam.setUserId(param.getUserId());
        poolParam.setQueryGoldInfo(true);

        List<PoolCommodityListItem> poolCommodityListItems = commodityListServiceProxy.getCommodityList(poolParam);

        return poolCommodityListItems.stream()
                .filter(Objects::nonNull)
                .map(this::convertToVipChoicenessCommodity)
                .collect(Collectors.toList());
    }

    private VipChoicenessCommodityListVO convertToVipChoicenessCommodity(PoolCommodityListItem item) {
        VipChoicenessCommodityListVO listVO = new VipChoicenessCommodityListVO();
        BeanUtils.copyProperties(item, listVO);

        // 处理销量字符串
        listVO.setMonthSalesStr(com.bxm.newidea.component.tools.StringUtils.salesNumTenThousandFormat(listVO.getMonthSales()));

        return listVO;
    }

}
