package com.bxm.fossicker.commodity.service.strategy;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.bxm.fossicker.commodity.common.constants.CommodityRedisKeyConstants;
import com.bxm.fossicker.commodity.common.enums.CommoditySortTypeEnum;
import com.bxm.fossicker.commodity.model.dto.CommodityGoldInfoDTO;
import com.bxm.fossicker.commodity.model.dto.CommodityPageInfo;
import com.bxm.fossicker.commodity.model.dto.CommoditySearchDTO;
import com.bxm.fossicker.commodity.model.param.CommoditySearchParam;
import com.bxm.fossicker.commodity.service.BlackCommodityService;
import com.bxm.fossicker.commodity.service.CommodityGoldService;
import com.bxm.fossicker.commodity.service.CommodityPoolService;
import com.bxm.fossicker.commodity.service.CommodityPriceInfoService;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.StringUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;

import java.util.List;


@Slf4j
public abstract class AbstractCommodityQueryService implements CommodityQueryService {

    @Autowired
    private CommodityGoldService commodityGoldService;

    @Autowired
    private CommodityPriceInfoService commodityPriceInfoService;

    @Autowired
    private CommodityLabelService commodityLabelService;

    @Autowired
    private RedisStringAdapter redisStringAdapter;

    @Autowired
    private BlackCommodityService blackCommodityService;

    @Autowired
    private CommodityPoolService commodityPoolService;

    /**
     * 组装金币信息
     *
     * @param commoditySearchDTO 商品信息
     * @param userId             用户id
     */
    public void assembleGoldInfo(CommoditySearchDTO commoditySearchDTO, Long userId) {
        CommodityGoldInfoDTO goldInfo = commodityGoldService.getGoldInfoByCommodityId(commoditySearchDTO.getGoodsId(), userId);
        if (null != goldInfo) {
            commoditySearchDTO.setGoldNum(goldInfo.getBrowseGold());
            commoditySearchDTO.setGoldStatus(goldInfo.getDrawFlag());
        }

        //设置新人专享角标
        commoditySearchDTO.setNewbieRight(commodityPriceInfoService.isNewbieRight(userId, commoditySearchDTO.getGoodsId()));
        commoditySearchDTO.setLabelUrl(commodityLabelService.goodsLabel(commoditySearchDTO.getNewbieRight(),
                commoditySearchDTO.getHandsel(), commoditySearchDTO.getGoodsId()));
    }

    /**
     * 从redis中获取关键字搜索信息
     * @param commoditySearchParam 查询参数
     * @param commodityRedisKey redis key
     * @return 关键字搜索的商品信息
     */
    public CommodityPageInfo<CommoditySearchDTO> getCommodityFromRedis(KeyGenerator commodityRedisKey,
                                                                       CommoditySearchParam commoditySearchParam) {

        CommodityPageInfo<CommoditySearchDTO> resultFromRedis = new CommodityPageInfo<>();
        String commodityByRedis = redisStringAdapter.getString(commodityRedisKey);

        if (StringUtils.isBlank(commodityByRedis)) {
            return null;
        }
        List<CommoditySearchDTO> commodityList = JSONArray.parseArray(commodityByRedis, CommoditySearchDTO.class);
        //剔除黑名单商品
        commodityList.removeIf(commoditySearchDTO -> blackCommodityService.isBlackCommodity(commoditySearchDTO.getGoodsId()));
        commodityList.forEach(commoditySearchDTO -> assembleGoldInfo(commoditySearchDTO, commoditySearchParam.getUserId()));
        resultFromRedis.setList(commodityList);
        resultFromRedis.setPageNum(commoditySearchParam.getPageNum());
        resultFromRedis.setPageSize(commoditySearchParam.getPageSize());
        //searchType设置
        resultFromRedis.setSearchType("1");
        if (!CollectionUtils.isEmpty(commodityList)) {
            resultFromRedis.setSearchType(commodityList.get(0).getSearchType());
        }
        return resultFromRedis;
    }

    /**
     * 创建商品查询redis key
     *
     * @param commoditySearchParam 商品查询参数
     * @param vipFlag vip标志位
     * @return redis key
     */
    public KeyGenerator createCommodityRedisKey(CommoditySearchParam commoditySearchParam, boolean vipFlag) {
        log.info("根据商品名搜索商品列表（createCommodityRedisKey）,查询参数为:{}", JSON.toJSONString(commoditySearchParam));
        KeyGenerator keyGenerator = CommodityRedisKeyConstants.COMMODITY_QUERY_BY_NAME.copy();
        keyGenerator.appendKey(commoditySearchParam.getCommodityName());
        keyGenerator.appendKey(commoditySearchParam.getPageNum());
        keyGenerator.appendKey("vip:" + vipFlag);
        if (StringUtils.isNotBlank(commoditySearchParam.getSortType())) {
            //根据code转换分类类型
            keyGenerator.appendKey("type:" + CommoditySortTypeEnum.getTypeByCode(commoditySearchParam.getSortType()));
        }

        if (null != commoditySearchParam.getPriceLowerLimit()) {

            keyGenerator.appendKey("price_lower:" + commoditySearchParam.getPriceLowerLimit());
        }

        if (null != commoditySearchParam.getPriceUpperLimit()) {
            keyGenerator.appendKey("price_upper:" + commoditySearchParam.getPriceUpperLimit());
        }

        if (null != commoditySearchParam.getCouponFlag()) {
            keyGenerator.appendKey("coupon_flag:" + commoditySearchParam.getCouponFlag());
        }

        return keyGenerator;
    }

    /**
     * 处理新人商品--若是新人商品且符合价格要求则设置佣金以及商品价格，若是新人商品但不符合价格要求则下架商品并剔除
     * 此操作需考虑商品佣金和商品价格的设置,避免被后续操作覆盖
     * @param newbieRightFlag 新人专享标志
     * @param discountPrice 券后价
     * @param goodsId 商品id
     * @param commoditySearchDTO 需设置的数据模型
     * @return true-后续不做任何处理, false后续需要剔除数据
     */
    public boolean processNewbieCommodity(boolean newbieRightFlag, Double discountPrice,
                                           Long goodsId, CommoditySearchDTO commoditySearchDTO) {
        //非新人商品不处理
        if (!newbieRightFlag) {
            return true;
        }

        //新人专享商品的特殊处理
        if (commodityPriceInfoService.isRuleNewbiePrice(discountPrice)) {
            //将商品佣金自动变为券后价
            commoditySearchDTO.setCommissionPrice(discountPrice);
            commoditySearchDTO.setCommodityPrice(0D);
            return true;

        } else {
            //商品下架且剔除该商品
            log.error("该新人商品:{}价格不符合商品价格要求,自动下架", goodsId);
            commodityPoolService.setNewbieInvalidStatus(goodsId);
            return false;
        }


    }


}
