package com.bxm.fossicker.commodity.service.strategy;

import com.alibaba.fastjson.JSON;
import com.bxm.fossicker.commodity.common.dto.CommodityCommissionAndCoupon;
import com.bxm.fossicker.commodity.common.dto.PlatformCommissionRateInfo;
import com.bxm.fossicker.commodity.common.enums.CommoditySortTypeEnum;
import com.bxm.fossicker.commodity.common.enums.LocalCommodityStatusEnum;
import com.bxm.fossicker.commodity.common.utils.CommodityCalUtils;
import com.bxm.fossicker.commodity.config.CommodityInfoProperties;
import com.bxm.fossicker.commodity.model.dto.CommodityInfoDTO;
import com.bxm.fossicker.commodity.model.dto.CommodityPageInfo;
import com.bxm.fossicker.commodity.model.dto.CommoditySearchDTO;
import com.bxm.fossicker.commodity.model.param.ClipboardQueryParam;
import com.bxm.fossicker.commodity.model.param.CommoditySearchParam;
import com.bxm.fossicker.commodity.model.param.GetCommodityDetailParam;
import com.bxm.fossicker.commodity.service.BlackCommodityService;
import com.bxm.fossicker.commodity.service.CommodityGoldService;
import com.bxm.fossicker.commodity.service.commodity.info.CommodityInfoSourceProxyService;
import com.bxm.fossicker.commodity.service.CommodityPriceInfoService;
import com.bxm.fossicker.commodity.service.externel.TbCommodityService;
import com.bxm.fossicker.user.facade.vip.VipFacadeService;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.StringUtils;
import com.taobao.api.request.TbkDgMaterialOptionalRequest;
import com.taobao.api.response.TbkDgMaterialOptionalResponse;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * 商品查询服务-淘宝
 * @author donghy
 */
@Service
@Slf4j
public class CommodityQueryByTbImpl extends AbstractCommodityQueryService {

    /**
     * 商品佣金服务
     */
    @Autowired
    private CommodityPriceInfoService commodityPriceInfoService;

    /**
     * 商品金币服务
     */
    @Autowired
    private CommodityGoldService commodityGoldService;

    /**
     * redis String服务
     */
    @Autowired
    private RedisStringAdapter redisStringAdapter;

    /**
     * 淘宝客商品服务
     */
    @Autowired
    private TbCommodityService tbCommodityService;

    /**
     * 商品信息配置服务
     */
    @Autowired
    private CommodityInfoProperties commodityInfoProperties;

    /**
     * 黑名单商品服务
     */
    @Autowired
    private BlackCommodityService blackCommodityService;

    /**
     * 本地商品数据服务
     */
    @Autowired
    CommodityInfoSourceProxyService commodityInfoSourceService;

    /**
     * vip服务
     */
    @Autowired
    private VipFacadeService vipFacadeService;



    @Override
    public CommodityPageInfo<CommoditySearchDTO> queryCommodityByName(CommoditySearchParam commoditySearchParam) {
        CommodityPageInfo<CommoditySearchDTO> resultPageInfo = new CommodityPageInfo<>();

        //搜索名称是http开头的只查询第一页
        String searchName = commoditySearchParam.getCommodityName();
        if (StringUtils.isNotBlank(searchName) && searchName.startsWith("http")) {
            resultPageInfo.setIsLastPage(true);
        }

        Long userId = commoditySearchParam.getUserId();
        //获取是否vip
        boolean vipFlag = vipFacadeService.isVip(userId);

        //从redis缓存中取
        KeyGenerator commodityRedisKey = createCommodityRedisKey(commoditySearchParam, vipFlag);
        CommodityPageInfo<CommoditySearchDTO> commodityFromRedis = getCommodityFromRedis(commodityRedisKey, commoditySearchParam);
        //缓存中有数据--组装动态信息后返回
        if (Objects.nonNull(commodityFromRedis) && !CollectionUtils.isEmpty(commodityFromRedis.getList())) {
            return commodityFromRedis;
        }
        //组装查询参数
        TbkDgMaterialOptionalRequest req = assembleQueryParam(commoditySearchParam);

        TbkDgMaterialOptionalResponse rsp = tbCommodityService.getCommodityListByName(req);

        if (null == rsp || !rsp.isSuccess() || CollectionUtils.isEmpty(rsp.getResultList())) {
            log.error("从淘宝接口获取到的商品列表信息为空");
            return resultPageInfo;
        }

        //商品列表-查询高佣接口
        //参数转换
        List<CommoditySearchDTO> commoditySearchList = convertResultList(rsp.getResultList(), userId, vipFlag);

        //入redis
        saveRedis(commodityRedisKey, commoditySearchList);

        //平台添加的商品金币信息
        resultPageInfo.setList(commoditySearchList);
        resultPageInfo.setPageNum(commoditySearchParam.getPageNum());
        resultPageInfo.setPageSize(commoditySearchParam.getPageSize());

        return resultPageInfo;
    }

    @Override
    public CommoditySearchDTO getClipboardCommodity(ClipboardQueryParam clipboardQueryParam) {
        return null;
    }


    /**
     * 存入redis
     * @param commodityRedisKey redis key
     * @param commodityList 存入redis的数据
     */
    private void saveRedis(KeyGenerator commodityRedisKey, List<CommoditySearchDTO> commodityList){

        redisStringAdapter.set(commodityRedisKey, JSON.toJSONString(commodityList), 3600L);

    }

    /**
     * 组装请求参数
     * @param commoditySearchParam 商品查询请求参数
     * @return 请求参数
     */
    private TbkDgMaterialOptionalRequest assembleQueryParam(CommoditySearchParam commoditySearchParam) {

        TbkDgMaterialOptionalRequest req = new TbkDgMaterialOptionalRequest();

        req.setPageSize(Long.valueOf(commoditySearchParam.getPageSize()));
        req.setPageNo(Long.valueOf(commoditySearchParam.getPageNum()));

        req.setAdzoneId(Long.valueOf(commodityInfoProperties.getTbCommodityZoneId()));
        req.setQ(commoditySearchParam.getCommodityName());
        if (StringUtils.isNotBlank(commoditySearchParam.getSortType())) {
            //根据code转换分类类型
            req.setSort(CommoditySortTypeEnum.getTypeByCode(commoditySearchParam.getSortType()));
        }

        if (null != commoditySearchParam.getPriceLowerLimit()) {
            req.setStartPrice(commoditySearchParam.getPriceLowerLimit());
        }

        if (null != commoditySearchParam.getPriceUpperLimit()) {
            req.setEndPrice(commoditySearchParam.getPriceUpperLimit());
        }

        if (null != commoditySearchParam.getCouponFlag() && 1 == commoditySearchParam.getCouponFlag()) {
            req.setHasCoupon(true);
        }


        return req;

    }

    /**
     * 转换结果list
     * @param tbkDataList 淘宝客商品数据list
     * @param userId 用户id
     * @param vipFlag vip标志位
     * @return 商品信息list
     */
    private List<CommoditySearchDTO> convertResultList(List<TbkDgMaterialOptionalResponse.MapData> tbkDataList, Long userId,
                                                       boolean vipFlag) {

        List<CommoditySearchDTO> resultList = new ArrayList<>();

        //遍历数据
        tbkDataList.forEach(tbkData->{

            Long goodsId = tbkData.getItemId();

            if (blackCommodityService.isBlackCommodity(goodsId)) {
                return;
            }
            float handsel = Float.parseFloat(tbkData.getPresaleDeposit()==null ? "0" : tbkData.getPresaleDeposit());
            CommoditySearchDTO commoditySearchDTO;
            // 封装参数
            GetCommodityDetailParam detailParam = new GetCommodityDetailParam();
            // 不查优惠券信息
            detailParam.setQueryCouponInfo(false);
            detailParam.setCommodityId(goodsId);
            //先从库中取
            CommodityInfoDTO localCommodity = commodityInfoSourceService.getLocalCommodity(detailParam);
            //库中能查询到商品且商品状态为商家则取库内数据
            if (null != localCommodity && LocalCommodityStatusEnum.Valid.getStatus() == localCommodity.getStatus()) {
                //从库中取,转换数据
                commoditySearchDTO = convertLocalCommodity(localCommodity, goodsId, userId, vipFlag);

            } else {
                commoditySearchDTO = convertTbCommodity(tbkData, goodsId, userId, vipFlag);
            }

            //佣金信息或者佣金为0都会做剔除
            if (null == commoditySearchDTO) {
                return;
            }
            commoditySearchDTO.setHandsel(handsel);


            //拼装金币信息
            assembleGoldInfo(commoditySearchDTO, userId);

            resultList.add(commoditySearchDTO);

        });

        return resultList;
    }


    /**
     * 判断是否为0元佣金
     * @param commissionPrice 佣金面值
     * @return 是0元则返回true,其他值或者null则返回false
     */
    private boolean isZeroCommission(Double commissionPrice) {
        return new Double("0").equals(commissionPrice);
    }

    /**
     * 转换本地商品
     *
     * @param localCommodity 本地商品
     * @param goodsId        商品id
     * @param userId         用户id
     * @param vipFlag vip标志 true-是vip
     * @return 商品数据
     */
    private CommoditySearchDTO convertLocalCommodity(CommodityInfoDTO localCommodity, Long goodsId, Long userId,
                                                     boolean vipFlag) {

        //取本地商品服务
        CommoditySearchDTO commoditySearchDTO = CommoditySearchDTO.builder()
                .reservePrice(localCommodity.getReservePrice())
                .commodityTitle(localCommodity.getCommodityTitle())
                .goodsId(localCommodity.getGoodsId())
                .imgUrl(localCommodity.getMainPic())
                .monthSales(null == localCommodity.getSalesNum() ? 0L : localCommodity.getSalesNum().longValue())
                .shopName(localCommodity.getShopName())
                .shopType(localCommodity.getShopType())
                .sourceType(1)
                .build();

        //查询高佣接口-设置优惠券和佣金
        PlatformCommissionRateInfo platformCommissionDTO =
                commodityPriceInfoService.getPlatformCommissionRateInfo(goodsId);

        //佣金信息为空,则不展示
        if (null == platformCommissionDTO) {
            log.error("查询到的佣金信息为空，该商品不做展示,商品id为:{}", goodsId);
            return null;
        }

        boolean specialCommissionFlag = platformCommissionDTO.isSpecialCommissionFlag();
        commoditySearchDTO.setNewbieRight(commodityPriceInfoService.isNewbieRight(userId, goodsId));
        Double platformCommissionPrice = localCommodity.getPlatformCommissionPrice();
        Double commissionPrice = CommodityCalUtils.calCommissionForPlatformCommissionWithVip(platformCommissionDTO,
                platformCommissionPrice, vipFlag);

        Double vipCommissionPrice = CommodityCalUtils.calVipCommission(platformCommissionPrice, platformCommissionDTO.getVipCommissionRate(), vipFlag, specialCommissionFlag, commissionPrice);

        //剔除佣金为0的数据
        if (isZeroCommission(commissionPrice)) {
            log.error("商品佣金为0，该商品不做展示,商品id为:{}", goodsId);
            return null;
        }
        commoditySearchDTO.setCouponPrice(localCommodity.getCouponPrice());
        commoditySearchDTO.setCommissionPrice(commissionPrice);
        //vip相关变动字段
        commoditySearchDTO.setVipCommissionPrice(vipCommissionPrice);
        commoditySearchDTO.setSpecialCommissionFlag(specialCommissionFlag);

        //价格为返后价
        commoditySearchDTO.setCommodityPrice(CommodityCalUtils.calPriceWithSubtractingCouponAndCommission(localCommodity.getReservePrice(),
                localCommodity.getCouponPrice(), commissionPrice));
        return commoditySearchDTO;
    }

    /**
     * 转换淘宝商品数据
     * @param tbkData 淘宝数据
     * @param goodsId 商品id
     * @param userId 用户id
     * @param vipFlag vip标志 true-是vip
     * @return 商品数据
     */
    private CommoditySearchDTO convertTbCommodity(TbkDgMaterialOptionalResponse.MapData tbkData, Long goodsId,
                                                  Long userId, boolean vipFlag){
        Double reservePrice = Double.valueOf(tbkData.getZkFinalPrice());
        CommoditySearchDTO commoditySearchDTO = CommoditySearchDTO.builder()
                .commodityPrice(Double.valueOf(tbkData.getZkFinalPrice()))
                .reservePrice(reservePrice)
                .commodityTitle(tbkData.getTitle())
                .goodsId(goodsId)
                .imgUrl(tbkData.getPictUrl())
                .monthSales(tbkData.getVolume())
                .shopName(tbkData.getShopTitle())
                .shopType(null == tbkData.getUserType() ? 0 : tbkData.getUserType().intValue())
                .sourceType(2)
                .build();

        //查询高佣接口-设置优惠券和佣金

        CommodityCommissionAndCoupon commissionInfoDTO =
                commodityPriceInfoService.getCommodityCommissionAndCoupon(goodsId, userId);

        //佣金信息为空,则不展示
        if (null == commissionInfoDTO) {
            log.error("查询到的佣金信息为空，该商品不做展示,商品id为:{}", goodsId);
            return null;
        }

        commoditySearchDTO.setNewbieRight(commodityPriceInfoService.isNewbieRight(userId, goodsId));
        Double commissionPrice = CommodityCalUtils.calCommissionWithVip(commissionInfoDTO, reservePrice, vipFlag);
        Double couponPrice = commissionInfoDTO.getCouponPrice();
        Double discountPrice = CommodityCalUtils.calDiscountPrice(commoditySearchDTO.getReservePrice(), commissionInfoDTO.getCouponPrice());

        //剔除佣金为0的数据
        if (isZeroCommission(commissionPrice)) {
            log.error("商品佣金为0，该商品不做展示,商品id为:{}", goodsId);
            return null;
        }
        commoditySearchDTO.setCouponPrice(couponPrice);
        commoditySearchDTO.setCommissionPrice(commissionPrice);
        commoditySearchDTO.setSpecialCommissionFlag(commissionInfoDTO.isSpecialCommissionFlag());
        commoditySearchDTO.setVipCommissionPrice(CommodityCalUtils.calVipCommission(discountPrice,
                commissionInfoDTO.getCommissionRateByThird(), commissionInfoDTO.getVipCommissionRate()));

        //价格为返后价
        commoditySearchDTO.setCommodityPrice(CommodityCalUtils.calPriceWithSubtractingCouponAndCommission(reservePrice,
                couponPrice, commissionPrice));

        return commoditySearchDTO;
    }



}
