package com.bxm.fossicker.commodity.service.strategy;


import com.alibaba.fastjson.JSON;
import com.bxm.fossicker.commodity.common.constants.SearchType;
import com.bxm.fossicker.commodity.common.utils.CommodityNumberUtils;
import com.bxm.fossicker.commodity.convert.CommodityDetailToSearchConvert;
import com.bxm.fossicker.commodity.model.dto.*;
import com.bxm.fossicker.commodity.model.param.ClipboardQueryParam;
import com.bxm.fossicker.commodity.model.param.CommodityDetailQueryParam;
import com.bxm.fossicker.commodity.model.param.CommoditySearchParam;
import com.bxm.fossicker.commodity.model.param.WySearchCommodityInfoParam;
import com.bxm.fossicker.commodity.service.CommodityInfoService;
import com.bxm.fossicker.commodity.service.externel.WyCommodityService;
import com.bxm.fossicker.user.facade.vip.VipFacadeService;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

@Service
@Slf4j
public class CommodityQueryByWyImpl extends AbstractCommodityQueryService {

    private final WyCommodityService wyCommodityService;

    private final VipFacadeService vipFacadeService;

    private final RedisStringAdapter redisStringAdapter;

    private final CommodityInfoService commodityInfoService;


    public CommodityQueryByWyImpl(WyCommodityService wyCommodityService, VipFacadeService vipFacadeService,
                                  RedisStringAdapter redisStringAdapter, CommodityInfoService commodityInfoService) {
        this.wyCommodityService = wyCommodityService;
        this.vipFacadeService = vipFacadeService;
        this.redisStringAdapter = redisStringAdapter;
        this.commodityInfoService = commodityInfoService;
    }

    @Override
    public CommodityPageInfo<CommoditySearchDTO> queryCommodityByName(CommoditySearchParam commoditySearchParam) {
        log.info("根据商品名搜索商品列表（queryCommodityByName）,查询参数为:{}", JSON.toJSONString(commoditySearchParam));
        CommodityPageInfo<CommoditySearchDTO> resultPage = new CommodityPageInfo<>();
        resultPage.setPageSize(commoditySearchParam.getPageSize());
        resultPage.setPageNum(commoditySearchParam.getPageNum());

        Long userId = commoditySearchParam.getUserId();
        //获取是否vip
        boolean vipFlag = vipFacadeService.isVip(userId);

        //从redis缓存中取
        KeyGenerator commodityRedisKey = createCommodityRedisKey(commoditySearchParam, vipFlag);
        if (log.isDebugEnabled()) {
            log.debug("【唯易】搜索商品，得到缓存key值:[{}]", commodityRedisKey.gen());
        }

        CommodityPageInfo<CommoditySearchDTO> commodityFromRedis = getCommodityFromRedis(commodityRedisKey, commoditySearchParam);
        //缓存中有数据--组装动态信息后返回
        if (Objects.nonNull(commodityFromRedis) && !CollectionUtils.isEmpty(commodityFromRedis.getList())) {
            return commodityFromRedis;
        }

        //请求维易超级搜索接口
        WySearchResultDTO wySearchResultDTO = wyCommodityService.searchCommodity(commoditySearchParam);

        if (Objects.isNull(wySearchResultDTO) || CollectionUtils.isEmpty(wySearchResultDTO.getCommodityList())) {
            log.error("搜索商品结果为空,查询参数:{}", JSON.toJSONString(commoditySearchParam));
            return resultPage;
        }

        //参数转换 -- 转换中错误信息以null返回,所以需要做一层过滤
        List<CommoditySearchDTO> resultList = wySearchResultDTO.getCommodityList().stream()
                .map(wyInfo -> getDetail(wyInfo, userId, wySearchResultDTO.getSearchType()))
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(resultList)) {
            redisStringAdapter.set(commodityRedisKey, JSON.toJSONString(resultList), 1800L);
        }

        resultPage.setList(resultList);
        resultPage.setSearchType(wySearchResultDTO.getSearchType());

        return resultPage;
    }

    @Override
    public CommoditySearchDTO getClipboardCommodity(ClipboardQueryParam clipboardQueryParam) {

        //组装参数
        CommoditySearchParam commoditySearchParam = new CommoditySearchParam();
        commoditySearchParam.setCommodityName(clipboardQueryParam.getCommodityName());
        commoditySearchParam.setPageSize(1);

        WySearchResultDTO wySearchResultDTO = wyCommodityService.searchCommodity(commoditySearchParam);
        //为null时表示未检索到商品
        if (Objects.isNull(wySearchResultDTO)) {
            if (log.isDebugEnabled()) {
                log.debug("根据剪切板，未查询到指定的商品信息,查询参数为:{}", JSON.toJSONString(commoditySearchParam));
            }
            return null;
        }

        // 判断是否有商品信息，如果不存在，则说明未检索到商品信息，但是需要提升给客户端
        if (CollectionUtils.isEmpty(wySearchResultDTO.getCommodityList())) {
            return CommoditySearchDTO.builder()
                    .remark(wySearchResultDTO.getMsg())
                    .searchType(wySearchResultDTO.getSearchType())
                    .build();

        }

        CommoditySearchDTO searchDTO = getDetail(wySearchResultDTO.getCommodityList().get(0),
                clipboardQueryParam.getUserId(),
                wySearchResultDTO.getSearchType());

        if (searchDTO != null && CommodityNumberUtils.isNullOrZero(searchDTO.getCommissionPrice())) {
            log.warn("商品佣金为0，该商品不做展示,查询参数：{}，商品信息：{}",
                    clipboardQueryParam,
                    searchDTO);
            return null;
        }

        return searchDTO;
    }

    /**
     * 根据维易返回的商品结果查询详情
     *
     * @param wySearchCommodityInfoDTO 维易商品信息
     * @param userId                   用户id
     * @param searchType               搜索类型
     * @return 转换后的商品搜索数据模型
     */
    private CommoditySearchDTO getDetail(WySearchCommodityInfoDTO wySearchCommodityInfoDTO, Long userId, String searchType) {


        if (Objects.isNull(wySearchCommodityInfoDTO)) {
            return null;
        }
        Long goodsId = Objects.isNull(wySearchCommodityInfoDTO.getItemId()) ?
                Long.valueOf(wySearchCommodityInfoDTO.getNumIid()) : wySearchCommodityInfoDTO.getItemId();

        CommodityDetailQueryParam detailParam = CommodityDetailQueryParam.builder()
                .goodsId(goodsId)
                .userId(userId)
                // 只查询本地
                .queryLocalOnly(true)
                // 不打印警告日志
                .doNotPrintWarnLog(true)
                // 金币信息
                .queryBrowerAward(true)
                .build();

        CommodityDetailDTO detail = commodityInfoService.getCommodityDetail(detailParam);
        // 商品详情检索失败，使用维易信息转化查询结果
        if (Objects.isNull(detail) || detail.getValidStatus() == 0) {
            if (log.isDebugEnabled()) {
                log.debug("商品详情本地检索失败，使用维易查询结果返回结果进行适配，wyCommodity：{}，userId：{},searchType:{}",
                        wySearchCommodityInfoDTO,
                        userId,
                        searchType);
            }
            // 根据wy的搜索信息进行转换
            detailParam.setWySearchCommodityInfoParam(WySearchCommodityInfoParam.builder()
                    .wyInfo(wySearchCommodityInfoDTO)
                    // 搜索的转换
                    .convertType((byte) 1)
                    .build());
            detail = commodityInfoService.getCommodityDetail(detailParam);

            if (Objects.isNull(detail)) {
                return null;
            }
        }

        CommoditySearchDTO searchDTO = CommodityDetailToSearchConvert.convert(detail);
        if (Objects.nonNull(searchDTO)) {
            searchDTO.setSearchType(StringUtils.isEmpty(searchType) ? SearchType.CONVERT_LINK : searchType);
        }
        return searchDTO;
    }

}
