package com.bxm.fossicker.vo;

import com.bxm.newidea.component.vo.Message;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;
import org.springframework.http.HttpStatus;

import java.util.Collection;

/**
 * 输出的标准Json格式数据， 用于替代原本黄瓜头条里面的Object输出
 * <code>ResponseJson.gen().data("mydata").code(Response.OK).output()</code>
 * @param <T> 泛型，Json中包含的具体的数据格式
 * @author liujia
 */
@ApiModel(description = "标准的Json响应格式数据")
public class ResponseJson<T> {

    @ApiModelProperty(required = true, value = "响应编码，200为正常，其他均为异常状态。注意和http返回编码区分。", example = "200/400/401/403/404/500")
    private int code;

    @ApiModelProperty(value = "操作编码不等于200时，返回对应的显示错误消息", example = "操作完成/操作成功/余额不足...")
    private String errorMsg = StringUtils.EMPTY;

    @ApiModelProperty(value = "响应数据主体")
    private T result;

    @ApiModelProperty(value = "当前请求用户的最新相关事件", hidden = true)
    private Collection<PushPayloadInfo> events;

    private ResponseJson() {
        this.code = HttpStatus.OK.value();
    }

    private ResponseJson(HttpStatus code) {
        this.code = code.value();
    }

    private ResponseJson(HttpStatus code, String errorMsg) {
        this.code = code.value();
        this.errorMsg = errorMsg;
    }

    private ResponseJson(Message message) {
        this.code = message.isSuccess() ? HttpStatus.OK.value() : HttpStatus.BAD_REQUEST.value();
        this.errorMsg = message.getLastMessage();
    }

    private ResponseJson(HttpStatus code, String errorMsg, T result) {
        this.code = code.value();
        this.errorMsg = errorMsg;
        this.result = result;
    }

    private ResponseJson(ResponseJson<T> source) {
        this.code = source.code;
        this.errorMsg = source.errorMsg;
        this.result = source.result;
    }

    public static <T> ResponseJson<T> build(Message message) {
        return new ResponseJson<>(message);
    }


    public static <T> ResponseJson<T> ok(T result) {
        return new ResponseJson<T>().body(result);
    }

    public static <T> ResponseJson<T> ok() {
        return new ResponseJson<>();
    }

    public static <T> ResponseJson<T> badReqeuset() {
        return new ResponseJson<>(HttpStatus.BAD_REQUEST);
    }

    public static <T> ResponseJson<T> badReqeuset(String errorMsg) {
        return new ResponseJson<>(HttpStatus.BAD_REQUEST, errorMsg);
    }

    public static <T> ResponseJson<T> serverError() {
        return new ResponseJson<>(HttpStatus.INTERNAL_SERVER_ERROR);
    }

    public static <T> ResponseJson<T> serverError(String errorMsg) {
        return new ResponseJson<>(HttpStatus.INTERNAL_SERVER_ERROR, errorMsg);
    }

    public static <T> ResponseJson<T> unauthorized() {
        return new ResponseJson<>(HttpStatus.UNAUTHORIZED);
    }

    public static <T> ResponseJson<T> unauthorized(String errorMsg) {
        return new ResponseJson<>(HttpStatus.UNAUTHORIZED, errorMsg);
    }

    /**
     * 操作是否成功
     * @return true表示操作成功
     */
    public boolean success() {
        return this.code == HttpStatus.OK.value();
    }

    public int getCode() {
        return this.code;
    }

    public ResponseJson<T> code(int code) {
        this.code = code;
        return this;
    }

    public String getErrorMsg() {
        return this.errorMsg;
    }

    public ResponseJson<T> errorMsg(String errorMsg) {
        this.errorMsg = errorMsg;
        return this;
    }

    public T getResult() {
        return this.result;
    }

    public ResponseJson<T> body(T result) {
        this.result = result;
        return this;
    }

    public Collection<PushPayloadInfo> getEvents() {
        return events;
    }

    public void setEvents(Collection<PushPayloadInfo> events) {
        this.events = events;
    }


    public void setCode(int code) {
        this.code = code;
    }

    public void setErrorMsg(String errorMsg) {
        this.errorMsg = errorMsg;
    }

    public void setResult(T result) {
        this.result = result;
    }

    @Override
    public String toString() {
        return ToStringBuilder.reflectionToString(this, ToStringStyle.MULTI_LINE_STYLE);
    }

}
