package com.bxm.thirdparty.platform.adapter.payment.refund.wx;

import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.bo.Message;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.thirdparty.payment.facade.PaymentConfigFacadeService;
import com.bxm.thirdparty.platform.adapter.payment.PayPlatformAction;
import com.bxm.thirdparty.platform.adapter.payment.enums.PaymentActionEnum;
import com.bxm.thirdparty.platform.adapter.payment.refund.wx.request.WxRefundRequest;
import com.bxm.thirdparty.platform.config.ThirdPartyConfigProperties;
import com.bxm.thirdparty.platform.constant.CommonConstant;
import com.bxm.thirdparty.platform.enums.PlatformEnum;
import com.bxm.thirdparty.platform.facade.response.RefundResponse;
import com.bxm.thirdparty.platform.mapper.PaymentOrderV2Mapper;
import com.bxm.thirdparty.platform.model.entity.PaymentOrderV2Entity;
import com.github.binarywang.wxpay.bean.request.BaseWxPayRequest;
import com.github.binarywang.wxpay.bean.request.WxPayRefundRequest;
import com.github.binarywang.wxpay.bean.result.WxPayRefundResult;
import com.github.binarywang.wxpay.exception.WxPayException;
import com.github.binarywang.wxpay.service.WxPayService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.Objects;

/**
 * @author lowi
 * @date 2023/3/27 10:06
 */
@Component
@Slf4j
public class WxRefundAction extends PayPlatformAction<WxRefundRequest> {

    @Resource
    private PaymentConfigFacadeService paymentConfigFacadeService;

    @Resource
    private ThirdPartyConfigProperties partyConfigProperties;

    @Resource
    private PaymentOrderV2Mapper paymentOrderV2Mapper;


    @Override
    public Message exec(WxRefundRequest request) {

        PaymentOrderV2Entity paymentOrderV2Entity = paymentOrderV2Mapper.getByOrderNo(Long.parseLong(request.getOrderNo()));

        WxPayService wxPayService = paymentConfigFacadeService.getWxClientByMchId(paymentOrderV2Entity.getAccountId());
        if (Objects.isNull(wxPayService)) {
            return Message.build(false).setMessage("支付账户不存在");
        }

        RefundResponse refundResponse = new RefundResponse();

        WxPayRefundRequest refundRequest = new WxPayRefundRequest();
        refundRequest.setTransactionId(paymentOrderV2Entity.getThirdPartyTradeNo());
        refundRequest.setOutRefundNo(request.getRequestOrderNo());
        // 金额需要转化为分
        refundRequest.setTotalFee(BaseWxPayRequest.yuanToFen(paymentOrderV2Entity.getAmount().toString()));
        refundRequest.setRefundFee(BaseWxPayRequest.yuanToFen(request.getRefundAmount().toString()));
        refundRequest.setAppid(wxPayService.getConfig().getAppId());
        refundRequest.setOpUserId(wxPayService.getConfig().getMchId());
        // 指定回调地址
        refundRequest.setNotifyUrl(buildNotifyUrl(paymentOrderV2Entity.getAccountId()));

        try {
            refundRequest.checkAndSign(wxPayService.getConfig());
        } catch (WxPayException e) {
            log.error(e.getMessage(), e);
            return Message.build(false).setMessage("校验签名失败");
        }

        try {
            WxPayRefundResult refundResult = wxPayService.refund(refundRequest);

            refundResponse.setResult(JSON.toJSONString(refundResult));
            if (StringUtils.equals(refundResult.getReturnCode(), CommonConstant.WECHAT_SUCCSS_CODE)) {
                refundResponse.setSuccess(true);
            } else {
                refundResponse.setSuccess(false);
                refundResponse.setErrorMsg(refundResult.getReturnMsg());
            }
            refundResponse.setOutOrderNo(request.getRefundOrderNo());
            refundResponse.setRefundOrderNo(request.getRequestOrderNo());
            refundResponse.setPaymentOrderNo(paymentOrderV2Entity.getOrderNo().toString());
        } catch (WxPayException e) {
            log.error(e.getMessage(), e);
            return Message.build(false).setMessage(e.getMessage());
        }
        return Message.build(true).addParam(CommonConstant.RESULT_DTO, refundResponse);
    }

    /**
     * 构建微信退款的回调地址，路径中包含账号ID信息
     *
     * @param accountId 账号ID
     * @return 回调地址
     */
    private String buildNotifyUrl(String accountId) {
        return partyConfigProperties.getServerUrl() + partyConfigProperties.getWxRefundCallbackUrl() + "/" + accountId;
    }

    @Override
    protected PaymentActionEnum matchAction() {
        return PaymentActionEnum.REFUND;
    }

    @Override
    protected PlatformEnum platform() {
        return PlatformEnum.WX;
    }
}
