package com.bxm.thirdparty.platform.queue;

import com.bxm.newidea.component.thread.NamedThreadFactory;
import com.bxm.thirdparty.platform.config.ServerNotifyProperties;
import com.bxm.thirdparty.platform.queue.bo.QueueBO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Component;

import java.util.concurrent.atomic.AtomicInteger;

/**
 * 业务通知线程管理
 *
 * @author liujia
 * @date 2023/03/27 20:53:14
 * @since 1.0.0
 */
@Slf4j
@Component
public class NotifyQueueExecutor implements ApplicationRunner {

    private ThreadPoolTaskExecutor taskExecutor;

    private AtomicInteger runningCounter = new AtomicInteger(0);

    private final ServerNotifyProperties properties;

    private final QueueManage queueManage;

    public NotifyQueueExecutor(ServerNotifyProperties properties, QueueManage queueManage) {
        this.properties = properties;
        this.queueManage = queueManage;
    }

    public boolean isFullLoad() {
        return runningCounter.get() <= 0;
    }

    void decrementAndGet() {
        runningCounter.decrementAndGet();
    }

    void incrementAndGet() {
        runningCounter.incrementAndGet();
    }

    ThreadPoolTaskExecutor getTaskExecutor() {
        return taskExecutor;
    }

    @Override
    public void run(ApplicationArguments args) throws Exception {
        if (properties.getQueueExecutorNum() <= 0) {
            throw new IllegalArgumentException("notify.config.queueExecutorNum不能小于0");
        }

        String namePrefix = "notify-queue-task";

        taskExecutor = new ThreadPoolTaskExecutor();
        taskExecutor.setMaxPoolSize(properties.getQueueExecutorNum());
        taskExecutor.setCorePoolSize(properties.getQueueExecutorNum());
        taskExecutor.setThreadFactory(new NamedThreadFactory(namePrefix, false));
        taskExecutor.setQueueCapacity(500);
        taskExecutor.setRejectedExecutionHandler((r, executor) -> {
            log.error("任务处理失败，超过容量限制");
            r.run();
        });
        taskExecutor.initialize();
        runningCounter = new AtomicInteger(properties.getQueueExecutorNum());
    }

    public void executor(QueueBO queueBO) {
        taskExecutor.execute(() -> {
            runningCounter.decrementAndGet();
            try {
                queueManage.notifyBusiness(queueBO);
            } catch (Exception e) {
                log.error(e.getMessage(), e);
            } finally {
                runningCounter.incrementAndGet();
            }
        });
    }
}
